#! /bin/bash
# The Broad Institute
# SOFTWARE COPYRIGHT NOTICE AGREEMENT
# This software and its documentation are copyright 2009 by the
# Broad Institute/Massachusetts Institute of Technology. All rights are
# reserved.

# This software is supplied without any warranty or guaranteed support
# whatsoever. Neither the Broad Institute nor MIT can be responsible for its
# use, misuse, or functionality.

PROGNAME=`basename $0`

source `dirname $0`/picard_defs.sh

function usage () {
    echo "USAGE: $PROGNAME [options] <tag-name>" >&2
    echo "Check out GATK sources with given (existing) tag label into new directory, build and test them." >&2
    echo "The directory $GATK_BUILD_ROOT/<branch-name> must not exist." >&2
    echo "-f    Force checkout and build even if the build directory pre-exists.  It is deleted and re-created in this case." >&2
}

function check_path () {
    if which $1 > /dev/null
    then
        echo > /dev/null
    else
        echo "ERROR: $1 not found on path.  Try 'use $2' before running this script." >&2
        exit 1
    fi
}

set -e

force=

while getopts "hf" options; do
  case $options in
    f ) force=1;;
    h ) usage
          exit 1;;
    \? ) usage
         exit 1;;
    * ) usage
          exit 1;;

  esac
done
shift $(($OPTIND - 1))

if (( $# != 1 ))
 then echo "ERROR: Incorrect number of arguments." >&2
      usage
      exit 1
fi

branch=$1

if gatk_tag_exists $branch
then echo > /dev/null
else
    echo "ERROR: Branch $branch does not exist." >&2
     exit 1
fi

build_parent=$GATK_BUILD_ROOT/$branch

check_path ant Ant-1.7

# delete pre-existing build directory if -f option
if [[ $force && -e $build_parent ]]
then
    read -p "Remove directory $build_parent? [y|n] " answer
    if [[ "x$answer" != xY && "x$answer" != xy ]]
    then
        echo "Aborting build." >&2
        exit 1
    fi
    echo Removing $build_parent
    rm -rf $build_parent
fi

if [[ -e $build_parent ]]
then echo "ERROR: $build_parent must not exist." >&2
     exit 1
fi

mkdir $build_parent
cd $build_parent
svn co $GATK_SVN_TAG_ROOT/$branch Sting
build_dir=$build_parent/Sting

cd Sting
ant clean dist

# This needs to be done after the above because the lib directory is not populated in a clean checkout.
ant -lib lib package -Dexecutable=GATK-Picard

