#! /bin/bash
# The Broad Institute
# SOFTWARE COPYRIGHT NOTICE AGREEMENT
# This software and its documentation are copyright 2009 by the
# Broad Institute/Massachusetts Institute of Technology. All rights are
# reserved.

# This software is supplied without any warranty or guaranteed support
# whatsoever. Neither the Broad Institute nor MIT can be responsible for its
# use, misuse, or functionality.

PROGNAME=`basename $0`

source `dirname $0`/picard_defs.sh

function usage () {
    echo "USAGE: $PROGNAME [options] <branch-name>" >&2
    echo "Check out Picard sources with given (existing) branch label into new directory, build and test them." >&2
    echo "The directory $BUILDROOT/<branch-name> must not exist." >&2
    echo "-f    Force checkout and build even if the build directory pre-exists.  It is deleted and re-created in this case." >&2
    echo "-t    Do not run test suite before building deployment file." >&2
}

function check_path () {
    if which $1 > /dev/null
    then
        echo > /dev/null
    else
        echo "ERROR: $1 not found on path.  Try 'use $2' before running this script." >&2
        exit 1
    fi
}

set -e

force=
run_test_suite=1

while getopts "hft" options; do
  case $options in
    f ) force=1;;
    t ) run_test_suite=0;;
    h ) usage
          exit 1;;
    \? ) usage
         exit 1;;
    * ) usage
          exit 1;;

  esac
done
shift $(($OPTIND - 1))

if (( $# != 1 ))
 then echo "ERROR: Incorrect number of arguments." >&2
      usage
      exit 1
fi

branch=$1

if branch_exists $branch
then echo > /dev/null
else
    echo "ERROR: Branch $branch does not exist." >&2
     exit 1
fi

build_parent=$BUILDROOT/$branch

check_path ant Ant-1.7
check_path Rscript R-2.8.1

# delete pre-existing build directory if -f option
if [[ $force && -e $build_parent ]]
then
    read -p "Remove directory $build_parent? [y|n] " answer
    if [[ "x$answer" != xY && "x$answer" != xy ]]
    then
        echo "Aborting build." >&2
        exit 1
    fi
    echo Removing $build_parent
    rm -rf $build_parent
fi

if [[ -e $build_parent ]]
then echo "ERROR: $build_parent must not exist." >&2
     exit 1
fi

mkdir $build_parent
cd $build_parent
svn co $SVNROOT/branches/$branch Picard
build_dir=$build_parent/Picard
svn co https://picard.svn.sourceforge.net/svnroot/picard/trunk Picard/Picard-public

cd Picard
# Shouldn't be necessary, but no harm
ant clean

if (( $run_test_suite ))
then
    echo "Running test suite..."
    ant -lib $build_dir/lib test-picard-public test
    echo "Test suite succeeded."
else
    echo "Skipped test suite."
fi

echo "Building deployment zip..."
ant -lib $build_dir/lib make-deployment-zip
echo "Deployment zip built successfully."
