# Script to generate a chart of internal control error rates from the appropriate
# metrics file.  Creates a single PDF with a plot for the aggregate error rate
# and a plot for each internal control for which matches were found.
#
# @author Tim Fennell

# Parse the arguments
args <- commandArgs(trailing=T)
metricsFile  <- args[1]
outputFile   <- args[2]
titlePrefix  <- args[3]

# Figure out where the metrics and the histogram are in the file and parse them out
startFinder <- scan(metricsFile, what="character", sep="\n", quiet=TRUE, blank.lines.skip=FALSE)

firstBlankLine=0

for (i in 1:length(startFinder)) {
        if (startFinder[i] == "") {
                if (firstBlankLine==0) {
                        firstBlankLine=i+1
                } else {
                        secondBlankLine=i+1
                        break
                }
        }
}

metrics <- read.table(metricsFile, header=T, sep="\t", skip=firstBlankLine)
pdf(outputFile)

# Setup the values needed for rendering the Y axis as a log axis
Y_AXIS_MIN = 0.001;
Y_AXIS_MAX = 1;
tmp <- Y_AXIS_MIN;
y_axis_marks <- c();
while (tmp<=Y_AXIS_MAX) { y_axis_marks <- c(y_axis_marks, tmp, recursive=TRUE); tmp <- tmp*10; }

# Get the data and plot it
ics <- as.vector(unique(metrics$INTERNAL_CONTROL));
reads <- unique(metrics$READ);
my_colors <- c(11,28);

for (my_ic in ics) {
    ic_subset <- subset(metrics, INTERNAL_CONTROL==my_ic);
    count_string <- paste(c("(n=", max(ic_subset$OBSERVATIONS), ")"), sep="", collapse="");

	plot(type="n", x=0, y=0, xlab="Cycle", ylab="Error Rate (log axis)",
         xlim=c(0,max(metrics$CYCLE)),
         ylim=c(Y_AXIS_MIN, Y_AXIS_MAX), log="y", yaxt="n",
	     main=paste(c(titlePrefix, my_ic, count_string, "IC Error Rate by Cycle"), sep=" ", collapse=" "),
	     col=my_colors
	    );

	# Draw the y-axis separately so that we can put the ticks where we want
	axis(side=2, at=y_axis_marks);

	legend("topleft", legend=paste("Read", reads), fill=my_colors);

	for (my_read in unique(ic_subset$READ)) {
			my_subset <- subset(ic_subset, READ==my_read);
			lines(type="l", col=my_colors[my_read], x=my_subset$CYCLE, y=pmax(my_subset$ERROR_RATE, Y_AXIS_MIN));
	}
}

dev.off();