package org.broadinstitute.cga.tools.seq;

import org.broadinstitute.cga.tools.seq.Wiggle;
import org.broadinstitute.cga.tools.seq.BamGrasp;
import org.broadinstitute.cga.tools.seq.FixedWidthBinary;
import net.sf.samtools.*;
import net.sf.samtools.util.CloseableIterator;
import java.io.*;
import java.lang.*;
import java.util.*;

public class Genome {
    protected static final int MINCHR = 0;
    protected static final int MAXCHR = 24;
    protected static final int COORDBASE = 1;
    public static final int[] MAXCHRLEN =   // (~10% larger than actual chr lengths)
	{
	    20000,
	  260000000, 260000000, 210000000, 200000000, 190000000, 180000000,
	  170000000, 160000000, 150000000, 140000000, 140000000, 140000000,
	  120000000, 120000000, 110000000, 100000000,  90000000,  80000000,
	   70000000,  70000000,  60000000,  60000000, 160000000,  60000000
	};

    // storage buffers (8bit or 16bit)
    //    private int contents_width = -1;
    public byte[][] contents = new byte[MAXCHR+1][];
    //public short[][] contents = new short[MAXCHR+1][];

    protected static final byte EMPTY = -128;
    protected static final byte WASNONZERO = EMPTY+1;   // used only temporarily (during "AND")
    protected static final byte MASKED = EMPTY+2;   // positions which will not be considered
    protected static final byte MINVAL = EMPTY+3;
    protected static final byte MAXVAL = 127;

    public boolean allEmpty = false;   // (array starts out all zeroes, not all EMPTY
    public boolean noMask = true;
    public long totContents = 0;
    public long numContents = 0;
    public int maxContents = 0;
    public int minContents = 0;

    public int getMinContents() { return(minContents); }
    public int getMaxContents() { return(maxContents); }
    public long getNumContents() { return(numContents); }
    public long getTotContents() { return(totContents); }
    public boolean isAllEmpty() { return(allEmpty); }
    public boolean hasNoMask() { return(noMask); }

    protected boolean quietFlag = false;

    public Genome() throws Exception {
        allocate();
    }

    public Genome(boolean qf) throws Exception {
        quietFlag = qf;
        allocate();
    }

    protected void allocate() throws Exception {
        for (int c=MINCHR;c<=MAXCHR;c++) contents[c] = new byte[MAXCHRLEN[c]+1];
        clear();
    }

    public void clear() {
	if (!allEmpty || !noMask) {
	    allEmpty = true;
	    noMask = true;
	    totContents = 0; numContents=0; minContents=EMPTY; maxContents=EMPTY;
	    for (int c=MINCHR;c<=MAXCHR;c++) Arrays.fill(contents[c],EMPTY);
	}
    }

    public void mask() {
	allEmpty = true;
	noMask = false;
	totContents = 0; numContents=0; minContents=EMPTY; maxContents=EMPTY;
	for (int c=MINCHR;c<=MAXCHR;c++) Arrays.fill(contents[c],MASKED);
    }

    //////////////////////////////////////////////////////////////////////
    //////////////////////////////////////////////////////////////////////

    protected static void checkchr(int chr) throws Exception {
        if (chr<MINCHR || chr>MAXCHR)
            throw new Exception("chr "+chr+" out of range");
    }

    protected static void checkpos(int chr, int pos) throws Exception {
        if (chr<MINCHR || chr>MAXCHR || pos<COORDBASE || pos>MAXCHRLEN[chr])
            throw new Exception("chr "+chr+":"+pos+" out of range");
    }

    protected void priv_setContents(int chr, int pidx, int newval) {
        contents[chr][pidx] = (byte)newval;
    }

    public boolean isMasked(int chr, int pos) throws Exception {
	checkpos(chr,pos);
        int val = (int)contents[chr][pos-COORDBASE];
        return(val==MASKED);
    }

    public boolean isMasked(int chr, int start, int end) throws Exception {
        checkpos(chr,start);
        checkpos(chr,end);
        if (start>end) throw new Exception("start>end");
        for (int i=start-COORDBASE;i<=end-COORDBASE;i++)
            if (contents[chr][i]==MASKED) return(true);
        return(false);
    }

    public boolean hasContents(int chr, int pos) throws Exception {
        checkpos(chr,pos);
	int val = (int)contents[chr][pos-COORDBASE];
	return(val>=MINVAL && val<=MAXVAL);
    }

    public boolean hasContents(int chr, int start, int end) throws Exception {
        checkpos(chr,start);
	checkpos(chr,end);
        if (start>end) throw new Exception("start>end");
	for (int i=start-COORDBASE;i<=end-COORDBASE;i++)
	    if (contents[chr][i]>=MINVAL && contents[chr][i]<=MAXVAL) return(true);
	return(false);
    }

    public int getContents(int chr, int pos) throws Exception {
        checkpos(chr,pos);
	int val = (int)contents[chr][pos-COORDBASE];
	if (val==EMPTY) System.out.println("Warning: that position is empty = "+EMPTY);
        if (val==MASKED) System.out.println("Warning: that position is masked = "+MASKED);
	return(val);
    }

    public int[] getContents(int chr, int start, int end) throws Exception {
        checkpos(chr,start);
        checkpos(chr,end);
        if (start>end) throw new Exception("start>end");
        int[] val = new int[end-start+1];
        boolean maskflag = false;
        boolean emptyflag = false;
        for (int i=0; i<end-start+1; i++) {
	    val[i] = (int)contents[chr][start-COORDBASE+i];
            if (val[i]==EMPTY) emptyflag=true;
            if (val[i]==MASKED) maskflag=true;
        }
        if (emptyflag && !quietFlag) System.out.println("Warning: range contains empty positions = "+EMPTY);
        if (maskflag && !quietFlag) System.out.println("Warning: range contains masked positions = "+MASKED);
        return(val);
    }

    public int getSumContents(int chr, int start, int end) throws Exception {
        checkpos(chr,start);
        checkpos(chr,end);
        if (start>end) throw new Exception("start>end");
	boolean maskflag = false;
	int sum = 0;
	for (int i=start-COORDBASE;i<=end-COORDBASE;i++) {
	    int val = contents[chr][i];
	    if (val==EMPTY) continue;
	    if (val==MASKED) { maskflag=true; continue; }
	    sum += val;
	}
        if (maskflag && !quietFlag) System.out.println("Warning: range contains masked positions = "+MASKED);
        return(sum);
    }

    public void setContents(int chr, int start, int newval) throws Exception {
        setContents(chr, start, start, newval);
    }

    public void setContents(int chr, int start, int end, int newval) throws Exception {
        checkpos(chr,start);
        checkpos(chr,end);
        if (start>end) throw new Exception("start>end");
        if (newval<MINVAL || newval>MAXVAL) throw new Exception("value "+newval+" out of range");
        for (int pidx=start-COORDBASE; pidx<=end-COORDBASE; pidx++) {
            int oldval = contents[chr][pidx];
            if (oldval==MASKED) continue;
            allEmpty = false;
	    priv_setContents(chr,pidx,newval);
            if (oldval>=MINVAL && oldval<=MAXVAL) totContents -= oldval;
	    else numContents++;
            totContents += newval;
            if (minContents==EMPTY || newval<minContents) minContents=newval;
            if (maxContents==EMPTY || newval>maxContents) maxContents=newval;
        }
    }

    public void setContents(int chr, int start, int end, int[] newvals) throws Exception {
        checkpos(chr,start);
        checkpos(chr,end);
        if (start>end) throw new Exception("start>end");
	int idx = 0;
        for (int pidx=start-COORDBASE; pidx<=end-COORDBASE; pidx++) {
	    int newval = newvals[idx++];
	    if (newval<MINVAL || newval>MAXVAL) throw new Exception("value "+newval+" out of range");
            int oldval = contents[chr][pidx];
            if (oldval==MASKED) continue;
            allEmpty = false;
	    priv_setContents(chr,pidx,newval);
            if (oldval>=MINVAL && oldval<=MAXVAL) totContents -= oldval;
            else numContents++;
            totContents += newval;
            if (minContents==EMPTY || newval<minContents) minContents=newval;
            if (maxContents==EMPTY || newval>maxContents) maxContents=newval;
        }
    }

    public void incrementWithCeiling(int chr, int pos) throws Exception {
	checkpos(chr,pos);
	int pidx = pos-COORDBASE;
	int oldval = contents[chr][pidx];
	if (oldval==MASKED || oldval==MAXVAL) return;
	int newval = oldval+1;
	if (oldval==EMPTY) newval = 1;
	allEmpty = false;
	totContents++;
	priv_setContents(chr,pidx,newval);
	if (minContents==EMPTY || newval<minContents) minContents=newval;
	if (maxContents==EMPTY || newval>maxContents) maxContents=newval;
    }


    /////////////////////////////////////////////////////////////////////
    /////////////////////////////////////////////////////////////////////

    public void saveWiggle(String filename) throws Exception {
	if (allEmpty) throw new Exception("No data to save!");

	BufferedWriter out = new BufferedWriter(new FileWriter(filename),1000000);
	out.write("track type=wiggle_0 name=wiggle viewLimits=" + minContents + ":" + maxContents + "\n");

	int oldchr = -1;
	int chr = MINCHR;
	int pos = 0;
	int wigchr = -1;
	int wigpos = -1;
	int clen = MAXCHRLEN[chr];
	while(true) {
	    do pos++; while (pos<=clen && !hasContents(chr,pos));
	    if (pos>clen) {
		chr++;
                if (chr>MAXCHR) break;
		clen = MAXCHRLEN[chr];
		pos = 0;
		continue;
	    }
	    if (chr!=wigchr || pos!=wigpos) {
		out.write("fixedStep chrom="+chrstring(chr)+" start="+pos+" step=1\n");
		if (chr != oldchr) {
		    if (!quietFlag) System.out.print(chrstring(chr)+" ");
		    oldchr=chr;
		}
		wigchr = chr;
		wigpos = pos;
	    }
	    out.write(getContents(chr,pos)+"\n");
	    wigpos++;
	}
	out.close();
	if (!quietFlag) System.out.print("\n");
    }   


    /////////////////////////////////////////////////////////////////////
    /////////////////////////////////////////////////////////////////////

    public void saveFixedWidthBinary(String fwbName, int width, int nullval, String indexName) throws Exception {
	saveFixedWidthBinary(fwbName, width, nullval, indexName, 1, 2, 3);
    }

    public void saveFixedWidthBinary(String fwbName, int outWidth, int nullval, String indexName,
				     int chrcol, int startcol, int endcol) throws Exception {
        // fwbName = name of fwb file to write
	// outWidth = width in bits (1,2,4,8,16,24,32)
        // nullval = value to output in case of EMPTY or MASKED
        // indexName = name of index file containing list of regions to output 
        // chrcol, startcol, endcol = columns to read within index file

        if (allEmpty) throw new Exception("No data to save!");

	FixedWidthBinary fwb = new FixedWidthBinary();
	if (!fwb.isLegalWidth(outWidth)) throw new Exception("Illegal width "+outWidth);
	int maxval = fwb.maxValForWidth(outWidth);
	fwb.loadIndex(indexName, chrcol, startcol, endcol);

	int bsz = 10000000;       // buffer size
	DataOutputStream out = new DataOutputStream(new BufferedOutputStream(new FileOutputStream(fwbName),bsz));

	// write each region
	int bitbuffer = 0;
	int bitbuffercount = 0;
	int nr = fwb.getNumRegions();
	int oldchr = -1;
	for (int r=0; r<fwb.getNumRegions(); r++) {
	    int chr = fwb.getRegionChr(r);
	    int start = fwb.getRegionStart(r);
	    int end = fwb.getRegionEnd(r);
	    if (chr != oldchr) {
		if (!quietFlag) System.out.print(chrstring(chr)+" ");
		oldchr=chr;
	    }
	    // write each position
	    for (int pos=start; pos<=end; pos++) {
		int val = nullval;
		if (chr>=MINCHR && chr<=MAXCHR && pos>=COORDBASE && pos<=MAXCHRLEN[chr]) {
		    val = (int)contents[chr][pos-COORDBASE];
		    if (val==EMPTY || val==MASKED) val = nullval;
		}
		if (val<0) throw new Exception("Can't store negative values in FWB: use applyFloor");
		if (val>maxval) throw new Exception("Can't store values above "+maxval+ 
                   " (e.g. "+val+") using width="+outWidth + ": use applyCeil");
		if (outWidth<8) {
		    bitbuffer <<= outWidth;
		    bitbuffer += val;
		    bitbuffercount += outWidth;
		    if (bitbuffercount==8) {
			out.writeByte(bitbuffer);
			bitbuffer = 0;
			bitbuffercount = 0;
		    }
		} else if (outWidth==8) {
		    out.writeByte(val);
		} else if (outWidth==16) {
		    out.writeShort(val);
		} else if (outWidth==24) {
		    int lowshort = val % 65536;
		    int highbyte = (val-lowshort) / 65536;
		    out.writeByte(highbyte);
		    out.writeShort(lowshort);
		} else if (outWidth==32) {
		    out.writeInt(val);
		} else {
		    throw new Exception("Impossible: width became invalid after initial check");
		}
	    } // next position
	} // next region
	if (bitbuffercount>0) { // flush bitbuffer
	    bitbuffer <<= (8-bitbuffercount);
	    out.writeByte(bitbuffer);
	}
	out.close();
        if (!quietFlag) System.out.print("\n");
    }

    /////////////////////////////////////////////////////////////////////
    /////////////////////////////////////////////////////////////////////

    public static String chrstring(int chr) throws Exception {
	checkchr(chr);
	if (chr==0) return(new String("chrM"));
	else if (chr==23) return (new String("chrX"));
	else if (chr==24) return (new String("chrY"));
	else return (new String("chr"+chr));
    }

    /////////////////////////////////////////////////////////////////////
    /////////////////////////////////////////////////////////////////////

    protected static final String[] modeStringList =
    { "max", "min", "sum", "and", "or", "zeroif", "mask", "mask0", "overwrite"};

    protected static final String defaultMode = "overwrite";

    protected int matchModeString(String modeString) throws Exception {
        int mode;
        for (mode=0;mode<modeStringList.length;mode++) {
            if (modeString.equalsIgnoreCase(modeStringList[mode])) break;
        }
        if (mode==modeStringList.length) throw new Exception("Unknown mode "+modeString);
	return(mode);
    }

    public void loadWiggle(String filename) throws Exception {
	if (!allEmpty) {
	    System.out.println("WARNING: Data still loaded.  Loading new file in mode '"+defaultMode+"'");
	}
        loadWiggle(filename,defaultMode);
    }

    public void loadWiggle(String filename, String modeString) throws Exception {
	loadWiggle(filename, modeString, (float)1, (float)0);
    }


    public void loadWiggle(String filename, String modeString, double scale_m, double scale_b) throws Exception {
        loadWiggle(filename, modeString, (float)scale_m, (float)scale_b);
    }

    public void loadWiggle(String filename, String modeString, int scale_m, int scale_b) throws Exception {
	loadWiggle(filename, modeString, (float)scale_m, (float)scale_b);
    }

    public void loadWiggle(String filename, double scale_m, double scale_b) throws Exception {
        loadWiggle(filename, defaultMode, (float)scale_m, (float)scale_b);
    }

    public void loadWiggle(String filename, int scale_m, int scale_b) throws Exception {
        loadWiggle(filename, defaultMode, (float)scale_m, (float)scale_b);
    }

    public void loadWiggle(String filename, float scale_m, float scale_b) throws Exception {
        loadWiggle(filename, defaultMode, scale_m, scale_b);
    }

    public void loadWiggle(String filename, String modeString, float scale_m, float scale_b) throws Exception {
	int mode = matchModeString(modeString);

	if (mode==3) preand();   // mode= "AND":   mark all nonzero positions as "WASNONZERO"
	if (mode==6 || mode==7) mask();     // mode= "MASK"/"MASK0":  start with all positions as "MASKED"

	Wiggle wig = new Wiggle(filename,quietFlag,scale_m,scale_b);

	long floored_val_ct = 0;
	long ceiled_val_ct = 0;
	long tot_loaded_ct = 0;

	int oldchr = -1;
	while(wig.next()) {
	    int chr = wig.getChr();
	    if (chr!=oldchr) {
		System.out.print(chrstring(chr)+" ");
		oldchr = chr;
	    }
	    int pos = wig.getPos();
            int val = wig.getValue();
	    checkpos(chr,pos);
	    int oldval = contents[chr][pos-COORDBASE];
	    int newval = EMPTY;
	    boolean do_nothing_flag = false;
	    switch(mode) {
            case 8:   // overwrite
                newval = val;
                break;
	    case 0:   // max
		if (oldval==EMPTY || val>oldval) newval = val;
		else newval = oldval;
		break;
	    case 1:   // min
		if (oldval==EMPTY || val<oldval) newval = val;
		else newval = oldval;
		break;
	    case 2:   // sum
		if (oldval==EMPTY) newval = val;
		else newval = oldval + val;
		break;
	    case 3:   // and
		if (oldval==EMPTY || oldval==0) do_nothing_flag = true;
		else newval = val; // (overwrite "WASNONZERO")
		break;
	    case 4:   // or
		if (oldval==EMPTY) newval = val;
		else newval = oldval | val;
		break;
	    case 5:   // zeroif
		if (val==0) do_nothing_flag = true;
		else newval = 0;
		break;
	    case 6:   // mask: non-mask positions set to empty
		if (val!=0) contents[chr][pos-COORDBASE] = EMPTY;  // (remove mask)
		do_nothing_flag = true;
		break;
            case 7:   // mask0: non-mask positions set to zero
                if (val!=0) contents[chr][pos-COORDBASE] = 0;
		do_nothing_flag = true;
                break;
		
	    default: throw new Exception("unknown loadWiggleMode");
	    }
	    if (!do_nothing_flag) {
		if (oldval==MASKED) continue;
                allEmpty = false;
		tot_loaded_ct++;
		if (newval<MINVAL) {
		    newval=MINVAL;
		    floored_val_ct++;
		}
		if (newval>MAXVAL) {
		    newval=MAXVAL;
		    ceiled_val_ct++;
		}
		priv_setContents(chr,pos-COORDBASE,newval);
		if (oldval>=MINVAL && oldval<=MAXVAL) totContents -= oldval;
		else numContents++;
		totContents += newval;
		if (minContents==EMPTY || newval<minContents) minContents=newval;
		if (maxContents==EMPTY || newval>maxContents) maxContents=newval;
	    }
	}
	wig.close();
        if (mode==3) postand();   // mode="and": zero all positions still marked as "WASNONZERO"
	if (!quietFlag) {
	    System.out.print("\n");
	    if (floored_val_ct>0) {
		System.out.println(floored_val_ct + " of " + tot_loaded_ct + " values were floored at MINVAL=" + MINVAL);
	    }
	    if (ceiled_val_ct>0) {
                System.out.println(ceiled_val_ct + " of " + tot_loaded_ct + " values were ceiled at MAXVAL=" + MAXVAL);
            }
	}
    }

    /////////////////////////////////////////////////////////////////////
    /////////////////////////////////////////////////////////////////////

    protected void preand() {
	if (!quietFlag) System.out.print("[pre-and] ");
	for (int chr=MINCHR;chr<=MAXCHR;chr++) {
	    for (int pidx=1-COORDBASE;pidx<=MAXCHRLEN[chr]-COORDBASE;pidx++) {
		int cc = contents[chr][pidx];
		if (cc!=MASKED && cc!=EMPTY && cc!=0) {
		    contents[chr][pidx] = WASNONZERO;
		}
	    }
	}
        allEmpty = true;
        totContents = 0;
	numContents = 0;
        maxContents = EMPTY;
        minContents = EMPTY;
    }

    protected void postand() {
        if (!quietFlag) System.out.print("[post-and] ");
	boolean flag = false;
	for (int chr=MINCHR;chr<=MAXCHR;chr++) {
	    for (int pidx=1-COORDBASE;pidx<=MAXCHRLEN[chr]-COORDBASE;pidx++) {
		if (contents[chr][pidx]==WASNONZERO) {
		    contents[chr][pidx] = 0;
		    numContents++;
		    flag = true;
		}
	    }
	}
	if (flag) {
	    if (minContents==EMPTY || 0<minContents) minContents=0;
	    if (maxContents==EMPTY || 0>maxContents) maxContents=0;
	}
    }

    //////////////////////////////////////////////////////////////////////
    //////////////////////////////////////////////////////////////////////


    public void loadCoverageByBase(String bamname) throws Exception {
	loadCoverageByBase(bamname,new String("none"),null,0);
    }

    public void loadCoverageByBase(String bamname, String blacklistname) throws Exception {
        loadCoverageByBase(bamname,blacklistname,null,0);
    }

    public void loadCoverageByBase(String bamname, String blacklistname, String refdir) throws Exception {
	loadCoverageByBase(bamname,blacklistname,refdir,0);
    }

    public void loadCoverageByBase(String bamname, int baseQualCutoff) throws Exception {
        loadCoverageByBase(bamname,new String("none"),null,baseQualCutoff);
    }

    public void loadCoverageByBase(String bamname, String blacklistname, int baseQualCutoff) throws Exception {
        loadCoverageByBase(bamname,blacklistname,null,baseQualCutoff);
    }

    public void loadCoverageByBase(String bamname, String blacklistname, String refdir, int baseQualCutoff) throws Exception {
	BamGrasp bam;
	if (refdir != null) bam = new BamGrasp(bamname, blacklistname, refdir);
	else bam = new BamGrasp(bamname, blacklistname);

        if (!allEmpty) {
            if (!quietFlag) System.out.println("Clearing memory; starting from all zero counts.");
            clear();
        }
	
	boolean parseBases = (baseQualCutoff>0);

	long start_time = System.currentTimeMillis();
	long idx = 0;
	for (int chr=MINCHR;chr<=MAXCHR;chr++) {
	    String seqname = bam.getChrName(chr);

	    CloseableIterator<SAMRecord> c = bam.reader.queryOverlapping(seqname,1,300000000);
	    //	    SAMRecordIterator c = bam.reader.queryOverlapping(seqname,1,300000000);
	    while(c.hasNext()) {
		SAMRecord x = c.next();

		idx++;
		if ((idx%1000000)==0) {
		    float elapsed = (System.currentTimeMillis() - start_time);
		    elapsed /= 1000;
		    float rate = idx / elapsed;
		    if (!quietFlag) System.out.println(idx + " records  " + elapsed + " seconds  "
                        + rate + " records/second  " + chr + ":" + bam.read.start);
		}

		// filtering
		if (x.getReadUnmappedFlag()) continue;          // filter out unmapped reads
		if (x.getDuplicateReadFlag()) continue;         // filter out duplicate reads
		if (x.getNotPrimaryAlignmentFlag()) continue;   // filter out non-primary alignments
		if (x.getMappingQuality()==0) continue;         // filter out MAPQ=0

		// if all positions in the read are masked, skip it (saves parsing time)
		boolean all_masked = true;
		for (int p=x.getAlignmentStart()-COORDBASE;p<=x.getAlignmentEnd()-COORDBASE;p++) {
		    if (contents[chr][p]!=MASKED) { all_masked = false; break; }
		}
		if (all_masked) continue;

                bam.parse(x,parseBases);
		if (bam.read.isBlacklisted) continue;

		// add read to coverage count
		int pos = bam.read.start;
		for (int base=0;base<bam.read.numBases;base++,pos++) {
		    if (baseQualCutoff>0 && bam.read.basequal[base]<baseQualCutoff) continue;
		    int oldval = contents[chr][pos-COORDBASE];
		    if (oldval==MASKED) continue;
		    if (oldval==EMPTY) oldval = 0;
		    int newval = oldval+1;
		    if (newval<=MAXVAL) setContents(chr,pos,newval);
		}
	    }
	    c.close();
	} // next chr
	
	bam.close();
     }

    /////////////////////////////////////////////////////////////////////////////////

    public void imposeCutoff(int cutoff) throws Exception{
	// ignores masked or empty positions
	// for all non-masked, non-empty positions:
	//    if contents >= cutoff, changes contents to 1
	//    if contents < cutoff, changes contents to 0
        for (int chr=MINCHR;chr<=MAXCHR;chr++) {
            for (int pidx=1-COORDBASE;pidx<=MAXCHRLEN[chr]-COORDBASE;pidx++) {
                int cc = contents[chr][pidx];
                if (cc!=MASKED && cc!=EMPTY) {
		    int newval = (cc>=cutoff)?1:0;
		    priv_setContents(chr,pidx,newval);
		}
            }
        }
	minContents = 0;
	maxContents = 1;
    }
    
    public void applyFloor(int floor) throws Exception{
        // ignores masked or empty positions
        // for all non-masked, non-empty positions:
        //    if contents >= floor, leaves contents as is
        //    if contents < floor, changes contents to floor
	long floored_val_ct = 0;
	long total_val_ct = 0;
        for (int chr=MINCHR;chr<=MAXCHR;chr++) {
            for (int pidx=1-COORDBASE;pidx<=MAXCHRLEN[chr]-COORDBASE;pidx++) {
                int cc = contents[chr][pidx];
                if (cc!=MASKED && cc!=EMPTY) {
		    total_val_ct++;
		    if (cc<floor) {
			priv_setContents(chr,pidx,floor);
			floored_val_ct++;
		    }
		}
            }
        }
        if (minContents<floor) minContents = floor;
	if (!quietFlag) {
	    System.out.println(floored_val_ct + " of " + total_val_ct + " values floored at " + floor);
	}
    }

    public void applyCeil(int ceil) throws Exception{
        // ignores masked or empty positions
        // for all non-masked, non-empty positions:
        //    if contents <= ceil, leaves contents as is
        //    if contents > ceil, changes contents to ceil
        long ceiled_val_ct = 0;
        long total_val_ct = 0;
        for (int chr=MINCHR;chr<=MAXCHR;chr++) {
            for (int pidx=1-COORDBASE;pidx<=MAXCHRLEN[chr]-COORDBASE;pidx++) {
                int cc = contents[chr][pidx];
                if (cc!=MASKED && cc!=EMPTY) {
                    total_val_ct++;
                    if (cc>ceil) {
                        priv_setContents(chr,pidx,ceil);
                        ceiled_val_ct++;
                    }
                }
            }
        }
        if (maxContents>ceil) maxContents = ceil;
	if (!quietFlag) {
	    System.out.println(ceiled_val_ct + " of " + total_val_ct + " values ceiled at " + ceil);
	}
    }


    /////////////////////////////////////////////////////////////////////////////////
 
    // inventory of nonempty regions
    // (not yet implemented)
    protected int[] reg_chr, reg_start, reg_end;
    protected int nreg = 0;
    protected static final int MAX_NREG = 1000000;

    protected void addreg(int chr, int start, int end) {
        // not yet implemented
    }

}
