package org.broadinstitute.cga.tools.seq;

import java.io.*;
import java.lang.*;
import java.util.*;

public class Reference {

    // location of the reference files

    private String refdir = null;
    boolean noRef = false;

    // storage of reference DNA

    private final int chunksize = 11000000;   // must be even
    private byte[] refdna = new byte[2*chunksize];
    private RandomAccessFile reader = null;
    private int reflen = -1;
    private int refchr = -1;
    private int refmin = -1;
    private int refmax = -1;

    // public methods

    public Reference(String dirname) {
	refdir = dirname;
	if (!refdir.equalsIgnoreCase("none") && !(new File(dirname)).exists()) {
	    System.out.println("Reference directory "+dirname +" does not exist");
	    refdir = new String("none");
	}
        if (refdir.equalsIgnoreCase("none")) {
	    System.out.println("No reference data available: will return all N's");
	    Arrays.fill(refdna,0,chunksize*2,(byte)'N');
	    noRef = true;
	}
    }

    public void close() throws Exception {
	if (reader != null) {
	    reader.close();
	    refchr = -1;
	}
    }

    public void finalize() throws Exception {
	close();
    }

    //////////////////////////////////////////////////////////////////////
    //////////////////////////////////////////////////////////////////////

    public String get(int chr, int pos) throws Exception {
	return(get(chr,pos,pos));
    }

    public String get(int chr, int start, int end) throws Exception {

	if (chr<0 || chr>24) throw new Exception("chr should be 0-24");
	if (start<1) throw new Exception("start<1");
	if (start>end) throw new Exception("start>end ("+start+">"+end+")");
	if (end-start+1>chunksize) throw new Exception("max retrievable size is "+chunksize);

	if (noRef) {    // refdir = "none"
	    refmin = start;
	    refmax = start+(2*chunksize)-1;
	} else {
	    if (refchr!=-1 && refchr!=chr) {   // close old file
		reader.close();
		refchr = -1;
	    }
	    if (refchr==-1) {    // open new file
		String refname = getRefname(chr);
		reader = new RandomAccessFile(new File(refname),"r");
		refchr = chr;
		long len = reader.length();
		if (len>Integer.MAX_VALUE) throw new Exception("reference file too long!");
		reflen = (int)len;
		refmin = -1;
		refmax = -1;
	    }
	    // cases:
	    //     (1) nothing loaded yet, or start is outside both chunks: load from start to 2 chunks later
	    //     (2) start is within second chunk: copy second chunk to first chunk and load new second chunk
	    //     (3) start is within first chunk: simply return what's already loaded
	    
	    if (refmin==-1 || start<refmin || start>refmax) {   // file is open but need to completely refresh buffer
		int bytesread = 0;
		if (start<=reflen) {
		    reader.seek(start-1);
		    bytesread = reader.read(refdna,0,2*chunksize);
		}
		if (bytesread < 2*chunksize) Arrays.fill(refdna,bytesread,chunksize*2,(byte)'N');
		refmin = start;
		refmax = start+(2*chunksize)-1;
	    }
	    if (start>=refmin+chunksize && start<=refmax) {    // start is within second chunk: advance one chunk
		System.arraycopy(refdna,chunksize,refdna,0,chunksize);
		refmin += chunksize;
		refmax += chunksize;
		int bytesread = 0;
		if (refmin+chunksize <= reflen) {
		    reader.seek(refmin+chunksize-1);
		    bytesread = reader.read(refdna,chunksize,chunksize);
		}
		if (bytesread<chunksize) Arrays.fill(refdna,chunksize+bytesread,chunksize*2,(byte)'N');
	    }
	}

	if (start<refmin || start>=refmin+chunksize) {
	    throw new Exception("Unforeseen problem in Reference.get()");
	}
	
	String rval = new String(refdna,start-refmin,end-start+1);
	return(rval.toUpperCase());
    }


    //////////////////////////////////////////////////////////////////////
    //////////////////////////////////////////////////////////////////////

    private String getRefname(int chr) throws Exception {
        String refname = null;
        if (chr>=1 && chr<=22) {
            refname = refdir + "/chr" + chr + ".txt";
        } else if (chr==23) {
            refname = refdir + "/chrX.txt";
        } else if (chr==24) {
            refname = refdir + "/chrY.txt";
	} else if (chr==0) {
	    refname = refdir + "/chrM.txt";
        } else {
            throw new Exception("chr should be 0-24");
        }
        return(refname);
    }

    //////////////////////////////////////////////////////////////////////
    //////////////////////////////////////////////////////////////////////

}
