/*
 * Decompiled with CFR 0.152.
 */
package chemaxon.calculations.nmr;

import chemaxon.calculations.Geometry;
import chemaxon.calculations.clean.Cleaner;
import chemaxon.calculations.hydrogenize.Hydrogenize;
import chemaxon.calculations.nmr.MoleculeWithFocusAtom;
import chemaxon.calculations.nmr.Multiplet;
import chemaxon.calculations.nmr.MultipletComparator;
import chemaxon.calculations.nmr.NMRCoefficient;
import chemaxon.calculations.nmr.NMRShiftPredictor;
import chemaxon.calculations.nmr.NMRShiftPredictorFactory;
import chemaxon.calculations.nmr.NMRSpectrum;
import chemaxon.calculations.nmr.Shift;
import chemaxon.calculations.nmr.io.NMRExporter;
import chemaxon.calculator.Calculator;
import chemaxon.calculator.CalculatorFormatter;
import chemaxon.calculator.CalculatorLogger;
import chemaxon.calculator.Command;
import chemaxon.calculator.Parameter;
import chemaxon.formats.MolExporter;
import chemaxon.license.Licensable;
import chemaxon.license.LicenseException;
import chemaxon.license.LicenseHandler;
import chemaxon.marvin.alignment.DihedralRotator;
import chemaxon.struc.MolAtom;
import chemaxon.struc.MolBond;
import chemaxon.struc.Molecule;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;

@Parameter(description="NMR Calculator")
public class NMRCalculator
implements Calculator,
Licensable {
    private static final int CAtNo = 6;
    private static final int HAtNo = 1;
    private static final double DIHEDRAL_THRESHOLD1 = 1.0;
    private static final double DIHEDRAL_THRESHOLD2 = 179.0;
    private Molecule molecule;
    private NMRSpectrum spectrum;
    @Parameter(name="nucleus", shortName="n", description="nucleus type")
    private NMRSpectrum.Nucleus nucleus = NMRSpectrum.Nucleus.C13;
    @Parameter(name="unit", shortName="u", description="measurement unit")
    private NMRSpectrum.Unit unit = NMRSpectrum.Unit.ppm;
    @Parameter(name="format", shortName="f", description="output format. 'jcamp' for JCampDX, or any molecule format (eg. sdf)")
    private String format = "jcamp";
    @Parameter(name="coupled", shortName="c", description="consider J-coupling during 1H NMR spectrum prediction")
    private boolean coupled = true;
    @Parameter(name="frequency", shortName="r", description="measurement frequency in MHz")
    private double frequency = 500.0;
    private final double DEFAULT_C13_HW = 2.0;
    private final double DEFAULT_H1_HW = 1.0;
    private float progress;
    private CalculatorLogger logger;

    public NMRCalculator() {
    }

    public NMRCalculator(NMRSpectrum spectrum) {
        this.spectrum = spectrum;
        this.coupled = spectrum.getCouplingNeeded();
        this.frequency = spectrum.getFrequency();
        this.molecule = spectrum.getMolecule();
        this.nucleus = spectrum.getNucleusType();
        this.unit = spectrum.getUnit();
    }

    @Override
    public boolean setMolecule(Molecule molecule) {
        this.molecule = molecule.cloneMolecule();
        this.invalidate();
        return true;
    }

    public NMRSpectrum.Nucleus getNucleus() {
        return this.nucleus;
    }

    public void setNucleus(NMRSpectrum.Nucleus nucleus) {
        this.nucleus = nucleus;
        this.invalidate();
    }

    public NMRSpectrum.Unit getUnit() {
        return this.unit;
    }

    public void setUnit(NMRSpectrum.Unit unit) {
        this.unit = unit;
        if (this.spectrum != null) {
            this.spectrum.setUnit(unit);
        }
    }

    private String getFormat() {
        return this.format;
    }

    private void setFormat(String format2) {
        this.format = format2;
        this.invalidate();
    }

    public boolean isCoupled() {
        return this.coupled;
    }

    public void setCoupled(boolean coupled) {
        this.coupled = coupled;
        if (this.spectrum != null) {
            this.spectrum.setCouplingNeeded(coupled);
        }
    }

    public double getFrequency() {
        return this.frequency;
    }

    public void setFrequency(double frequency) {
        this.frequency = frequency;
        if (this.spectrum != null) {
            this.spectrum.setFrequency(frequency);
        }
    }

    private void invalidate() {
        this.spectrum = null;
    }

    @Command(name={"nmr", "cnmr", "hnmr"}, description={"Calculates NMR spectrum for the given molecule.", "Calculates 13C NMR spectrum for the given molecule.", "Calculates 1H NMR spectrum for the given molecule."}, options={"", "n=13C c=false", "n=1H"}, example="--unit Hz test.mol", group=Command.Group.NMR)
    public NMRSpectrum calculate() throws LicenseException {
        this.checkLicense();
        this.progress = 0.0f;
        if (this.spectrum == null) {
            int i;
            int atNo;
            this.spectrum = new NMRSpectrum(this.nucleus);
            this.molecule.expandSgroups();
            if (this.nucleus == NMRSpectrum.Nucleus.H1) {
                Hydrogenize.addHAtoms(this.molecule);
            }
            this.spectrum.setMolecule(this.molecule);
            this.spectrum.setUnit(this.unit);
            this.spectrum.setCouplingNeeded(this.coupled);
            this.spectrum.setFrequency(this.frequency);
            NMRShiftPredictor shiftPredictor = NMRShiftPredictorFactory.createNMRShiftPredictor();
            shiftPredictor.setNucleus(this.nucleus);
            switch (this.nucleus) {
                case C13: {
                    atNo = 6;
                    shiftPredictor.setTrainingResults(NMRCoefficient.C13);
                    if (this.unit == NMRSpectrum.Unit.ppm) {
                        this.spectrum.setHalfWidth(2.0 / this.frequency);
                        break;
                    }
                    this.spectrum.setHalfWidth(2.0);
                    break;
                }
                case H1: {
                    atNo = 1;
                    shiftPredictor.setTrainingResults(NMRCoefficient.H1);
                    if (this.unit == NMRSpectrum.Unit.ppm) {
                        this.spectrum.setHalfWidth(1.0 / this.frequency);
                        break;
                    }
                    this.spectrum.setHalfWidth(1.0);
                    break;
                }
                default: {
                    atNo = 6;
                    shiftPredictor.setTrainingResults(NMRCoefficient.C13);
                    if (this.unit == NMRSpectrum.Unit.ppm) {
                        this.spectrum.setHalfWidth(2.0 / this.frequency);
                        break;
                    }
                    this.spectrum.setHalfWidth(2.0);
                }
            }
            MolAtom[] atoms = this.molecule.getAtomArray();
            int atomCount = atoms.length;
            int numberOfNMRActiveNuclei = 0;
            for (MolAtom atom : atoms) {
                if (atom.getAtno() != atNo) continue;
                ++numberOfNMRActiveNuclei;
            }
            this.spectrum.setNumberOfNMRActiveNuclei(numberOfNMRActiveNuclei);
            Shift[] shifts = new Shift[numberOfNMRActiveNuclei];
            Object couplingConstants = null;
            if (atomCount > 0) {
                couplingConstants = new double[atomCount - 1][];
                for (int i2 = 0; i2 < ((double[][])couplingConstants).length; ++i2) {
                    couplingConstants[i2] = new double[i2 + 1];
                }
            } else {
                couplingConstants = new double[0][];
            }
            this.progress = 0.1f;
            int count = 0;
            for (i = 0; i < atomCount; ++i) {
                MolAtom[] neighbor;
                MolAtom atom = this.molecule.getAtom(i);
                if (atom.getAtno() != atNo) continue;
                double shift = shiftPredictor.predict(new MoleculeWithFocusAtom(this.molecule, i));
                shifts[count] = new Shift(i, shift);
                shifts[count].setShiftError(shiftPredictor.getShiftError());
                shifts[count].setClassification(shiftPredictor.getClassification());
                if (atom.getAtno() == 1 && (neighbor = atom.getLigands()).length == 1 && neighbor[0].getAtno() != 6) {
                    shifts[count].setClassification(NMRCoefficient.ErrorClassification.poor);
                }
                ++count;
            }
            if (this.unit == NMRSpectrum.Unit.Hz) {
                for (i = 0; i < shifts.length; ++i) {
                    shifts[i].scaleShift(this.frequency);
                }
            }
            this.spectrum.setShifts(shifts);
            this.progress = 0.3f;
            if (this.nucleus == NMRSpectrum.Nucleus.H1) {
                this.estimateVicinalJs((double[][])couplingConstants);
            }
            if (this.unit == NMRSpectrum.Unit.ppm) {
                for (i = 0; i < ((double[][])couplingConstants).length; ++i) {
                    int j = 0;
                    while (j < couplingConstants[i].length) {
                        double[] dArray = couplingConstants[i];
                        int n = j++;
                        dArray[n] = dArray[n] / this.frequency;
                    }
                }
            }
            this.spectrum.setCouplingConstants((double[][])couplingConstants);
            this.progress = 0.8f;
            double halfWidth = this.spectrum.getHalfWidth();
            ArrayList<Multiplet> ms = new ArrayList<Multiplet>();
            boolean[] atomsExamined = new boolean[numberOfNMRActiveNuclei];
            for (int i3 = 0; i3 < shifts.length; ++i3) {
                int[] indices;
                if (atomsExamined[i3]) continue;
                Multiplet m = new Multiplet(this.coupled);
                m.addAtom(shifts[i3].getAtomIndex());
                m.setShift(shifts[i3].getShift());
                m.setShiftError(shifts[i3].getShiftError());
                m.setShiftErrorClassification(shifts[i3].getShiftErrorClassification());
                atomsExamined[i3] = true;
                for (int j = i3 + 1; j < shifts.length; ++j) {
                    if (atomsExamined[j] || !(Math.abs(shifts[i3].getShift() - shifts[j].getShift()) < Double.MIN_VALUE)) continue;
                    m.addAtom(shifts[j].getAtomIndex());
                    atomsExamined[j] = true;
                }
                for (int ind : indices = m.getAtomIndices()) {
                    double[] coupling = new double[this.molecule.getAtomCount()];
                    for (int j = 0; j < coupling.length; ++j) {
                        if (ind > j) {
                            coupling[j] = couplingConstants[ind - 1][j];
                            continue;
                        }
                        if (ind >= j) continue;
                        coupling[j] = couplingConstants[j - 1][ind];
                    }
                    for (int ind2 : indices) {
                        coupling[ind2] = 0.0;
                    }
                    m.addCouplingConstants(coupling);
                }
                m.setHalfWidth(halfWidth);
                m.generateMultiplet();
                ms.add(m);
            }
            Collections.sort(ms, new MultipletComparator());
            Multiplet[] multiplets = new Multiplet[ms.size()];
            for (int i4 = 0; i4 < multiplets.length; ++i4) {
                multiplets[i4] = (Multiplet)ms.get(i4);
            }
            this.spectrum.setMultiplets(multiplets);
            this.progress = 0.9f;
            double domainMin = 0.0;
            double domainMax = 0.0;
            if (this.molecule.getAtomCount() > 0 && multiplets.length > 0) {
                domainMin = multiplets[0].getDomainMin() - 10.0 * halfWidth;
                if (domainMin > 0.0) {
                    domainMin = 0.0;
                }
                domainMax = multiplets[multiplets.length - 1].getDomainMax() + 10.0 * halfWidth;
                switch (this.nucleus) {
                    case C13: {
                        if (!(domainMax < 200.0)) break;
                        domainMax = 200.0;
                        break;
                    }
                    case H1: {
                        if (!(domainMax < 10.0)) break;
                        domainMax = 10.0;
                    }
                }
            } else {
                domainMin = 0.0;
                switch (this.nucleus) {
                    case C13: {
                        domainMax = 200.0;
                        break;
                    }
                    case H1: {
                        domainMax = 10.0;
                    }
                }
            }
            this.spectrum.setDomainMin(domainMin);
            this.spectrum.setDomainMax(domainMax);
            this.spectrum.computeFunctionScale();
        }
        this.progress = 1.0f;
        return this.spectrum;
    }

    private void estimateVicinalJs(double[][] couplingConstants) {
        double ROT_ANGLE = Math.toRadians(120.0);
        double RT = 2494.2;
        double KCAL_TO_JOULE = 4184.0;
        Molecule molClone = this.molecule.cloneMolecule();
        int[][] rings = molClone.getCSSR();
        MolBond[] bonds = molClone.getBondArray();
        boolean isCleaned = Cleaner.clean(molClone, 3, "c30");
        if (!isCleaned) {
            return;
        }
        Geometry geom = new Geometry();
        geom.setMolecule(molClone);
        DihedralRotator rotator = new DihedralRotator(molClone);
        for (MolBond bond : bonds) {
            MolAtom[] ligands2;
            MolAtom[] ligands1;
            MolAtom atom1 = bond.getAtom1();
            MolAtom atom2 = bond.getAtom2();
            int ind1 = molClone.indexOf(atom1);
            int ind2 = molClone.indexOf(atom2);
            int atNo1 = atom1.getAtno();
            int atNo2 = atom2.getAtno();
            int countJ = atom1.getExplicitHcount() * atom2.getExplicitHcount();
            if (atNo1 != 6 || atNo2 != 6 || countJ <= 0) continue;
            int[] indexH1 = new int[atom1.getExplicitHcount()];
            int[] indexH2 = new int[atom2.getExplicitHcount()];
            int count = 0;
            for (MolAtom ligand : ligands1 = atom1.getLigands()) {
                if (ligand.getAtno() != 1) continue;
                indexH1[count] = molClone.indexOf(ligand);
                ++count;
            }
            count = 0;
            for (MolAtom ligand : ligands2 = atom2.getLigands()) {
                if (ligand.getAtno() != 1) continue;
                indexH2[count] = molClone.indexOf(ligand);
                ++count;
            }
            if (this.isRotatableBond(molClone, bond, rings)) {
                int i;
                int j;
                double partFunc = 0.0;
                double[][] individualJs = new double[indexH1.length][indexH2.length];
                for (int i2 = 0; i2 < 3; ++i2) {
                    geom.setMolecule(molClone);
                    geom.calculateDreidingEnergy();
                    double boltzmannFactor = Math.exp(-4184.0 * geom.dreidingEnergy() / 2494.2);
                    partFunc += boltzmannFactor;
                    for (j = 0; j < individualJs.length; ++j) {
                        int k = 0;
                        while (k < individualJs[j].length) {
                            double dihedral = geom.dihedral(indexH1[j], ind1, ind2, indexH2[k]);
                            double[] dArray = individualJs[j];
                            int n = k++;
                            dArray[n] = dArray[n] + this.karplus(dihedral) * boltzmannFactor;
                        }
                    }
                    rotator.rotateBy(ROT_ANGLE, indexH1[0], ind1, ind2, indexH2[0]);
                }
                double J = 0.0;
                for (i = 0; i < individualJs.length; ++i) {
                    for (j = 0; j < individualJs[i].length; ++j) {
                        double[] dArray = individualJs[i];
                        int n = j;
                        dArray[n] = dArray[n] / partFunc;
                        J += individualJs[i][j];
                    }
                }
                J /= (double)countJ;
                for (i = 0; i < indexH1.length; ++i) {
                    for (j = 0; j < indexH2.length; ++j) {
                        if (indexH1[i] < indexH2[j]) {
                            couplingConstants[indexH2[j] - 1][indexH1[i]] = J;
                            continue;
                        }
                        if (indexH1[i] <= indexH2[j]) continue;
                        couplingConstants[indexH1[i] - 1][indexH2[j]] = J;
                    }
                }
                continue;
            }
            for (int i = 0; i < indexH1.length; ++i) {
                for (int j = 0; j < indexH2.length; ++j) {
                    double dihedral = geom.dihedral(indexH1[i], ind1, ind2, indexH2[j]);
                    if (Math.abs(dihedral) < 1.0) {
                        dihedral = 0.0;
                    }
                    if (Math.abs(dihedral) > 179.0) {
                        dihedral = dihedral > 0.0 ? 180.0 : -180.0;
                    }
                    double J = this.karplus(dihedral);
                    if (indexH1[i] < indexH2[j]) {
                        couplingConstants[indexH2[j] - 1][indexH1[i]] = J;
                        continue;
                    }
                    if (indexH1[i] <= indexH2[j]) continue;
                    couplingConstants[indexH1[i] - 1][indexH2[j]] = J;
                }
            }
        }
    }

    private boolean isRotatableBond(Molecule mol, MolBond bond, int[][] rings) {
        int ind1 = mol.indexOf(bond.getAtom1());
        int ind2 = mol.indexOf(bond.getAtom2());
        if (bond.getType() == 1) {
            for (int i = 0; i < rings.length; ++i) {
                boolean isRingAtom1 = false;
                boolean isRingAtom2 = false;
                for (int j = 0; j < rings[i].length; ++j) {
                    if (rings[i][j] == ind1) {
                        isRingAtom1 = true;
                    }
                    if (rings[i][j] != ind2) continue;
                    isRingAtom2 = true;
                }
                if (!isRingAtom1 || !isRingAtom2) continue;
                return false;
            }
            return true;
        }
        return false;
    }

    private double karplus(double dihedral) {
        double A = 7.76;
        double B = -1.1;
        double C = 1.4;
        double cosDihedral = Math.cos(Math.toRadians(dihedral));
        double J = 7.76 * Math.pow(cosDihedral, 2.0) + -1.1 * cosDihedral + 1.4;
        return J;
    }

    @Override
    public boolean isLicensed() {
        return LicenseHandler.getInstance().isLicensed("NMR Predictor", null);
    }

    private void checkLicense() throws LicenseException {
        LicenseHandler.getInstance().checkLicense("NMR Predictor", null);
    }

    @Override
    public void setLicenseEnvironment(String env) {
    }

    public float getProgress() {
        return this.progress;
    }

    @Override
    public CalculatorFormatter getFormatter() {
        return new CalculatorFormatter(){

            @Override
            protected String convertSingle(Object o) {
                return this.getAsString(o);
            }

            @Override
            public boolean needHeader() {
                return false;
            }

            private String getAsString(Object o) {
                if (o instanceof NMRSpectrum) {
                    try {
                        if (NMRCalculator.this.format.equalsIgnoreCase("jcamp")) {
                            return NMRExporter.exportToJcampDX((NMRSpectrum)o);
                        }
                        return MolExporter.exportToFormat(NMRExporter.exportToMol((NMRSpectrum)o), NMRCalculator.this.format);
                    }
                    catch (IOException e) {
                        return o.toString();
                    }
                }
                return o.toString();
            }
        };
    }

    @Override
    public void setLogger(CalculatorLogger logger) {
        this.logger = logger;
    }
}

