/*
 * Decompiled with CFR 0.152.
 */
package chemaxon.checkers;

import chemaxon.checkers.BondChecker;
import chemaxon.checkers.CheckerInfo;
import chemaxon.checkers.StructureCheckerErrorType;
import chemaxon.checkers.StructureCheckerHelper;
import chemaxon.checkers.result.StructureCheckerResult;
import chemaxon.struc.MolAtom;
import chemaxon.struc.MolBond;
import chemaxon.struc.Molecule;
import java.util.ArrayList;
import java.util.List;

@CheckerInfo(name="Bond Angle Checker", localMenuName="Bond Angle", description="Detects unpreferred bond angles in structural formulas", noErrorMessage="No bonds found with wrong angle", oneErrorMessage="wrong bond angle found", moreErrorMessage="wrong bond angles found", actionStringToken="bondangle")
public class BondAngleChecker
extends BondChecker {
    private static List<Double> allowedAngles = new ArrayList<Double>();
    private static List<Double> allowedAnglesForDoubleBond = new ArrayList<Double>();
    private static List<Double> allowedAnglesForTripleBond = new ArrayList<Double>();
    private static List<Double> allowedAnglesForNodeWith3Bonds = new ArrayList<Double>();
    private static List<Double> allowedAnglesForNodeWith4Bonds = new ArrayList<Double>();
    private final double tolerance;

    public BondAngleChecker() {
        super(StructureCheckerErrorType.BOND_ANGLE);
        this.tolerance = 0.02;
    }

    public BondAngleChecker(double tolerance) {
        super(StructureCheckerErrorType.BOND_ANGLE);
        this.tolerance = tolerance;
    }

    @Override
    protected StructureCheckerResult check1(Molecule molecule) {
        if (molecule.getDim() == 3 || molecule.getDim() == 0) {
            return null;
        }
        return super.check1(molecule);
    }

    @Override
    protected boolean check(Molecule molecule, MolBond bond) {
        if (molecule.isRingBond(molecule.indexOf(bond)) || bond.getType() == 9) {
            return false;
        }
        MolAtom atomWithFourBonds = null;
        if (bond.getAtom1().getBondCount() == 4) {
            atomWithFourBonds = bond.getAtom1();
        } else if (bond.getAtom2().getBondCount() == 4) {
            atomWithFourBonds = bond.getAtom2();
        }
        if (atomWithFourBonds != null && this.checkForRingWithTwoTerminalBonds(molecule, bond, atomWithFourBonds)) {
            MolBond bond2;
            int i;
            MolBond nonRingBond = null;
            ArrayList<Double> cosAs = new ArrayList<Double>();
            for (i = 0; i < atomWithFourBonds.getBondCount(); ++i) {
                bond2 = atomWithFourBonds.getBond(i);
                if (!bond2.equals(bond) && !molecule.isRingBond(molecule.indexOf(bond2))) {
                    nonRingBond = bond2;
                    continue;
                }
                if (bond2.equals(bond)) continue;
                cosAs.add(StructureCheckerHelper.calculateCosAFromScalarMul(bond, bond2));
            }
            for (i = 0; i < atomWithFourBonds.getBondCount(); ++i) {
                bond2 = atomWithFourBonds.getBond(i);
                double cosA = StructureCheckerHelper.calculateCosAFromScalarMul(nonRingBond, bond2);
                if (bond2.equals(nonRingBond) || !molecule.isRingBond(molecule.indexOf(bond2)) || this.find(cosA, cosAs, true)) continue;
                return true;
            }
            return false;
        }
        MolAtom atomWithThreeBonds = null;
        if (bond.getAtom1().getBondCount() == 3) {
            atomWithThreeBonds = bond.getAtom1();
        } else if (bond.getAtom2().getBondCount() == 3) {
            atomWithThreeBonds = bond.getAtom2();
        }
        if (atomWithThreeBonds != null && this.checkForRingWithOneTerminalBond(molecule, bond, atomWithThreeBonds)) {
            ArrayList<Double> cosAs = new ArrayList<Double>();
            for (int i = 0; i < atomWithThreeBonds.getBondCount(); ++i) {
                double cosA;
                MolBond bond2 = atomWithThreeBonds.getBond(i);
                if (bond.equals(bond2) || this.find(cosA = StructureCheckerHelper.calculateCosAFromScalarMul(bond, bond2), cosAs, true)) continue;
                cosAs.add(cosA);
            }
            return !cosAs.isEmpty();
        }
        List<MolBond> neighbourBonds = StructureCheckerHelper.createNeighbourBonds(molecule, bond, true);
        for (int j = 0; j < molecule.getBondCount(); ++j) {
            MolBond bond2 = molecule.getBond(j);
            if (bond.equals(bond2) || !neighbourBonds.contains(bond2) || !this.badAngle(bond, bond2)) continue;
            return true;
        }
        return false;
    }

    private boolean checkForRingWithOneTerminalBond(Molecule molecule, MolBond bond, MolAtom atomWithThreeBonds) {
        int ringBondCount = 0;
        for (int i = 0; i < atomWithThreeBonds.getBondCount(); ++i) {
            MolBond bond1 = atomWithThreeBonds.getBond(i);
            if (bond.equals(bond1) || !molecule.isRingBond(molecule.indexOf(bond1))) continue;
            ++ringBondCount;
        }
        return ringBondCount == 2;
    }

    private boolean checkForRingWithTwoTerminalBonds(Molecule molecule, MolBond bond, MolAtom atom) {
        int ringBondCount = 0;
        for (int i = 0; i < atom.getBondCount(); ++i) {
            MolBond bond1 = atom.getBond(i);
            if (bond1.equals(bond) || !molecule.isRingBond(molecule.indexOf(bond1))) continue;
            ++ringBondCount;
        }
        boolean specialRingCondition = ringBondCount == 2;
        return specialRingCondition;
    }

    public static int getRingIndex(Molecule molecule, MolBond bond) {
        int index = molecule.indexOf(bond);
        int[][] sssr = molecule.getSSSRBonds();
        for (int i = 0; i < sssr.length; ++i) {
            int[] is = sssr[i];
            for (int j = 0; j < is.length; ++j) {
                if (is[j] != index) continue;
                return i;
            }
        }
        return -1;
    }

    private boolean badAngle(MolBond bond1, MolBond bond2) {
        double cosA = StructureCheckerHelper.calculateCosAFromScalarMul(bond1, bond2);
        if (cosA < -1.0) {
            cosA = -1.0;
        } else if (cosA > 1.0) {
            cosA = 1.0;
        }
        List<Double> allowedAngles = this.getAllowedAngles(bond1, bond2);
        return !this.find(Math.acos(cosA), allowedAngles, false);
    }

    private List<Double> getAllowedAngles(MolBond bond1, MolBond bond2) {
        if (bond1.getType() == 3 || bond2.getType() == 3) {
            return allowedAnglesForTripleBond;
        }
        if (bond1.getType() == 2 || bond2.getType() == 2) {
            return allowedAnglesForDoubleBond;
        }
        MolAtom atom = bond1.getAtom1() == bond2.getAtom1() || bond1.getAtom1() == bond2.getAtom2() ? bond1.getAtom1() : bond1.getAtom2();
        if (atom.getBondCount() == 3) {
            return allowedAnglesForNodeWith3Bonds;
        }
        if (atom.getBondCount() == 4) {
            return allowedAnglesForNodeWith4Bonds;
        }
        return allowedAngles;
    }

    private boolean find(double rad, List<Double> allowedAngles, boolean remove) {
        for (double rads : allowedAngles) {
            if (!(Math.abs(rad - rads) < this.tolerance)) continue;
            if (remove) {
                allowedAngles.remove(rads);
            }
            return true;
        }
        return false;
    }

    static {
        allowedAngles.add(2.0943951023931953);
        allowedAnglesForNodeWith3Bonds.add(2.0943951023931953);
        allowedAnglesForNodeWith4Bonds.add(Math.PI);
        allowedAnglesForNodeWith4Bonds.add(1.5707963267948966);
        allowedAnglesForNodeWith4Bonds.add(1.0471975511965976);
        allowedAnglesForNodeWith4Bonds.add(2.0943951023931953);
        allowedAnglesForDoubleBond.add(Math.PI);
        allowedAnglesForDoubleBond.add(1.5707963267948966);
        allowedAnglesForDoubleBond.add(2.0943951023931953);
        allowedAnglesForTripleBond.add(Math.PI);
    }
}

