/*
 * Decompiled with CFR 0.152.
 */
package chemaxon.common.util;

public class GeomCalc {
    public static final int DEFAULT_VECTOR_CAPACITY = 64;
    public static final int DEFAULT_MATRIX_CAPACITY = 256;
    private static int vectorCapacity = 64;
    private static int matrixCapacity = 256;
    private static float[][] vectors = new float[64][4];
    private static float[][] matrices = new float[256][16];
    private static Queue freeVectorIndices = null;
    private static Queue freeMatrixIndices = null;
    private static boolean[] allocatedVectorIndices = null;
    private static boolean[] allocatedMatrixIndices = null;

    public static String vectorToString(int vIdx) {
        float[] v = vectors[vIdx];
        return "| " + v[0] + " |\n" + "| " + v[1] + " |\n" + "| " + v[2] + " |\n" + "| " + v[3] + " |\n";
    }

    public static synchronized int newVector() {
        if (freeVectorIndices.isEmpty()) {
            GeomCalc.increaseVectorCapacity();
        }
        try {
            int i = freeVectorIndices.pop();
            GeomCalc.allocatedVectorIndices[i] = true;
            GeomCalc.vectors[i][0] = 0.0f;
            GeomCalc.vectors[i][1] = 0.0f;
            GeomCalc.vectors[i][2] = 0.0f;
            GeomCalc.vectors[i][3] = 1.0f;
            return i;
        }
        catch (Queue.Underflow u) {
            u.printStackTrace();
            throw new GeomCalcException("Fatal internal error: too many vectors have been allocated. ");
        }
    }

    public static synchronized int newVector(int vIdx) {
        if (freeVectorIndices.isEmpty()) {
            GeomCalc.increaseVectorCapacity();
        }
        try {
            int i = freeVectorIndices.pop();
            GeomCalc.allocatedVectorIndices[i] = true;
            GeomCalc.vectors[i][0] = vectors[vIdx][0];
            GeomCalc.vectors[i][1] = vectors[vIdx][1];
            GeomCalc.vectors[i][2] = vectors[vIdx][2];
            GeomCalc.vectors[i][3] = 1.0f;
            return i;
        }
        catch (Queue.Underflow u) {
            throw new GeomCalcException("Fatal internal error: too many vectors have been allocated.");
        }
    }

    public static synchronized int newVector(float x, float y, float z) throws GeomCalcException {
        if (freeVectorIndices.isEmpty()) {
            GeomCalc.increaseVectorCapacity();
        }
        try {
            int i = freeVectorIndices.pop();
            GeomCalc.allocatedVectorIndices[i] = true;
            GeomCalc.vectors[i][0] = x;
            GeomCalc.vectors[i][1] = y;
            GeomCalc.vectors[i][2] = z;
            GeomCalc.vectors[i][3] = 1.0f;
            return i;
        }
        catch (Queue.Underflow u) {
            throw new GeomCalcException("Fatal internal error: too many vectors have been allocated.");
        }
    }

    public static synchronized int newVector(float[] xyz) throws GeomCalcException {
        if (freeVectorIndices.isEmpty()) {
            GeomCalc.increaseVectorCapacity();
        }
        try {
            int i = freeVectorIndices.pop();
            GeomCalc.allocatedVectorIndices[i] = true;
            GeomCalc.vectors[i][0] = xyz[0];
            GeomCalc.vectors[i][1] = xyz[1];
            GeomCalc.vectors[i][2] = xyz[2];
            GeomCalc.vectors[i][3] = 1.0f;
            return i;
        }
        catch (Queue.Underflow u) {
            throw new GeomCalcException("Fatal internal error: too many vectors have been allocated.");
        }
    }

    public static synchronized void deleteVector(int i) throws GeomCalcException {
        if (i == 0) {
            return;
        }
        GeomCalc.checkAllocatedVector(i);
        try {
            freeVectorIndices.push(i);
            GeomCalc.allocatedVectorIndices[i] = false;
        }
        catch (Queue.Overflow o) {
            throw new GeomCalcException("Fatal internal error: inconsistency.");
        }
    }

    public static float[] getVector(int vectorIndex) {
        return vectors[vectorIndex];
    }

    public static float[] getFloatVector(int vectorIndex) {
        float[] ret = new float[3];
        System.arraycopy(vectors[vectorIndex], 0, ret, 0, 3);
        return ret;
    }

    public static float[] getVectorAsArray(int vectorIndex) {
        float[] ret = new float[4];
        System.arraycopy(vectors[vectorIndex], 0, ret, 0, 4);
        return ret;
    }

    public static float getX(int vectorIndex) {
        return vectors[vectorIndex][0];
    }

    public static float getY(int vectorIndex) {
        return vectors[vectorIndex][1];
    }

    public static float getZ(int vectorIndex) {
        return vectors[vectorIndex][2];
    }

    public static float get(int vectorIndex, int i) {
        return vectors[vectorIndex][i];
    }

    public static void setVector(int vectorIndex, float x, float y, float z) throws GeomCalcException {
        GeomCalc.checkAllocatedVector(vectorIndex);
        float[] v = vectors[vectorIndex];
        v[0] = x;
        v[1] = y;
        v[2] = z;
        v[3] = 1.0f;
    }

    public static void setVector(int vectorIndex, float[] coords) throws GeomCalcException {
        GeomCalc.checkAllocatedVector(vectorIndex);
        float[] v = vectors[vectorIndex];
        v[0] = coords[0];
        v[1] = coords[1];
        v[2] = coords[2];
        v[3] = coords.length == 4 ? coords[3] : 1.0f;
    }

    public static void setVector(int vectorIndex, int vector) throws GeomCalcException {
        GeomCalc.checkAllocatedVector(vectorIndex);
        GeomCalc.checkAllocatedVector(vector);
        float[] v1 = vectors[vectorIndex];
        float[] v2 = vectors[vector];
        v1[0] = v2[0];
        v1[1] = v2[1];
        v1[2] = v2[2];
        v1[3] = v2[3];
    }

    public static void setVectorX(int vectorIndex, float x) throws GeomCalcException {
        GeomCalc.checkAllocatedVector(vectorIndex);
        GeomCalc.vectors[vectorIndex][0] = x;
    }

    public static void setVectorY(int vectorIndex, float y) throws GeomCalcException {
        GeomCalc.checkAllocatedVector(vectorIndex);
        GeomCalc.vectors[vectorIndex][1] = y;
    }

    public static void setVectorZ(int vectorIndex, float z) throws GeomCalcException {
        GeomCalc.checkAllocatedVector(vectorIndex);
        GeomCalc.vectors[vectorIndex][2] = z;
    }

    public static void setVectorCoordinate(int vectorIndex, int i, float v) throws GeomCalcException {
        GeomCalc.checkAllocatedVector(vectorIndex);
        GeomCalc.vectors[vectorIndex][i] = v;
    }

    public static void setNullVector(int vectorIndex) throws GeomCalcException {
        GeomCalc.checkAllocatedVector(vectorIndex);
        GeomCalc.vectors[vectorIndex][0] = 0.0f;
        GeomCalc.vectors[vectorIndex][1] = 0.0f;
        GeomCalc.vectors[vectorIndex][2] = 0.0f;
        GeomCalc.vectors[vectorIndex][3] = 1.0f;
    }

    public static void increase(int v1, int v2) throws GeomCalcException {
        for (int i = 0; i < 3; ++i) {
            GeomCalc.vectors[v1][i] = vectors[v1][i] + vectors[v2][i];
        }
    }

    public static void increase(int v1, float[] v2) throws GeomCalcException {
        for (int i = 0; i < 3; ++i) {
            GeomCalc.vectors[v1][i] = vectors[v1][i] + v2[i];
        }
    }

    public static int add(int v1, int v2) throws GeomCalcException {
        int r = GeomCalc.newVector();
        for (int i = 0; i < 3; ++i) {
            GeomCalc.vectors[r][i] = vectors[v1][i] + vectors[v2][i];
        }
        return r;
    }

    public static int add(int v1, float[] v2) throws GeomCalcException {
        int r = GeomCalc.newVector();
        for (int i = 0; i < 3; ++i) {
            GeomCalc.vectors[r][i] = vectors[v1][i] + v2[i];
        }
        return r;
    }

    public static int add(float[] v1, float[] v2) throws GeomCalcException {
        int r = GeomCalc.newVector();
        for (int i = 0; i < 3; ++i) {
            GeomCalc.vectors[r][i] = v1[i] + v2[i];
        }
        return r;
    }

    public static void decrease(int v1, int v2) throws GeomCalcException {
        for (int i = 0; i < 3; ++i) {
            GeomCalc.vectors[v1][i] = vectors[v1][i] - vectors[v2][i];
        }
    }

    public static void decrease(int v1, float[] v2) throws GeomCalcException {
        for (int i = 0; i < 3; ++i) {
            GeomCalc.vectors[v1][i] = vectors[v1][i] - v2[i];
        }
    }

    public static void sincrease(int v, float d) {
        for (int i = 0; i < 3; ++i) {
            GeomCalc.vectors[v][i] = vectors[v][i] + d;
        }
    }

    public static void sdecrease(int v, float d) {
        for (int i = 0; i < 3; ++i) {
            GeomCalc.vectors[v][i] = vectors[v][i] - d;
        }
    }

    public static int sub(int v1, int v2) throws GeomCalcException {
        int r = GeomCalc.newVector();
        for (int i = 0; i < 3; ++i) {
            GeomCalc.vectors[r][i] = vectors[v1][i] - vectors[v2][i];
        }
        return r;
    }

    public static void scale(int v1, float value) throws GeomCalcException {
        int i = 0;
        while (i < 3) {
            float[] fArray = vectors[v1];
            int n = i++;
            fArray[n] = fArray[n] * value;
        }
    }

    public static void scale(int v1, float vx, float vy, float vz) throws GeomCalcException {
        float[] fArray = vectors[v1];
        fArray[0] = fArray[0] * vx;
        float[] fArray2 = vectors[v1];
        fArray2[1] = fArray2[1] * vy;
        float[] fArray3 = vectors[v1];
        fArray3[2] = fArray3[2] * vz;
    }

    public static int mult(int v1, float value) throws GeomCalcException {
        int r = GeomCalc.newVector();
        for (int i = 0; i < 3; ++i) {
            GeomCalc.vectors[r][i] = vectors[v1][i] * value;
        }
        return r;
    }

    public static float dot(int vi1, int vi2) throws GeomCalcException {
        float[] v1 = vectors[vi1];
        float[] v2 = vectors[vi2];
        return v1[0] * v2[0] + v1[1] * v2[1] + v1[2] * v2[2];
    }

    public static int cross(int vi1, int vi2) throws GeomCalcException {
        int r = GeomCalc.newVector();
        float[] v1 = vectors[vi1];
        float[] v2 = vectors[vi2];
        GeomCalc.vectors[r][0] = v1[1] * v2[2] - v1[2] * v2[1];
        GeomCalc.vectors[r][1] = -v1[0] * v2[2] + v1[2] * v2[0];
        GeomCalc.vectors[r][2] = v1[0] * v2[1] - v1[1] * v2[0];
        GeomCalc.vectors[r][3] = 1.0f;
        return r;
    }

    public static double length(int i) throws GeomCalcException {
        float[] v = vectors[i];
        return Math.sqrt(v[0] * v[0] + v[1] * v[1] + v[2] * v[2]);
    }

    public static float length2(int i) throws GeomCalcException {
        float[] v = vectors[i];
        return v[0] * v[0] + v[1] * v[1] + v[2] * v[2];
    }

    public static void normalize(int i) throws GeomCalcException {
        double t = GeomCalc.length(i);
        if (t != 0.0 && t != 1.0) {
            float[] fArray = vectors[i];
            fArray[0] = (float)((double)fArray[0] / t);
            float[] fArray2 = vectors[i];
            fArray2[1] = (float)((double)fArray2[1] / t);
            float[] fArray3 = vectors[i];
            fArray3[2] = (float)((double)fArray3[2] / t);
        }
    }

    public static double angle(int v1, int v2) throws GeomCalcException {
        return GeomCalc.angleRadian(v1, v2) / Math.PI * 180.0;
    }

    public static double angle(int v1, int v2, int v3) {
        int p1 = GeomCalc.sub(v1, v2);
        int p2 = GeomCalc.sub(v3, v2);
        double angle = GeomCalc.angle(p1, p2);
        GeomCalc.deleteVector(p1);
        GeomCalc.deleteVector(p2);
        return angle;
    }

    public static double angleRadian(int v1, int v2) throws GeomCalcException {
        double length1 = GeomCalc.length(v1);
        double length2 = GeomCalc.length(v2);
        if (length1 == 0.0 || length2 == 0.0) {
            return 0.0;
        }
        double d = (double)GeomCalc.dot(v1, v2) / (length1 * length2);
        if (d >= 1.0) {
            return 0.0;
        }
        if (d <= -1.0) {
            return Math.PI;
        }
        return Math.acos(d);
    }

    public static double dihedral(int v1, int v2, int v3, int v4) {
        int p4;
        int p3;
        int n2;
        int p2;
        int p1 = GeomCalc.sub(v1, v2);
        int n1 = GeomCalc.cross(p1, p2 = GeomCalc.sub(v3, v2));
        double dihedral = GeomCalc.angle(n1, n2 = GeomCalc.cross(p3 = GeomCalc.sub(v2, v3), p4 = GeomCalc.sub(v4, v3)));
        dihedral = dihedral != 0.0 && GeomCalc.dot(n1, p4) < 0.0f ? dihedral : -dihedral;
        GeomCalc.deleteVector(p1);
        GeomCalc.deleteVector(p2);
        GeomCalc.deleteVector(p3);
        GeomCalc.deleteVector(p4);
        GeomCalc.deleteVector(n1);
        GeomCalc.deleteVector(n2);
        return dihedral;
    }

    public static double distance(int v1, int v2) {
        float[] p1 = vectors[v1];
        float[] p2 = vectors[v2];
        return Math.sqrt(Math.pow(p2[2] - p1[2], 2.0) + Math.pow(p2[1] - p1[1], 2.0) + Math.pow(p2[0] - p1[0], 2.0));
    }

    public static double distance2(int v1, int v2) {
        float[] p1 = vectors[v1];
        float[] p2 = vectors[v2];
        return Math.pow(p2[2] - p1[2], 2.0) + Math.pow(p2[1] - p1[1], 2.0) + Math.pow(p2[0] - p1[0], 2.0);
    }

    public static int half(int vi1, int vi2) {
        int r = GeomCalc.newVector();
        float[] v1 = vectors[vi1];
        float[] v2 = vectors[vi2];
        GeomCalc.vectors[r][0] = (v1[0] + v2[0]) / 2.0f;
        GeomCalc.vectors[r][1] = (v1[1] + v2[1]) / 2.0f;
        GeomCalc.vectors[r][2] = (v1[2] + v2[2]) / 2.0f;
        GeomCalc.vectors[r][3] = 1.0f;
        return r;
    }

    public static int half(float[] v1, float[] v2) {
        int r = GeomCalc.newVector();
        GeomCalc.vectors[r][0] = (v1[0] + v2[0]) / 2.0f;
        GeomCalc.vectors[r][1] = (v1[1] + v2[1]) / 2.0f;
        GeomCalc.vectors[r][2] = (v1[2] + v2[2]) / 2.0f;
        GeomCalc.vectors[r][3] = 1.0f;
        return r;
    }

    public static double distance(float[] v1, float[] v2) {
        return Math.sqrt(Math.pow(v2[2] - v1[2], 2.0) + Math.pow(v2[1] - v1[1], 2.0) + Math.pow(v2[0] - v1[0], 2.0));
    }

    public static double angle(float[] v1, float[] v2, float[] v3) {
        int p1 = GeomCalc.newVector(v1[0] - v2[0], v1[1] - v2[1], v1[2] - v2[2]);
        int p2 = GeomCalc.newVector(v3[0] - v2[0], v3[1] - v2[1], v3[2] - v2[2]);
        double a = GeomCalc.angle(p1, p2);
        GeomCalc.deleteVector(p1);
        GeomCalc.deleteVector(p2);
        return a;
    }

    public static double angleRadian(float[] v1, float[] v2, float[] v3) {
        int p1 = GeomCalc.newVector(v1[0] - v2[0], v1[1] - v2[1], v1[2] - v2[2]);
        int p2 = GeomCalc.newVector(v3[0] - v2[0], v3[1] - v2[1], v3[2] - v2[2]);
        double a = GeomCalc.angleRadian(p1, p2);
        GeomCalc.deleteVector(p1);
        GeomCalc.deleteVector(p2);
        return a;
    }

    public static double dihedral(float[] v1, float[] v2, float[] v3, float[] v4) {
        int p1 = GeomCalc.newVector(v1[0], v1[1], v1[2]);
        int p2 = GeomCalc.newVector(v2[0], v2[1], v2[2]);
        int p3 = GeomCalc.newVector(v3[0], v3[1], v3[2]);
        int p4 = GeomCalc.newVector(v4[0], v4[1], v4[2]);
        double dihedral = GeomCalc.dihedral(p1, p2, p3, p4);
        GeomCalc.deleteVector(p1);
        GeomCalc.deleteVector(p2);
        GeomCalc.deleteVector(p3);
        GeomCalc.deleteVector(p4);
        return dihedral;
    }

    public static double distance(double[] v1, double[] v2) {
        return Math.sqrt(Math.pow(v2[2] - v1[2], 2.0) + Math.pow(v2[1] - v1[1], 2.0) + Math.pow(v2[0] - v1[0], 2.0));
    }

    public static double distance(double x1, double y1, double z1, double x2, double y2, double z2) {
        return Math.sqrt(Math.pow(z2 - z1, 2.0) + Math.pow(y2 - y1, 2.0) + Math.pow(x2 - x1, 2.0));
    }

    public static double angle(double[] v1, double[] v2, double[] v3) {
        int p1 = GeomCalc.newVector((float)(v1[0] - v2[0]), (float)(v1[1] - v2[1]), (float)(v1[2] - v2[2]));
        int p2 = GeomCalc.newVector((float)(v3[0] - v2[0]), (float)(v3[1] - v2[1]), (float)(v3[2] - v2[2]));
        double a = GeomCalc.angle(p1, p2);
        GeomCalc.deleteVector(p1);
        GeomCalc.deleteVector(p2);
        return a;
    }

    public static double angleRadian(double[] v1, double[] v2, double[] v3) {
        int p1 = GeomCalc.newVector((float)(v1[0] - v2[0]), (float)(v1[1] - v2[1]), (float)(v1[2] - v2[2]));
        int p2 = GeomCalc.newVector((float)(v3[0] - v2[0]), (float)(v3[1] - v2[1]), (float)(v3[2] - v2[2]));
        double a = GeomCalc.angleRadian(p1, p2);
        GeomCalc.deleteVector(p1);
        GeomCalc.deleteVector(p2);
        return a;
    }

    public static double dihedral(double[] v1, double[] v2, double[] v3, double[] v4) {
        int p1 = GeomCalc.newVector((float)v1[0], (float)v1[1], (float)v1[2]);
        int p2 = GeomCalc.newVector((float)v2[0], (float)v2[1], (float)v2[2]);
        int p3 = GeomCalc.newVector((float)v3[0], (float)v3[1], (float)v3[2]);
        int p4 = GeomCalc.newVector((float)v4[0], (float)v4[1], (float)v4[2]);
        double dihedral = GeomCalc.dihedral(p1, p2, p3, p4);
        GeomCalc.deleteVector(p1);
        GeomCalc.deleteVector(p2);
        GeomCalc.deleteVector(p3);
        GeomCalc.deleteVector(p4);
        return dihedral;
    }

    public static String matrixToString(int mIdx) {
        float[] m = matrices[mIdx];
        return "| " + m[0] + " " + m[1] + " " + m[2] + " " + m[3] + " |\n" + "| " + m[4] + " " + m[5] + " " + m[6] + " " + m[7] + " |\n" + "| " + m[8] + " " + m[9] + " " + m[10] + " " + m[11] + " |\n" + "| " + m[12] + " " + m[13] + " " + m[14] + " " + m[15] + " |\n";
    }

    public static String matrixToString(float[] m) {
        return "| " + m[0] + " " + m[1] + " " + m[2] + " " + m[3] + " |\n" + "| " + m[4] + " " + m[5] + " " + m[6] + " " + m[7] + " |\n" + "| " + m[8] + " " + m[9] + " " + m[10] + " " + m[11] + " |\n" + "| " + m[12] + " " + m[13] + " " + m[14] + " " + m[15] + " |\n";
    }

    public static synchronized int newMatrix() throws GeomCalcException {
        if (freeMatrixIndices.isEmpty()) {
            GeomCalc.increaseMatrixCapacity();
        }
        try {
            int i = freeMatrixIndices.pop();
            GeomCalc.allocatedMatrixIndices[i] = true;
            return i;
        }
        catch (Queue.Underflow u) {
            throw new GeomCalcException("Fatal internal error: too many matrices have been allocated.");
        }
    }

    public static synchronized int newMatrix(float[] coords) throws GeomCalcException {
        if (freeMatrixIndices.isEmpty()) {
            GeomCalc.increaseMatrixCapacity();
        }
        try {
            int i = freeMatrixIndices.pop();
            GeomCalc.allocatedMatrixIndices[i] = true;
            System.arraycopy(coords, 0, matrices[i], 0, 16);
            return i;
        }
        catch (Queue.Underflow u) {
            throw new GeomCalcException("Fatal internal error: too many matrices have been allocated.");
        }
    }

    public static synchronized void deleteMatrix(int i) throws GeomCalcException {
        if (i == 0) {
            return;
        }
        GeomCalc.checkAllocatedMatrix(i);
        try {
            freeMatrixIndices.push(i);
            GeomCalc.allocatedMatrixIndices[i] = false;
        }
        catch (Queue.Overflow o) {
            throw new GeomCalcException("Fatal internal error: inconsistency.");
        }
    }

    public static float[] getMatrix(int matrixIndex) {
        return matrices[matrixIndex];
    }

    public static float[] getMatrixAsArray(int matrixIndex) {
        float[] ret = new float[16];
        System.arraycopy(matrices[matrixIndex], 0, ret, 0, 16);
        return ret;
    }

    public static void setIdentity(int matrixIndex) throws GeomCalcException {
        GeomCalc.checkAllocatedMatrix(matrixIndex);
        float[] m = matrices[matrixIndex];
        m[0] = 1.0f;
        m[1] = 0.0f;
        m[2] = 0.0f;
        m[3] = 0.0f;
        m[4] = 0.0f;
        m[5] = 1.0f;
        m[6] = 0.0f;
        m[7] = 0.0f;
        m[8] = 0.0f;
        m[9] = 0.0f;
        m[10] = 1.0f;
        m[11] = 0.0f;
        m[12] = 0.0f;
        m[13] = 0.0f;
        m[14] = 0.0f;
        m[15] = 1.0f;
    }

    public static void setIdentity(float[] m) {
        m[0] = 1.0f;
        m[1] = 0.0f;
        m[2] = 0.0f;
        m[3] = 0.0f;
        m[4] = 0.0f;
        m[5] = 1.0f;
        m[6] = 0.0f;
        m[7] = 0.0f;
        m[8] = 0.0f;
        m[9] = 0.0f;
        m[10] = 1.0f;
        m[11] = 0.0f;
        m[12] = 0.0f;
        m[13] = 0.0f;
        m[14] = 0.0f;
        m[15] = 1.0f;
    }

    public static void setIdentity(double[] m) {
        m[0] = 1.0;
        m[1] = 0.0;
        m[2] = 0.0;
        m[3] = 0.0;
        m[4] = 0.0;
        m[5] = 1.0;
        m[6] = 0.0;
        m[7] = 0.0;
        m[8] = 0.0;
        m[9] = 0.0;
        m[10] = 1.0;
        m[11] = 0.0;
        m[12] = 0.0;
        m[13] = 0.0;
        m[14] = 0.0;
        m[15] = 1.0;
    }

    public static void setMatrix(int matrixIndex, float[] coords) throws GeomCalcException {
        GeomCalc.checkAllocatedMatrix(matrixIndex);
        System.arraycopy(coords, 0, matrices[matrixIndex], 0, 16);
    }

    public static void setMatrixRow(int matrixIndex, int rowIndex, int vectorIndex) {
        float[] m = matrices[matrixIndex];
        float[] v = vectors[vectorIndex];
        m[rowIndex * 4] = v[0];
        m[rowIndex * 4 + 1] = v[1];
        m[rowIndex * 4 + 2] = v[2];
    }

    public static void setRowVectorSystem(int matrixIndex, int vi1, int vi2, int vi3) {
        float[] m = matrices[matrixIndex];
        m[0] = vectors[vi1][0];
        m[1] = vectors[vi1][1];
        m[2] = vectors[vi1][2];
        m[3] = 0.0f;
        m[4] = vectors[vi2][0];
        m[5] = vectors[vi2][1];
        m[6] = vectors[vi2][2];
        m[7] = 0.0f;
        m[8] = vectors[vi3][0];
        m[9] = vectors[vi3][1];
        m[10] = vectors[vi3][2];
        m[11] = 0.0f;
        m[12] = 0.0f;
        m[13] = 0.0f;
        m[14] = 0.0f;
        m[15] = 1.0f;
    }

    public static int invertMatrix(int mIdx) throws GeomCalcException {
        int im = GeomCalc.newMatrix();
        GeomCalc.setIdentity(im);
        float a = matrices[mIdx][0];
        float b = matrices[mIdx][1];
        float c = matrices[mIdx][2];
        float d = matrices[mIdx][4];
        float e = matrices[mIdx][5];
        float f = matrices[mIdx][6];
        float g = matrices[mIdx][8];
        float h = matrices[mIdx][9];
        float i = matrices[mIdx][10];
        GeomCalc.matrices[im][0] = e * i - f * h;
        GeomCalc.matrices[im][1] = h * c - b * i;
        GeomCalc.matrices[im][2] = b * f - e * c;
        GeomCalc.matrices[im][4] = f * g - d * i;
        GeomCalc.matrices[im][5] = a * i - c * g;
        GeomCalc.matrices[im][6] = d * c - a * f;
        GeomCalc.matrices[im][8] = d * h - e * g;
        GeomCalc.matrices[im][9] = b * g - a * h;
        GeomCalc.matrices[im][10] = a * e - b * d;
        return im;
    }

    public static void invertMatrix(int mIdx, int imIdx) throws GeomCalcException {
        GeomCalc.setIdentity(imIdx);
        float a = matrices[mIdx][0];
        float b = matrices[mIdx][1];
        float c = matrices[mIdx][2];
        float d = matrices[mIdx][4];
        float e = matrices[mIdx][5];
        float f = matrices[mIdx][6];
        float g = matrices[mIdx][8];
        float h = matrices[mIdx][9];
        float i = matrices[mIdx][10];
        GeomCalc.matrices[imIdx][0] = e * i - f * h;
        GeomCalc.matrices[imIdx][1] = h * c - b * i;
        GeomCalc.matrices[imIdx][2] = b * f - e * c;
        GeomCalc.matrices[imIdx][4] = f * g - d * i;
        GeomCalc.matrices[imIdx][5] = a * i - c * g;
        GeomCalc.matrices[imIdx][6] = d * c - a * f;
        GeomCalc.matrices[imIdx][8] = d * h - e * g;
        GeomCalc.matrices[imIdx][9] = b * g - a * h;
        GeomCalc.matrices[imIdx][10] = a * e - b * d;
    }

    public static void increaseMatrix(int m1, int m2) {
        for (int i = 0; i < 16; ++i) {
            float[] fArray = matrices[m1];
            int n = i;
            fArray[n] = fArray[n] + matrices[m2][i];
        }
    }

    public static void decreaseMatrix(int m1, int m2) {
        for (int i = 0; i < 16; ++i) {
            float[] fArray = matrices[m1];
            int n = i;
            fArray[n] = fArray[n] - matrices[m2][i];
        }
    }

    public static void multMatrix(int a, int b, float[] c) {
        GeomCalc.multMatrix(matrices[a], matrices[b], c);
    }

    public static void multMatrix(float[] a, float[] b, float[] c) {
        for (int i = 0; i < 4; ++i) {
            for (int j = 0; j < 4; ++j) {
                c[i * 4 + j] = 0.0f;
                for (int k = 0; k < 4; ++k) {
                    int n = i * 4 + j;
                    c[n] = c[n] + a[i * 4 + k] * b[k * 4 + j];
                }
            }
        }
    }

    public static void scaleMatrix(int mIdx, float v) {
        int i = 0;
        while (i < 16) {
            float[] fArray = matrices[mIdx];
            int n = i++;
            fArray[n] = fArray[n] * v;
        }
    }

    public static void transposeMatrix(int mIdx) {
        float[] m = matrices[mIdx];
        float v = m[1];
        m[1] = m[4];
        m[4] = v;
        v = m[2];
        m[2] = m[8];
        m[8] = v;
        v = m[3];
        m[3] = m[12];
        m[12] = v;
        v = m[6];
        m[6] = m[9];
        m[9] = v;
        v = m[7];
        m[7] = m[13];
        m[13] = v;
        v = m[11];
        m[11] = m[14];
        m[14] = v;
    }

    public static int multiplyVectorWithMatrix(int vIdx, int mIdx) throws GeomCalcException {
        int r = GeomCalc.newVector();
        float[] v = vectors[vIdx];
        float[] m = matrices[mIdx];
        GeomCalc.vectors[r][0] = m[0] * v[0] + m[4] * v[1] + m[8] * v[2] + m[12] * v[3];
        GeomCalc.vectors[r][1] = m[1] * v[0] + m[5] * v[1] + m[9] * v[2] + m[13] * v[3];
        GeomCalc.vectors[r][2] = m[2] * v[0] + m[6] * v[1] + m[10] * v[2] + m[14] * v[3];
        GeomCalc.vectors[r][3] = m[3] * v[0] + m[7] * v[1] + m[11] * v[2] + m[15] * v[3];
        return r;
    }

    public static int multiplyVectorWithMatrix(int vIdx, float[] m) throws GeomCalcException {
        int r = GeomCalc.newVector();
        float[] v = vectors[vIdx];
        GeomCalc.vectors[r][0] = m[0] * v[0] + m[4] * v[1] + m[8] * v[2] + m[12];
        GeomCalc.vectors[r][1] = m[1] * v[0] + m[5] * v[1] + m[9] * v[2] + m[13];
        GeomCalc.vectors[r][2] = m[2] * v[0] + m[6] * v[1] + m[10] * v[2] + m[14];
        return r;
    }

    public static void multVectorWithMatrix(int vIdx, int mIdx) throws GeomCalcException {
        int r = GeomCalc.newVector();
        float[] v = vectors[vIdx];
        float[] m = matrices[mIdx];
        GeomCalc.vectors[r][0] = m[0] * v[0] + m[4] * v[1] + m[8] * v[2] + m[12] * v[3];
        GeomCalc.vectors[r][1] = m[1] * v[0] + m[5] * v[1] + m[9] * v[2] + m[13] * v[3];
        GeomCalc.vectors[r][2] = m[2] * v[0] + m[6] * v[1] + m[10] * v[2] + m[14] * v[3];
        GeomCalc.vectors[r][3] = m[3] * v[0] + m[7] * v[1] + m[11] * v[2] + m[15] * v[3];
        if (vectors[r][3] != 0.0f) {
            float[] fArray = vectors[r];
            fArray[0] = fArray[0] / vectors[r][3];
            float[] fArray2 = vectors[r];
            fArray2[1] = fArray2[1] / vectors[r][3];
            float[] fArray3 = vectors[r];
            fArray3[2] = fArray3[2] / vectors[r][3];
            GeomCalc.vectors[r][3] = 1.0f;
        }
        GeomCalc.setVector(vIdx, vectors[r]);
        GeomCalc.deleteVector(r);
    }

    public static void multVectorWithMatrix(int vIdx, float[] m) throws GeomCalcException {
        int r = GeomCalc.newVector();
        float[] v = vectors[vIdx];
        GeomCalc.vectors[r][0] = m[0] * v[0] + m[4] * v[1] + m[8] * v[2] + m[12];
        GeomCalc.vectors[r][1] = m[1] * v[0] + m[5] * v[1] + m[9] * v[2] + m[13];
        GeomCalc.vectors[r][2] = m[2] * v[0] + m[6] * v[1] + m[10] * v[2] + m[14];
        GeomCalc.setVector(vIdx, vectors[r]);
        GeomCalc.deleteVector(r);
    }

    public static void rotateX(float[] v, double sinalpha, double cosalpha) {
        double newy = cosalpha * (double)v[1] + sinalpha * (double)v[2];
        double newz = -sinalpha * (double)v[1] + cosalpha * (double)v[2];
        v[1] = (float)newy;
        v[2] = (float)newz;
    }

    public static void rotateY(float[] v, double sinalpha, double cosalpha) {
        double newx = cosalpha * (double)v[0] - sinalpha * (double)v[2];
        double newz = sinalpha * (double)v[0] + cosalpha * (double)v[2];
        v[0] = (float)newx;
        v[2] = (float)newz;
    }

    public static void rotateZ(float[] v, double sinalpha, double cosalpha) {
        double newx = cosalpha * (double)v[0] + sinalpha * (double)v[1];
        double newy = -sinalpha * (double)v[0] + cosalpha * (double)v[1];
        v[0] = (float)newx;
        v[1] = (float)newy;
    }

    public static void rotate(int vi, float s, float c, float t, int axis) {
        int mi = GeomCalc.newMatrix();
        float[] m = matrices[mi];
        float[] v = vectors[axis];
        m[0] = t * v[0] * v[0] + c;
        m[1] = t * v[0] * v[1] + s * v[2];
        m[2] = t * v[0] * v[2] - s * v[1];
        m[4] = t * v[0] * v[1] - s * v[2];
        m[5] = t * v[1] * v[1] + c;
        m[6] = t * v[1] * v[2] + s * v[0];
        m[8] = t * v[0] * v[1] + s * v[1];
        m[9] = t * v[1] * v[2] - s * v[0];
        m[10] = t * v[2] * v[2] + c;
        m[15] = 1.0f;
        GeomCalc.multVectorWithMatrix(vi, mi);
        GeomCalc.deleteMatrix(mi);
    }

    public static int getRotationMatrix(float s, float c, float t, int axis) {
        int mi = GeomCalc.newMatrix();
        float[] m = matrices[mi];
        float[] v = vectors[axis];
        m[0] = t * v[0] * v[0] + c;
        m[1] = t * v[0] * v[1] + s * v[2];
        m[2] = t * v[0] * v[2] - s * v[1];
        m[4] = t * v[0] * v[1] - s * v[2];
        m[5] = t * v[1] * v[1] + c;
        m[6] = t * v[1] * v[2] + s * v[0];
        m[8] = t * v[0] * v[1] + s * v[1];
        m[9] = t * v[1] * v[2] - s * v[0];
        m[10] = t * v[2] * v[2] + c;
        m[15] = 1.0f;
        return mi;
    }

    public static void rotateX(int vIdx, double sinalpha, double cosalpha) {
        double newy = cosalpha * (double)vectors[vIdx][1] + sinalpha * (double)vectors[vIdx][2];
        double newz = -sinalpha * (double)vectors[vIdx][1] + cosalpha * (double)vectors[vIdx][2];
        GeomCalc.vectors[vIdx][1] = (float)newy;
        GeomCalc.vectors[vIdx][2] = (float)newz;
    }

    public static void rotateY(int vIdx, double sinalpha, double cosalpha) {
        double newx = cosalpha * (double)vectors[vIdx][0] - sinalpha * (double)vectors[vIdx][2];
        double newz = sinalpha * (double)vectors[vIdx][0] + cosalpha * (double)vectors[vIdx][2];
        GeomCalc.vectors[vIdx][0] = (float)newx;
        GeomCalc.vectors[vIdx][2] = (float)newz;
    }

    public static void rotateZ(int vIdx, double sinalpha, double cosalpha) {
        double newx = cosalpha * (double)vectors[vIdx][0] + sinalpha * (double)vectors[vIdx][1];
        double newy = -sinalpha * (double)vectors[vIdx][0] + cosalpha * (double)vectors[vIdx][1];
        GeomCalc.vectors[vIdx][0] = (float)newx;
        GeomCalc.vectors[vIdx][1] = (float)newy;
    }

    private static int getFreeVectorIndexCount() {
        int count = 0;
        for (int i = 0; i < allocatedVectorIndices.length; ++i) {
            if (!allocatedVectorIndices[i]) continue;
            ++count;
        }
        return count;
    }

    private static int getUsedVectorIndexCount() {
        return GeomCalc.getFreeVectorIndexCount();
    }

    private static void increaseVectorCapacity() {
        try {
            vectorCapacity = freeVectorIndices.increaseCapacity();
            boolean[] newVectorIndices = new boolean[vectorCapacity];
            System.arraycopy(allocatedVectorIndices, 0, newVectorIndices, 0, allocatedVectorIndices.length);
            allocatedVectorIndices = newVectorIndices;
            float[][] newVectors = new float[vectorCapacity][4];
            System.arraycopy(vectors, 0, newVectors, 0, vectors.length);
            vectors = newVectors;
        }
        catch (Exception c) {
            throw new RuntimeException("Internal Fatal Error: " + c);
        }
    }

    private static void increaseMatrixCapacity() {
        try {
            matrixCapacity = freeMatrixIndices.increaseCapacity();
            boolean[] newMatrixIndices = new boolean[matrixCapacity];
            System.arraycopy(allocatedMatrixIndices, 0, newMatrixIndices, 0, allocatedMatrixIndices.length);
            allocatedMatrixIndices = newMatrixIndices;
            float[][] newMatrices = new float[matrixCapacity][16];
            System.arraycopy(matrices, 0, newMatrices, 0, matrices.length);
            matrices = newMatrices;
        }
        catch (Exception c) {
            throw new RuntimeException("Internal Fatal Error: " + c);
        }
    }

    private static void checkAllocatedVector(int i) throws GeomCalcException {
        if (!allocatedVectorIndices[i]) {
            throw new GeomCalcException("Fatal internal error: vector " + i + " is not allocated.");
        }
    }

    private static void checkAllocatedMatrix(int i) throws GeomCalcException {
        if (!allocatedMatrixIndices[i]) {
            throw new GeomCalcException("Fatal internal error: matrix " + i + " is not allocated.");
        }
    }

    static {
        try {
            int i;
            freeVectorIndices = new Queue(vectorCapacity);
            freeMatrixIndices = new Queue(matrixCapacity);
            for (i = 1; i < vectorCapacity; ++i) {
                freeVectorIndices.push(i);
            }
            for (i = 1; i < matrixCapacity; ++i) {
                freeMatrixIndices.push(i);
            }
            allocatedVectorIndices = new boolean[vectorCapacity];
            allocatedMatrixIndices = new boolean[matrixCapacity];
        }
        catch (Exception c) {
            throw new RuntimeException("Internal Fatal Error: " + c);
        }
    }

    public static class GeomTransformations {
        public static int linePerpendicularToLineInPlane(int a, int b, int c) {
            int n = GeomCalc.sub(b, a);
            GeomCalc.normalize(n);
            int w = GeomCalc.sub(c, a);
            float t = GeomCalc.dot(w, n);
            int nxt = GeomCalc.mult(n, t);
            int d = GeomCalc.add(a, nxt);
            int direction = GeomCalc.sub(c, d);
            GeomCalc.normalize(direction);
            GeomCalc.deleteVector(d);
            GeomCalc.deleteVector(n);
            GeomCalc.deleteVector(w);
            GeomCalc.deleteVector(nxt);
            return direction;
        }

        public static void howtoRotateAtoGetB(int a, int b, double[] angle, int[] directionVector) {
            directionVector[0] = GeomCalc.cross(a, b);
            GeomCalc.normalize(directionVector[0]);
            angle[0] = Math.acos(GeomCalc.dot(a, b)) * 180.0 / Math.PI;
        }
    }

    private static class Queue {
        private int first = 0;
        private int last = 0;
        private boolean empty = true;
        private int[] items = null;

        public Queue(int capacity) {
            this.items = new int[capacity];
        }

        public void push(int item) throws Overflow {
            if (!this.empty && this.last == this.first) {
                throw new Overflow();
            }
            this.empty = false;
            this.items[this.last] = item;
            this.last = (this.last + 1) % this.items.length;
        }

        public int pop() throws Underflow {
            if (this.empty) {
                throw new Underflow();
            }
            int item = this.items[this.first];
            this.first = (this.first + 1) % this.items.length;
            this.empty = this.first == this.last;
            return item;
        }

        public boolean isEmpty() {
            return this.empty;
        }

        public int increaseCapacity() {
            int newLength = this.items.length;
            int[] newitems = new int[newLength += this.items.length >= 65536 ? 32768 : this.items.length];
            this.first = 0;
            this.empty = false;
            this.last = this.first;
            while (this.last < newLength - this.items.length) {
                newitems[this.last] = this.items.length + this.last;
                ++this.last;
            }
            this.items = newitems;
            return newLength;
        }

        public class Underflow
        extends Exception {
            public Underflow() {
                super("Queue underflow");
            }
        }

        public class Overflow
        extends Exception {
            public Overflow() {
                super("Queue overflow");
            }
        }
    }

    public static class GeomCalcException
    extends RuntimeException {
        public GeomCalcException(String msg) {
            super(msg);
        }
    }
}

