/*
 * Decompiled with CFR 0.152.
 */
package chemaxon.enumeration.supergraph;

import chemaxon.calculations.clean.Cleaner;
import chemaxon.common.util.IntVector;
import chemaxon.core.util.BondTable;
import chemaxon.enumeration.AromUtil;
import chemaxon.enumeration.ExpansionCounter;
import chemaxon.enumeration.ExpansionException;
import chemaxon.enumeration.ExpansionHelper;
import chemaxon.enumeration.ExpansionUtil;
import chemaxon.enumeration.bracket.PolymerUtil;
import chemaxon.enumeration.homology.HomologyConstants;
import chemaxon.enumeration.homology.HomologyConversionUtil;
import chemaxon.enumeration.supergraph.ExpansionRecord;
import chemaxon.enumeration.supergraph.MarkushAromata;
import chemaxon.enumeration.supergraph.SupergraphException;
import chemaxon.formats.MolExporter;
import chemaxon.license.Licensable;
import chemaxon.license.LicenseException;
import chemaxon.license.LicenseHandler;
import chemaxon.marvin.util.MolImportUtil;
import chemaxon.sss.search.SearchUtil;
import chemaxon.struc.MolAtom;
import chemaxon.struc.MolBond;
import chemaxon.struc.Molecule;
import chemaxon.struc.MoleculeGraph;
import chemaxon.struc.RgMolecule;
import chemaxon.struc.SelectionMolecule;
import chemaxon.struc.Sgroup;
import chemaxon.struc.sgroup.MulticenterSgroup;
import chemaxon.struc.sgroup.RepeatingUnitSgroup;
import chemaxon.util.BackTrack;
import chemaxon.util.ConfigTools;
import chemaxon.util.IntArray;
import chemaxon.util.IntRange;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Set;
import java.util.Stack;
import java.util.TreeSet;
import java.util.Vector;

public class Supergraph
extends Molecule
implements Licensable {
    private static final long serialVersionUID = -3412792285180718571L;
    private String licenseEnvironment = "";
    private static final String NL = System.getProperty("line.separator");
    private static final String HELP = "Usage:" + NL + "  java chemaxon.enumeration.supergraph.Supergraph" + NL + "       [opts] <mol file/string>" + NL + NL + "opts: data to be written into atom labels, a combination of IPGROWSA" + NL + "      I: atom ID" + NL + "      P: parent ID" + NL + "      G: group number" + NL + "      R: root atom index (1-based)" + NL + "      O: original atom index (1-based)" + NL + "      W: walk (1-based atom index series in graph union)" + NL + "      S: save structure in molecule + property" + NL + "      A: pre-Markush-aromatize" + NL + NL + "Writes graph union also if options U and/or W are specified." + NL;
    public static final String SGROUP = "SGROUP";
    public static final String SG_STRUCTURE = "SG_STRUCTURE";
    private static final String SG_ATOMID = "SG_ATOMID";
    private static final String SG_UNION_ATOMID = "SG_UNION_ATOMID";
    private static final String SG_BOND_ATOMID = "SG_BOND_ATOMID";
    private static final String ORIG_ATOM_INDEX = "ORIG_ATOM_INDEX";
    private static final int AROM_ORIG = 0;
    private static final int AROM_ANY = 1;
    private static final int AROM_SAME = 2;
    private static final int AROM_IMPLIES_AROM = 3;
    private static final int NONAROM_IMPLIES_NONAROM = 4;
    private static final int BONDCODE_BASE = 16;
    private static final int MULTIBONDCODE_SHIFT = 16;
    private static final int[][] BOND_TYPES = ExpansionUtil.BOND_TYPES;
    private HomologyConversionUtil homologyConverter = null;
    private Molecule origMarkush = null;
    private Molecule markush = null;
    private Molecule origMol = null;
    private MoleculeGraph origUnion = null;
    private int origUnionAtomCount = 0;
    private RgMolecule origRgMol = null;
    private Sgroup[] origSgroups = null;
    private Stack<MolAtom>[] atomStacks = null;
    private Stack<MolBond> bondStack = null;
    private int nextID = 0;
    private int[] order = null;
    private IntArray parentIDArray = null;
    private IntArray groupArray = null;
    private IntArray unionIDArray = null;
    private int[] parentIDs = null;
    private int[] groups = null;
    private int[] unionIDs = null;
    private int[] firstChildIDs = null;
    private int[] nextSiblingIDs = null;
    private int[] atomIds = null;
    private boolean[] isAAAtom = null;
    private boolean[] isAABond = null;
    private static final int HAS_AA_UNKNOWN = -1;
    private static final int HAS_AA_TRUE = 0;
    private static final int HAS_AA_FALSE = 1;
    private int hasAAInfo = -1;
    private ExpansionHelper expansionHelper = null;
    private int[] expansionIDs = null;
    private MolAtom[] expansionAtoms = null;
    private Collection<ExpansionRecord> expansionRecords = null;
    public final Comparator<MolAtom> ATOM_EXPANSION_COMPARATOR = new Comparator<MolAtom>(){

        @Override
        public int compare(MolAtom a1, MolAtom a2) {
            int id1 = Supergraph.getID(a1);
            int id2 = Supergraph.getID(a2);
            return Supergraph.this.expansionIDs[id1] - Supergraph.this.expansionIDs[id2];
        }

        @Override
        public boolean equals(Object obj) {
            return this.equals(obj);
        }
    };
    private static final String LIGAND_INDEX = "LIGAND_INDEX";

    public Supergraph() {
        this.setValenceCheckEnabled(false);
    }

    public Supergraph(Molecule mol) throws SupergraphException {
        this.setValenceCheckEnabled(false);
        this.setMolecule(mol);
    }

    @Override
    public void setLicenseEnvironment(String env) {
        this.licenseEnvironment = env;
    }

    @Override
    public boolean isLicensed() {
        return LicenseHandler.getInstance().isLicensed("Markush Search", this.licenseEnvironment);
    }

    private void checkLicense() throws LicenseException {
        LicenseHandler.getInstance().checkLicense("Markush Search", this.licenseEnvironment);
    }

    private void writeSupergraph(ObjectOutputStream oos) throws IOException {
        int i;
        oos.writeByte(1);
        oos.writeObject(this.parentIDs);
        oos.writeObject(this.groups);
        oos.writeObject(this.unionIDs);
        int count = this.getAtomCount();
        oos.writeInt(count);
        for (i = 0; i < count; ++i) {
            MolAtom atom = this.getAtom(i);
            oos.writeInt(Supergraph.getID(atom));
            oos.writeByte(atom.getAtno());
        }
        count = this.getBondCount();
        oos.writeInt(count);
        for (i = 0; i < count; ++i) {
            MolBond bond = this.getBond(i);
            oos.writeInt(this.indexOf(bond.getAtom1()));
            oos.writeInt(this.indexOf(bond.getAtom2()));
            oos.writeByte(bond.getType());
        }
    }

    private void readSupergraph(ObjectInputStream ois) throws IOException, ClassNotFoundException {
        int i;
        byte version = ois.readByte();
        if (version > 1) {
            throw new IOException("Cannot deserialize Supergraph with future version (" + version + ")");
        }
        this.parentIDs = (int[])ois.readObject();
        this.groups = (int[])ois.readObject();
        this.unionIDs = (int[])ois.readObject();
        int count = ois.readInt();
        this.theAtoms = new MolAtom[count];
        for (i = 0; i < count; ++i) {
            int id = ois.readInt();
            int atno = ois.readUnsignedByte();
            MolAtom origAtom = this.origUnion.getAtom(this.unionIDs[id]);
            MolAtom atom = (MolAtom)origAtom.clone();
            Supergraph.setID(atom, id);
            if (atom.getAtno() != atno) {
                atom.setAtno(atno);
            }
            atom.setAttach(0);
            atom.setMaxRepetitions(1);
            atom.setMinRepetitions(1);
            atom.setLinkNodeOuterAtom(0, -1);
            atom.setLinkNodeOuterAtom(1, -1);
            this.add(atom);
        }
        count = ois.readInt();
        this.theBonds = new MolBond[count];
        for (i = 0; i < count; ++i) {
            int index1 = ois.readInt();
            int index2 = ois.readInt();
            byte type = ois.readByte();
            MolBond bond = new MolBond(this.getAtom(index1), this.getAtom(index2));
            bond.setType(type);
            this.add(bond);
        }
    }

    public Molecule saveStructure() throws SupergraphException {
        byte[] ser = null;
        try {
            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            ObjectOutputStream oos = new ObjectOutputStream(new BufferedOutputStream(baos));
            this.writeSupergraph(oos);
            oos.close();
            ser = baos.toByteArray();
        }
        catch (IOException e) {
            throw new SupergraphException("Could not save Supergraph data.", e);
        }
        Molecule mol = this.markush.cloneMoleculeWithDocument();
        mol.setPropertyObject(SG_STRUCTURE, ser);
        return mol;
    }

    public void setMoleculeWithAromatization(Molecule mol) throws SupergraphException {
        MarkushAromata ma = new MarkushAromata();
        ma.aromatize(mol);
        this.setMolecule(mol);
    }

    public void loadStructure(Molecule mol) throws SupergraphException {
        if (mol.getPropertyObject(SG_STRUCTURE) == null) {
            MarkushAromata ma = new MarkushAromata();
            ma.aromatize(mol);
            this.setMolecule(mol);
            return;
        }
        byte[] ser = (byte[])mol.getPropertyObject(SG_STRUCTURE);
        if (ser == null) {
            throw new SupergraphException("SG_STRUCTURE molecule property is not set.\nmolecule name: " + this.markush.getName());
        }
        mol.setPropertyObject(SG_STRUCTURE, null);
        this.setOriginalStructures(mol, true);
        ObjectInputStream ois = null;
        try {
            ois = new ObjectInputStream(new BufferedInputStream(new ByteArrayInputStream(ser)));
            this.readSupergraph(ois);
        }
        catch (IOException e) {
            throw new SupergraphException("Could not load Supergraph data.", e);
        }
        catch (ClassNotFoundException e) {
            throw new SupergraphException("Could not load Supergraph data.", e);
        }
        finally {
            if (ois != null) {
                try {
                    ois.close();
                }
                catch (Exception e) {
                    throw new SupergraphException("Could not load Supergraph data.", e);
                }
            }
        }
        this.setAdditionalData();
    }

    public Molecule getMolecule() {
        return this.markush;
    }

    public void setMolecule(Molecule mol) throws SupergraphException {
        this.setMolecule(mol, true);
    }

    public void setMolecule(Molecule mol, boolean create) throws SupergraphException {
        try {
            ExpansionCounter.checkMarkushStructure(mol);
        }
        catch (ExpansionException e) {
            throw new SupergraphException("" + e.getMessage() + "\nmolecule name: " + mol.getName(), e);
        }
        this.setOriginalStructures(mol, create);
        if (create) {
            this.expand();
        } else {
            this.setDummyData();
        }
        this.setAdditionalData();
        if (this.hasExpandedMarkushFeature(true)) {
            this.checkLicense();
        }
    }

    private void setDummyData() {
        Molecule mol = this.origMol.cloneMolecule();
        if (mol instanceof RgMolecule) {
            mol = ((RgMolecule)mol).getRoot();
        }
        this.fuse(mol, false);
        int count = this.getAtomCount();
        this.parentIDs = new int[count];
        this.groups = new int[count];
        this.unionIDs = new int[count];
        this.nextID = 0;
        for (int i = 0; i < count; ++i) {
            this.setID(this.getAtom(i));
            this.parentIDs[i] = -1;
            this.groups[i] = -1;
            this.unionIDs[i] = i;
        }
    }

    private void setAdditionalData() {
        this.firstChildIDs = new int[this.parentIDs.length];
        this.nextSiblingIDs = new int[this.parentIDs.length];
        this.atomIds = new int[this.parentIDs.length];
        this.hasAAInfo = -1;
        boolean hasAA = false;
        boolean[] tIsAAAtom = null;
        boolean[] tIsAABond = null;
        this.isAAAtom = null;
        this.isAABond = null;
        Arrays.fill(this.firstChildIDs, -1);
        Arrays.fill(this.nextSiblingIDs, -1);
        Arrays.fill(this.atomIds, -1);
        int count = this.getAtomCount();
        for (int i = 0; i < count; ++i) {
            boolean aa;
            int id;
            MolAtom atom = this.getAtom(i);
            this.atomIds[i] = id = Supergraph.getID(atom);
            this.firstChildIDs[id] = -i - 1;
            int r = id;
            while ((r = this.parentIDs[r]) != -1) {
                if (this.firstChildIDs[r] != -1) {
                    int c = this.firstChildIDs[r];
                    while (this.nextSiblingIDs[c] != -1) {
                        c = this.nextSiblingIDs[c];
                    }
                    this.nextSiblingIDs[c] = id;
                    break;
                }
                this.firstChildIDs[r] = id;
                id = r;
            }
            if (!(aa = this.isAmbiguousAromaticAtom(i))) continue;
            if (!hasAA) {
                tIsAAAtom = new boolean[count];
                tIsAABond = new boolean[this.getBondCount()];
            }
            hasAA = true;
            tIsAAAtom[i] = true;
            int bc = atom.getBondCount();
            for (int j = 0; j < bc; ++j) {
                MolBond b = atom.getBond(j);
                MolAtom na = b.getOtherAtom(atom);
                int naIndex = this.indexOf(na);
                if (naIndex >= i || !tIsAAAtom[naIndex]) continue;
                int bIdx = this.indexOf(b);
                tIsAABond[bIdx] = this.isAmbiguousAromaticBond0(i, naIndex);
            }
        }
        if (hasAA) {
            this.hasAAInfo = 0;
            this.isAAAtom = tIsAAAtom;
            this.isAABond = tIsAABond;
        } else {
            this.hasAAInfo = 1;
        }
    }

    public int getAtomIndex(int id) {
        return this.firstChildIDs[id] < 0 ? -this.firstChildIDs[id] - 1 : -1;
    }

    public int[] getChildrenIDs(int id) {
        int childCount = 0;
        int c = this.firstChildIDs[id];
        while (c >= 0) {
            ++childCount;
            c = this.nextSiblingIDs[c];
        }
        int[] res = new int[childCount];
        int childIndex = 0;
        int c2 = this.firstChildIDs[id];
        while (c2 >= 0) {
            res[childIndex++] = c2;
            c2 = this.nextSiblingIDs[c2];
        }
        return res;
    }

    public int[] getOtherGroupSiblings(int idx) {
        int id = this.getID(idx);
        IntVector v = new IntVector();
        int parentID = this.getParentIDByID(id);
        while (parentID >= 0) {
            int groupID = this.getGroupByID(id);
            int c = this.firstChildIDs[parentID];
            while (c >= 0) {
                if (this.getGroupByID(c) != groupID) {
                    int i = this.getAtomIndex(c);
                    if (i > 0) {
                        v.addElement(i);
                    } else {
                        IntVector descV = new IntVector();
                        descV.add(c);
                        while (descV.size() > 0) {
                            int desc = descV.remove(0);
                            int d = this.firstChildIDs[desc];
                            while (d >= 0) {
                                i = this.getAtomIndex(d);
                                if (i > 0) {
                                    v.addElement(i);
                                } else {
                                    descV.add(d);
                                }
                                d = this.nextSiblingIDs[d];
                            }
                        }
                    }
                }
                c = this.nextSiblingIDs[c];
            }
            id = parentID;
            parentID = this.getParentIDByID(parentID);
        }
        return v.toArray();
    }

    public int getIDCount() {
        return this.parentIDs.length;
    }

    public Molecule expand(int[] hit) throws SupergraphException {
        return this.expand(null, hit);
    }

    public Molecule expand(Collection<ExpansionRecord> expansionRecords, int[] hit) throws SupergraphException {
        return this.expand(expansionRecords, hit, null, false, true);
    }

    @Deprecated
    public Molecule expand(int[] hit, Molecule query, boolean expandHomology) throws SupergraphException {
        return this.expand(null, hit, query, expandHomology, true);
    }

    public Molecule expand(Collection<ExpansionRecord> expansionRecords, int[] hit, Molecule query, boolean expandHomology, boolean deconvertHomologies) throws SupergraphException {
        int i;
        if (this.expansionIDs == null) {
            this.expansionIDs = new int[this.parentIDs.length];
            this.expansionAtoms = new MolAtom[this.parentIDs.length];
        }
        Arrays.fill(this.expansionIDs, -1);
        this.nextID = 0;
        if (this.origRgMol != null) {
            AromUtil.initAromatize(this.origRgMol);
        }
        Molecule mol = this.origMol.cloneMolecule();
        boolean v = this.origMol.isValenceCheckEnabled();
        mol.setValenceCheckEnabled(false);
        if (mol.getDim() == 0) {
            Cleaner.clean(mol, 2, null);
        }
        this.storeOrigMol(mol);
        for (i = 0; i < hit.length; ++i) {
            if (hit[i] < 0) continue;
            int id = this.getID(hit[i]);
            this.setExpansionIDs(mol, id);
            if (this.groups[id] >= -1) continue;
            int mid = Supergraph.getBondAtomID(this.groups[id]);
            this.setExpansionIDs(mol, mid);
        }
        this.regulateHomologyExpansion(mol, hit, expandHomology);
        this.expansionRecords = expansionRecords;
        this.expandAtoms(mol, 0);
        this.removeHelperAtoms(mol);
        this.expandBonds(mol);
        if (deconvertHomologies) {
            HomologyConversionUtil.deconvertHomologies(mol, false);
        }
        if (mol instanceof RgMolecule) {
            RgMolecule rgmol = (RgMolecule)mol;
            AromUtil.finishAromatize(rgmol);
            mol = ExpansionUtil.fixRgroupDefs(rgmol);
        }
        Supergraph.clearMarkers(mol);
        for (i = 0; i < hit.length; ++i) {
            if (hit[i] < 0) continue;
            hit[i] = mol.indexOf(this.expansionAtoms[hit[i]]);
        }
        Arrays.fill(this.expansionAtoms, null);
        this.expansionRecords = null;
        if (query != null) {
            Supergraph.copyQuery(mol, hit, query);
        }
        mol.setValenceCheckEnabled(v);
        mol.valenceCheck();
        return mol;
    }

    private void regulateHomologyExpansion(Molecule mol, int[] hit, boolean expandHomology) {
        MolAtom atom;
        int i;
        if (expandHomology) {
            for (int i2 = 0; i2 < hit.length; ++i2) {
                int correctedID;
                if (hit[i2] < 0) continue;
                hit[i2] = correctedID = this.getID(hit[i2]);
            }
            return;
        }
        IntVector homologyAtoms = new IntVector();
        int n = this.markush.getGraphUnion().getAtomCount();
        for (i = 0; i < n; ++i) {
            atom = this.markush.getAtom(i);
            if (!HomologyConstants.isHomologyRAtom(atom)) continue;
            homologyAtoms.add(i);
        }
        n = mol.getGraphUnion().getAtomCount();
        for (i = 0; i < n; ++i) {
            atom = mol.getAtom(i);
            if (atom.getAtno() != 134 || !HomologyConstants.isHomologyRAtom(atom)) continue;
            HomologyConversionUtil.deconvertAtom(atom);
        }
        if (homologyAtoms.size() > 0) {
            for (i = 0; i < this.groups.length; ++i) {
                int origInd = this.getHomologyConvertedIndexByID0(i);
                if (!homologyAtoms.contains(origInd)) continue;
                this.expansionIDs[i] = -1;
            }
        }
        for (i = 0; i < hit.length; ++i) {
            if (hit[i] < 0) continue;
            int id = this.getID(hit[i]);
            int userDefParent = this.getUserDefParentID(hit[i]);
            hit[i] = userDefParent == -1 ? id : userDefParent;
        }
    }

    public int getUserDefParentID(int sgIndex) {
        int id = this.getID(sgIndex);
        if (this.getOrigAtomIndexByID(id) == this.getHomologyConvertedIndexByID0(id)) {
            return -1;
        }
        while (id != -1 && this.getOrigAtomIndexByID(id) != this.getHomologyConvertedIndexByID0(id)) {
            id = this.getParentIDByID(id);
        }
        return id;
    }

    private static void copyQuery(Molecule mol, int[] hit, Molecule query) {
        int i;
        for (i = 0; i < hit.length; ++i) {
            if (hit[i] < 0) continue;
            mol.getAtom(hit[i]).set(query.getAtom(i));
        }
        for (i = query.getBondCount() - 1; i >= 0; --i) {
            MolBond qbond = query.getBond(i);
            int qa1 = query.indexOf(qbond.getAtom1());
            int qa2 = query.indexOf(qbond.getAtom2());
            int ta1 = hit[qa1];
            int ta2 = hit[qa2];
            if (ta1 < 0 || ta2 < 0 || ta1 == ta2) continue;
            MolBond tbond = mol.getAtom(ta1).getBondTo(mol.getAtom(ta2));
            tbond.setType(qbond.getType());
        }
    }

    private void expandAtoms(Molecule mol, int id) throws SupergraphException {
        int group = this.groups[id];
        int pid = this.parentIDs[id];
        for (int i = id; i < this.groups.length && this.groups[i] == group && this.parentIDs[i] == pid; ++i) {
            if (this.expansionIDs[i] == -1 || !this.expandAtom(mol, this.expansionAtoms[i], this.expansionIDs[i])) continue;
            this.expansionAtoms[i] = null;
            this.expandAtoms(mol, this.expansionIDs[i]);
        }
    }

    private boolean expandAtom(Molecule mol, MolAtom atom, int id) throws SupergraphException {
        this.nextID = id;
        int group = this.groups[id];
        if (Supergraph.isRepeatingUnitMarker(atom)) {
            this.expandRepeatingUnit(mol, atom, group);
        } else if (atom.isLinkNode()) {
            this.expandLinkNode(mol, atom, group);
        } else if (atom.getAtno() == 134) {
            this.expandRgroup(mol, atom, group);
        } else if (atom.getAtno() == 128) {
            int uid = Supergraph.getUnionID(atom);
            atom.setList(null);
            atom.setAtno(group);
            this.setID(atom);
            Supergraph.setUnionID(atom, uid);
            this.expansionAtoms[id] = atom;
            if (this.expansionRecords != null) {
                this.expansionRecords.add(this.createAtomListExpansionRecord(atom, group));
            }
        }
        return this.nextID > id;
    }

    private void expandBonds(Molecule mol) throws SupergraphException {
        TreeSet<MolAtom> set = new TreeSet<MolAtom>(this.ATOM_EXPANSION_COMPARATOR);
        for (int i = mol.getAtomCount() - 1; i >= 0; --i) {
            MolAtom atom = mol.getAtom(i);
            if (this.expansionIDs[Supergraph.getID(atom)] == -1) continue;
            set.add(atom);
        }
        while (!set.isEmpty()) {
            MolAtom atom1 = set.first();
            int id = Supergraph.getID(atom1);
            int eid = this.expansionIDs[id];
            int group = this.groups[eid];
            int oid = Supergraph.getBondAtomID(group);
            MolAtom atom2 = this.expansionAtoms[oid];
            if (atom2 != null) {
                MolBond bond = atom1.getBondTo(atom2);
                int bondType = Supergraph.getBondType(group);
                MulticenterSgroup msg = null;
                if (bond == null) {
                    msg = Supergraph.getRepresentedMultiBond(mol, atom2, atom1);
                    if (this.expansionRecords != null) {
                        this.expansionRecords.add(this.createMultiBondExpansionRecord(msg, atom2, atom1));
                    }
                    mol.ungroupSgroup(msg);
                    bond = new MolBond(atom1, atom2);
                    bond.setType(bondType);
                    mol.add(bond);
                } else {
                    bond.setType(bondType);
                    if (this.expansionRecords != null) {
                        this.expansionRecords.add(this.createBondListExpansionRecord(atom1, atom2, bondType));
                    }
                }
            }
            this.expansionIDs[id] = -1;
            this.expansionAtoms[id] = null;
            Supergraph.setID(atom1, eid);
            this.expansionAtoms[eid] = atom1;
            if (this.expansionIDs[eid] != -1) continue;
            set.remove(atom1);
        }
    }

    private static MulticenterSgroup getRepresentedMultiBond(Molecule mol, MolAtom sgroupAtom, MolAtom outerAtom) throws SupergraphException {
        MulticenterSgroup[] sgroups = ExpansionUtil.getMulticenterSgroups(mol, sgroupAtom);
        for (int i = 0; i < sgroups.length; ++i) {
            MolAtom ca = sgroups[i].getCentralAtom();
            if (!ca.isBoundTo(outerAtom)) continue;
            return sgroups[i];
        }
        throw new SupergraphException("Represented multibond is not found.\nmolecule name: " + mol.getName());
    }

    private void setExpansionIDs(Molecule mol, int id) throws SupergraphException {
        int group = this.groups[id];
        int pid = this.parentIDs[id];
        int i = id;
        while (pid != -1 && this.expansionIDs[pid] == -1) {
            while (this.parentIDs[--i] == pid && this.groups[i] == group) {
            }
            this.expansionIDs[pid] = i + 1;
            i = pid;
            group = this.groups[i];
            pid = this.parentIDs[i];
        }
        if (pid != -1 && this.groups[this.expansionIDs[pid]] != group) {
            throw new SupergraphException("Incompatible atom set: pid=" + pid + " expansionIDs[pid]=" + this.expansionIDs[pid] + " groups[expansionIDs[pid]]=" + this.groups[this.expansionIDs[pid]] + " group=" + group + "\nmolecule name: " + mol.getName());
        }
    }

    private void setOriginalStructures(Molecule mol, boolean create) {
        int i;
        this.origMarkush = mol.cloneMolecule();
        for (i = mol.getGraphUnion().getAtomCount() - 1; i >= 0; --i) {
            mol.getAtom(i).putProperty(ORIG_ATOM_INDEX, i);
        }
        mol.setGUIContracted(true);
        mol.ungroupSgroups(0);
        this.homologyConverter = new HomologyConversionUtil();
        if (create && !HomologyConstants.hasHomologyRAtom(mol)) {
            this.homologyConverter.setMol(mol);
            this.markush = this.homologyConverter.getConvertedMol();
        } else {
            this.homologyConverter.setConvertedMol(mol);
            if (!(mol instanceof RgMolecule)) {
                this.markush = new RgMolecule();
                ((RgMolecule)this.markush).setRoot(mol.cloneMolecule());
            } else {
                this.markush = mol;
            }
        }
        this.markush.expandSgroups();
        this.origMol = this.markush.cloneMolecule();
        this.origUnion = this.origMol.getGraphUnion();
        this.origUnionAtomCount = this.origUnion.getAtomCount();
        this.origRgMol = this.origMol instanceof RgMolecule ? (RgMolecule)this.origMol : null;
        for (i = this.origUnionAtomCount - 1; i >= 0; --i) {
            Supergraph.setUnionID(this.origUnion.getAtom(i), i);
        }
        this.origSgroups = this.origMol.getSgroupArray();
        this.expansionHelper = new ExpansionHelper(this.origMol);
    }

    public static void clearMarkers(Molecule mol) {
        MoleculeGraph union = mol.getGraphUnion();
        for (int i = union.getAtomCount() - 1; i >= 0; --i) {
            MolAtom atom = union.getAtom(i);
            atom.removeProperty(ORIG_ATOM_INDEX);
            atom.removeProperty(LIGAND_INDEX);
            atom.removeProperty(SG_ATOMID);
            atom.removeProperty(SG_BOND_ATOMID);
            atom.removeProperty(SG_UNION_ATOMID);
        }
        AromUtil.clearAmbigAromRingMarkers(mol);
    }

    private void expand() throws SupergraphException {
        try {
            this.findExpansionOrder();
        }
        catch (ExpansionException e) {
            throw new SupergraphException("" + e.getMessage() + "\nmolecule name: " + this.markush.getName(), e);
        }
        this.parentIDArray = new IntArray();
        this.groupArray = new IntArray();
        this.unionIDArray = new IntArray();
        this.parentIDs = null;
        this.groups = null;
        this.unionIDs = null;
        this.setValenceCheckEnabled(false);
        Molecule mol = this.origMol.cloneMolecule();
        if (mol instanceof RgMolecule) {
            mol = ((RgMolecule)mol).getRoot();
        }
        this.fuse(mol, false);
        this.expandAtoms();
        this.expandMultiBonds();
        this.expandBonds();
        this.removeIsolatedAtoms();
        this.removeIsolatedBonds();
        this.parentIDs = this.parentIDArray.toArray();
        this.groups = this.groupArray.toArray();
        this.unionIDs = this.unionIDArray.toArray();
        this.parentIDArray = null;
        this.groupArray = null;
        this.unionIDArray = null;
    }

    private void findExpansionOrder() throws ExpansionException {
        int[] sequence = this.expansionHelper.findExpansionSequence();
        this.order = new int[this.origUnionAtomCount + this.origMol.getSgroupCount()];
        Arrays.fill(this.order, -1);
        for (int i = 0; i < sequence.length; ++i) {
            this.order[sequence[i]] = i;
        }
    }

    private void expandAtoms() throws SupergraphException {
        this.initAtoms();
        ArrayList<MolAtom> helperAtomList = new ArrayList<MolAtom>();
        this.addHelperAtoms(helperAtomList);
        if (this.origRgMol != null) {
            AromUtil.initAromatize(this.origRgMol);
        }
        for (int i = 0; i < this.atomStacks.length; ++i) {
            while (!this.atomStacks[i].empty()) {
                MolAtom atom = this.atomStacks[i].pop();
                if (this.indexOf(atom) == -1) continue;
                if (Supergraph.isRepeatingUnitMarker(atom)) {
                    this.expandRepeatingUnit(atom, helperAtomList);
                    continue;
                }
                if (atom.isLinkNode()) {
                    this.expandLinkNode(atom, helperAtomList);
                    continue;
                }
                if (atom.getAtno() == 134) {
                    this.expandRgroup(atom, helperAtomList);
                    continue;
                }
                if (atom.getAtno() != 128) continue;
                this.expandAtomList(atom);
            }
        }
        this.atomStacks = null;
        this.removeHelperAtoms(helperAtomList);
    }

    private void initAtoms() {
        int i;
        int m = 0;
        for (i = 0; i < this.order.length; ++i) {
            m = Math.max(this.order[i], m);
        }
        this.atomStacks = new Stack[++m];
        for (i = 0; i < m; ++i) {
            this.atomStacks[i] = new Stack();
        }
        this.nextID = 0;
        this.store();
    }

    private void storeOrigMol(MoleculeGraph mol) {
        this.store(mol, true);
    }

    private void store(MoleculeGraph mol) {
        this.store(mol, false);
    }

    private void store(MoleculeGraph mol, boolean includeRgroupAttachments) {
        int count = mol.getAtomCount();
        for (int i = 0; i < count; ++i) {
            MolAtom atom = mol.getAtom(i);
            if (Supergraph.isHelperAtom(atom) || !includeRgroupAttachments && atom.getAtno() == 138) continue;
            int id = this.setID(atom);
            this.expansionAtoms[id] = atom;
        }
        if (mol instanceof Molecule) {
            this.storeSgroups((Molecule)mol);
        }
    }

    private void storeSgroups(Molecule mol) {
        ArrayList<Sgroup> ungroupList = new ArrayList<Sgroup>();
        int count = mol.getSgroupCount();
        for (int i = 0; i < count; ++i) {
            Sgroup sgroup = mol.getSgroup(i);
            int x = sgroup.findCrossingBonds().length;
            if (!(sgroup instanceof RepeatingUnitSgroup) || PolymerUtil.isPolymerSgroup(sgroup) || x > 4) continue;
            if (x > 1) {
                MolAtom sgroupMarker = this.createSgroupMarker(sgroup);
                int id = this.setID(sgroupMarker);
                this.expansionAtoms[id] = sgroupMarker;
                continue;
            }
            ungroupList.add(sgroup);
        }
        for (Sgroup sgroup : ungroupList) {
            mol.ungroupSgroup(sgroup);
        }
    }

    private void store() {
        int count = this.getAtomCount();
        for (int i = 0; i < count; ++i) {
            MolAtom atom = this.getAtom(i);
            this.store(atom, -1, -1);
        }
        this.storeSgroups(this, -1, -1);
    }

    private void store(MoleculeGraph fragment, int pid, int rep, ArrayList<MolAtom> helperAtomList) {
        int count = fragment.getAtomCount();
        for (int i = 0; i < count; ++i) {
            MolAtom atom = fragment.getAtom(i);
            if (Supergraph.isHelperAtom(atom)) {
                helperAtomList.add(atom);
                continue;
            }
            if (atom.getAtno() == 138) continue;
            this.store(atom, pid, rep);
        }
        if (fragment instanceof Molecule) {
            this.storeSgroups((Molecule)fragment, pid, rep);
        }
    }

    private void store(MolAtom atom, int pid, int group) {
        int id = this.setID(atom);
        this.parentIDArray.setWithExpansion(id, pid);
        this.groupArray.setWithExpansion(id, group);
        int uid = Supergraph.getUnionID(atom);
        this.unionIDArray.setWithExpansion(id, uid);
        if (this.atomStacks != null && this.isExpandable(atom)) {
            this.atomStacks[this.order[uid]].push(atom);
        }
    }

    private void storeSgroups(Molecule fragment, int pid, int rep) {
        ArrayList<Sgroup> ungroupList = new ArrayList<Sgroup>();
        int count = fragment.getSgroupCount();
        for (int i = 0; i < count; ++i) {
            Sgroup sgroup = fragment.getSgroup(i);
            int x = sgroup.findCrossingBonds().length;
            if (!(sgroup instanceof RepeatingUnitSgroup) || PolymerUtil.isPolymerSgroup(sgroup) || x > 4) continue;
            if (x > 1) {
                MolAtom sgroupMarker = this.createSgroupMarker(sgroup);
                this.store(sgroupMarker, pid, rep);
                this.add(sgroupMarker);
                continue;
            }
            ungroupList.add(sgroup);
        }
        for (Sgroup sgroup : ungroupList) {
            fragment.ungroupSgroup(sgroup);
        }
    }

    private MolAtom createSgroupMarker(Sgroup sgroup) {
        MolAtom marker = new MolAtom(131);
        marker.putProperty(SGROUP, sgroup);
        int sgindex = this.getOrigSgroupIndex(sgroup);
        assert (sgindex != -1);
        int uid = this.origUnionAtomCount + sgindex;
        Supergraph.setUnionID(marker, uid);
        return marker;
    }

    private int getOrigSgroupIndex(Sgroup sgroup) {
        int atomCount = sgroup.getAtomCount();
        int[] uids = Supergraph.getUnionIDs(sgroup);
        for (int i = 0; i < this.origSgroups.length; ++i) {
            Sgroup origSgroup = this.origSgroups[i];
            if (origSgroup.getAtomCount() != atomCount || !origSgroup.getClass().equals(sgroup.getClass()) || !Arrays.equals(Supergraph.getUnionIDs(origSgroup), uids)) continue;
            return i;
        }
        return -1;
    }

    private static int[] getUnionIDs(Sgroup sgroup) {
        int[] uids = new int[sgroup.getAtomCount()];
        for (int i = 0; i < uids.length; ++i) {
            uids[i] = Supergraph.getUnionID(sgroup.getAtom(i));
        }
        Arrays.sort(uids);
        return uids;
    }

    private static boolean isRepeatingUnitMarker(MolAtom atom) {
        return atom.getProperty(SGROUP) instanceof RepeatingUnitSgroup;
    }

    private void extractFrag(MolAtom atom, Molecule fragment) {
        int i;
        HashSet<MolAtom> nodeSet = new HashSet<MolAtom>();
        this.extractFragNodes(atom, nodeSet);
        MolAtom[] fragNodes = new MolAtom[nodeSet.size()];
        nodeSet.toArray(fragNodes);
        Arrays.sort(fragNodes, new Comparator<MolAtom>(){

            @Override
            public int compare(MolAtom a1, MolAtom a2) {
                return Supergraph.this.indexOf(a1) - Supergraph.this.indexOf(a2);
            }
        });
        for (i = 0; i < fragNodes.length; ++i) {
            this.theAtoms[this.indexOf((MolAtom)fragNodes[i])] = null;
            fragment.add(fragNodes[i]);
        }
        for (i = fragment.getAtomCount() - 1; i >= 0; --i) {
            MolAtom node = fragment.getAtom(i);
            for (int j = node.getBondCount() - 1; j >= 0; --j) {
                MolBond edge = node.getBond(j);
                if (fragment.indexOf(node) >= fragment.indexOf(edge.getOtherAtom(node))) continue;
                this.theBonds[this.indexOf((MolBond)edge)] = null;
                fragment.add(edge);
            }
        }
    }

    private void extractFragNodes(MolAtom atom, Collection<MolAtom> nodes) {
        nodes.add(atom);
        for (int i = atom.getBondCount() - 1; i >= 0; --i) {
            MolAtom ligand = atom.getLigand(i);
            if (nodes.contains(ligand)) continue;
            this.extractFragNodes(ligand, nodes);
        }
        if (atom.getAtno() == 137) {
            MulticenterSgroup msg = this.findContainingMulticenterSgroup(atom);
            for (int i = msg.getAtomCount() - 1; i >= 0; --i) {
                MolAtom ligand = msg.getAtom(i);
                if (nodes.contains(ligand)) continue;
                this.extractFragNodes(ligand, nodes);
            }
        } else {
            MulticenterSgroup[] msgs = ExpansionUtil.getMulticenterSgroups(this, atom);
            for (int i = 0; i < msgs.length; ++i) {
                MolAtom ligand = msgs[i].getCentralAtom();
                if (nodes.contains(ligand)) continue;
                this.extractFragNodes(ligand, nodes);
            }
        }
    }

    private static void getCorrespondingAtoms(MolAtom[] newAtoms, MoleculeGraph newGraph, MolAtom[] oldAtoms) {
        for (int i = 0; i < oldAtoms.length; ++i) {
            newAtoms[i] = Supergraph.getCorrespondingAtom(newGraph, oldAtoms[i]);
            assert (newAtoms[i] != null);
        }
    }

    private static MolAtom getCorrespondingAtom(MoleculeGraph newGraph, MolAtom atom) {
        int id = Supergraph.getID(atom);
        for (int i = newGraph.getAtomCount() - 1; i >= 0; --i) {
            if (Supergraph.getID(newGraph.getAtom(i)) != id) continue;
            return newGraph.getAtom(i);
        }
        return null;
    }

    private static void createCorrespondingBonds(MolBond[] newBonds1, MolAtom[] newAtoms1, MolBond[] oldBonds1, MolAtom[] oldAtoms1, MolBond[] newBonds2, MolAtom[] newAtoms2, MolBond[] oldBonds2, MolAtom[] oldAtoms2) {
        Hashtable<MolBond, MolBond> correspondence = new Hashtable<MolBond, MolBond>();
        Supergraph.createCorrespondingBonds(newBonds1, newAtoms1, oldBonds1, oldAtoms1, correspondence);
        Supergraph.createCorrespondingBonds(newBonds2, newAtoms2, oldBonds2, oldAtoms2, correspondence);
    }

    private static void createCorrespondingBonds(MolBond[] newBonds, MolAtom[] newAtoms, MolBond[] oldBonds, MolAtom[] oldAtoms, Hashtable<MolBond, MolBond> correspondence) {
        for (int i = 0; i < oldBonds.length; ++i) {
            newBonds[i] = correspondence.get(oldBonds[i]);
            if (newBonds[i] != null) continue;
            newBonds[i] = oldBonds[i].cloneBond(newAtoms[i], oldBonds[i].getOtherAtom(oldAtoms[i]));
            correspondence.put(oldBonds[i], newBonds[i]);
        }
    }

    private void expandRepeatingUnit(MolAtom atom, ArrayList<MolAtom> helperAtomList) throws SupergraphException {
        RepeatingUnitSgroup repeatingUnit = (RepeatingUnitSgroup)atom.getProperty(SGROUP);
        IntRange range = new IntRange(repeatingUnit.getSubscript());
        range.setMaxCount(10);
        int pid = Supergraph.getID(atom);
        this.isolate(atom);
        MolBond[][] xbonds = ExpansionUtil.getXBonds(repeatingUnit);
        MolBond[] xbonds1 = xbonds[0];
        MolBond[] xbonds2 = xbonds[1];
        boolean moiety = xbonds1 == null;
        SelectionMolecule repeatingGraph = repeatingUnit.getSgroupGraph();
        MolAtom[] xatoms1 = !moiety ? repeatingUnit.getCrossingAtoms(xbonds1) : null;
        MolAtom[] xatoms2 = !moiety ? repeatingUnit.getCrossingAtoms(xbonds2) : null;
        this.ungroupSgroup(repeatingUnit, 0);
        MolAtom[] atoms1 = !moiety ? new MolAtom[xatoms1.length] : null;
        MolAtom[] atoms2 = !moiety ? new MolAtom[xatoms2.length] : null;
        MolBond[] bonds1 = !moiety ? new MolBond[xbonds1.length] : null;
        MolBond[] bonds2 = !moiety ? new MolBond[xbonds2.length] : null;
        int[] indexes = new int[repeatingGraph.getAtomCount()];
        while (range.hasNext()) {
            int repetitions = range.next();
            SelectionMolecule part = new SelectionMolecule();
            if (range.hasNext()) {
                ExpansionUtil.getAtomIndexes(indexes, (MoleculeGraph)repeatingGraph, this);
                Molecule newGraph = new Molecule();
                this.clonecopy(indexes, newGraph);
                if (!moiety) {
                    Supergraph.getCorrespondingAtoms(atoms1, newGraph, xatoms1);
                    Supergraph.getCorrespondingAtoms(atoms2, newGraph, xatoms2);
                    Supergraph.createCorrespondingBonds(bonds1, atoms1, xbonds1, xatoms1, bonds2, atoms2, xbonds2, xatoms2);
                }
                this.fuse(newGraph, false);
                if (!moiety) {
                    int i;
                    for (i = 0; i < bonds1.length; ++i) {
                        if (bonds1[i].getParent() == this) continue;
                        this.add(bonds1[i]);
                    }
                    for (i = 0; i < bonds2.length; ++i) {
                        if (bonds2[i].getParent() == this) continue;
                        this.add(bonds2[i]);
                    }
                }
                ExpansionUtil.insertRepeatingPart(this, newGraph, repetitions, part, bonds1, atoms1, bonds2, atoms2);
            } else {
                ExpansionUtil.insertRepeatingPart(this, repeatingGraph, repetitions, part, xbonds1, xatoms1, xbonds2, xatoms2);
            }
            this.store(part, pid, repetitions, helperAtomList);
        }
    }

    private void expandRepeatingUnit(Molecule mol, MolAtom atom, int repetitions) throws SupergraphException {
        RepeatingUnitSgroup repeatingUnit = (RepeatingUnitSgroup)atom.getProperty(SGROUP);
        SelectionMolecule part = new SelectionMolecule();
        try {
            ExpansionUtil.insertRepeatingPart(mol, repeatingUnit, repetitions, part);
        }
        catch (IllegalArgumentException e) {
            throw new SupergraphException("" + e.getMessage() + "\nmolecule name: " + this.markush.getName(), e);
        }
        this.store(part);
        if (this.expansionRecords != null) {
            this.expansionRecords.add(this.createRepeatingUnitExpansionRecord(repeatingUnit, repetitions));
        }
    }

    private void expandLinkNode(MolAtom atom, ArrayList<MolAtom> helperAtomList) throws SupergraphException {
        int minrep = atom.getMinRepetitions();
        int maxrep = atom.getMaxRepetitions();
        int pid = Supergraph.getID(atom);
        MolAtom[] innerLigands = atom.getAtno() == 134 && this.origRgMol != null ? MolImportUtil.getLigandsInOrder(atom) : null;
        MolBond[] linkBonds = ExpansionUtil.getLinkBonds(this, atom);
        MolBond outer0Bond = linkBonds[0];
        MolBond outer1Bond = linkBonds[1];
        MolBond linkerBond = linkBonds[2];
        MolAtom outer0Atom = outer0Bond.getOtherAtom(atom);
        MolAtom outer1Atom = outer1Bond.getOtherAtom(atom);
        int outer0Index = -1;
        int outer1Index = -1;
        if (innerLigands != null) {
            for (int j = 0; j < innerLigands.length; ++j) {
                if (innerLigands[j] == outer0Atom) {
                    outer0Index = j;
                    continue;
                }
                if (innerLigands[j] == outer1Atom) {
                    outer1Index = j;
                    continue;
                }
                innerLigands[j].putProperty(LIGAND_INDEX, j);
            }
        }
        atom.setMaxRepetitions(1);
        this.isolate(outer0Bond);
        this.isolate(outer1Bond);
        MulticenterSgroup[] sgroups = ExpansionUtil.getMulticenterSgroups(this, atom);
        for (int i = 0; i < sgroups.length; ++i) {
            sgroups[i].removeAtom(atom);
        }
        Supergraph fragment = new Supergraph();
        this.extractFrag(atom, fragment);
        for (int i = this.getSgroupCount() - 1; i >= 0; --i) {
            MulticenterSgroup msg;
            MolAtom ca;
            Sgroup sg = this.getSgroup(i);
            if (sg.getType() != 14 || fragment.indexOf(ca = (msg = (MulticenterSgroup)sg).getCentralAtom()) == -1) continue;
            fragment.setSgroupParent(ca, msg, true);
            for (int j = msg.getAtomCount() - 1; j >= 0; --j) {
                fragment.setSgroupParent(msg.getAtom(j), msg, true);
            }
        }
        int innerAtomIndex = fragment.indexOf(atom);
        MolAtom[] linkAtoms = null;
        boolean[] isOuterGroup = new boolean[sgroups.length];
        for (int i = 0; i < sgroups.length; ++i) {
            boolean bl = isOuterGroup[i] = fragment.indexOf(sgroups[i].getCentralAtom()) == -1;
            if (isOuterGroup[i]) {
                if (linkAtoms != null) continue;
                linkAtoms = new MolAtom[maxrep];
                continue;
            }
            sgroups[i].add(atom);
        }
        for (int rep = minrep; rep <= maxrep; ++rep) {
            MolBond partOuter0Bond;
            Molecule part = ((Molecule)fragment).cloneMolecule();
            MolAtom link0Atom = part.getAtom(innerAtomIndex);
            MolBond molBond = partOuter0Bond = outer0Bond != null ? outer0Bond.cloneBond(outer0Atom, link0Atom) : null;
            if (linkAtoms != null) {
                linkAtoms[0] = link0Atom;
            }
            for (int i = 1; i < rep; ++i) {
                Molecule currfrag = ((Molecule)fragment).cloneMolecule();
                MolAtom link1Atom = currfrag.getAtom(innerAtomIndex);
                MolBond linkBond = linkerBond.cloneBond(link0Atom, link1Atom);
                linkBond.setSetSeq(0);
                part.fuse(currfrag, false);
                part.add(linkBond);
                if (linkAtoms != null) {
                    linkAtoms[i] = link1Atom;
                }
                link0Atom = link1Atom;
            }
            this.store(part, pid, rep, helperAtomList);
            MolBond partOuter1Bond = outer1Bond.cloneBond(outer1Atom, link0Atom);
            this.fuse(part, false);
            this.add(partOuter0Bond);
            this.add(partOuter1Bond);
            if (linkAtoms != null) {
                for (int j = 0; j < sgroups.length; ++j) {
                    if (!isOuterGroup[j]) continue;
                    for (int k = 0; k < rep; ++k) {
                        this.setSgroupParent(linkAtoms[k], sgroups[j], true);
                    }
                }
            }
            if (innerLigands == null) continue;
            MolAtom patom = outer0Atom;
            MolAtom natom = null;
            MolBond nbond = null;
            MolAtom helper = null;
            MolAtom ratom = partOuter0Bond.getOtherAtom(outer0Atom);
            while (ratom != outer1Atom) {
                natom = Supergraph.getOtherLink(ratom, patom);
                nbond = ratom.getBondTo(natom);
                innerLigands[outer0Index] = patom;
                if (natom != outer1Atom) {
                    helper = Supergraph.createHelperAtom();
                    this.add(helper);
                    helperAtomList.add(helper);
                    Supergraph.setID(helper, -1 * this.indexOf(helper));
                    this.splitBond(this, nbond, helper);
                    innerLigands[outer1Index] = helper;
                } else {
                    innerLigands[outer1Index] = outer1Atom;
                }
                for (int j = ratom.getBondCount() - 1; j >= 0; --j) {
                    MolAtom ligand = ratom.getLigand(j);
                    if (ligand == innerLigands[outer0Index] || ligand == innerLigands[outer1Index]) continue;
                    int ligandIndex = (Integer)ligand.getProperty(LIGAND_INDEX);
                    innerLigands[ligandIndex] = ligand;
                    ligand.removeProperty(LIGAND_INDEX);
                }
                MolImportUtil.setLigandOrders(ratom, innerLigands);
                patom = helper;
                ratom = natom;
            }
        }
    }

    private static MolAtom getOtherLink(MolAtom atom, MolAtom link) {
        for (int i = atom.getBondCount() - 1; i >= 0; --i) {
            MolAtom ligand = atom.getLigand(i);
            if (ligand == link || ligand.getProperty(LIGAND_INDEX) != null) continue;
            return ligand;
        }
        return null;
    }

    private void expandLinkNode(Molecule mol, MolAtom atom, int repetitions) throws SupergraphException {
        SelectionMolecule part = new SelectionMolecule();
        try {
            ExpansionUtil.insertLinkPart(mol, atom, repetitions, part);
        }
        catch (IllegalArgumentException e) {
            throw new SupergraphException("" + e.getMessage() + "\nmolecule name: " + this.markush.getName(), e);
        }
        this.store(part);
        if (this.expansionRecords != null) {
            this.expansionRecords.add(this.createLinkNodeExpansionRecord(atom, repetitions));
        }
    }

    private void expandRgroup(MolAtom atom, ArrayList<MolAtom> helperAtomList) throws SupergraphException {
        Molecule rgroup;
        int i;
        int rgindex = this.getRgroupIndex(atom);
        if (rgindex == -1) {
            return;
        }
        int pid = Supergraph.getID(atom);
        if (ExpansionUtil.getMulticenterSgroups(this, atom).length > 0) {
            throw new SupergraphException("R-atom in multicenter group: " + (this.unionIDArray.get(pid) + 1) + "\nmolecule name: " + this.markush.getName());
        }
        MolAtom[] outerAtoms = MolImportUtil.getLigandsInOrder(atom);
        MolBond[] linkBonds = new MolBond[outerAtoms.length];
        for (int i2 = 0; i2 < outerAtoms.length; ++i2) {
            linkBonds[i2] = atom.getBondTo(outerAtoms[i2]);
        }
        MolAtom ratom = this.origUnion.getAtom(Supergraph.getUnionID(atom));
        int memberCount = this.origRgMol.getRgroupMemberCount(rgindex);
        Molecule[] rgroups = new Molecule[memberCount];
        for (i = 0; i < memberCount; ++i) {
            rgroup = this.origRgMol.getRgroupMember(rgindex, i).cloneMolecule();
            rgroup.setValenceCheckEnabled(false);
            AromUtil.aromatize(ratom, rgroup);
            rgroups[i] = rgroup;
        }
        this.isolate(atom);
        for (i = 0; i < memberCount; ++i) {
            rgroup = rgroups[i];
            this.store(rgroup, pid, i, helperAtomList);
            try {
                ExpansionUtil.insertRgroupMember(this, atom, rgroup, linkBonds, null, false, null);
                continue;
            }
            catch (IllegalArgumentException e) {
                throw new SupergraphException("" + e.getMessage() + "\nmolecule name: " + this.markush.getName(), e);
            }
        }
    }

    private void expandRgroup(Molecule mol, MolAtom atom, int memberIndex) throws SupergraphException {
        int rgindex = this.getRgroupIndex(atom);
        if (rgindex == -1) {
            return;
        }
        Molecule rgroup = this.origRgMol.getRgroupMember(rgindex, memberIndex).cloneMolecule();
        AromUtil.aromatize(atom, rgroup);
        this.store(rgroup);
        try {
            ExpansionUtil.insertRgroupMember(mol, atom, rgroup, true);
        }
        catch (IllegalArgumentException e) {
            throw new SupergraphException("" + e.getMessage() + "molecule name: " + this.markush.getName(), e);
        }
        if (this.expansionRecords != null) {
            this.expansionRecords.add(this.createRgroupExpansionRecord(atom, rgindex, memberIndex));
        }
    }

    private int getRgroupIndex(MolAtom atom) {
        return this.origRgMol.findRgroupIndex(atom.getRgroup());
    }

    private void expandAtomList(MolAtom atom) {
        MulticenterSgroup[] sgroups = ExpansionUtil.getMulticenterSgroups(this, atom);
        for (int i = 0; i < sgroups.length; ++i) {
            sgroups[i].removeAtom(atom);
        }
        int pid = Supergraph.getID(atom);
        int uid = Supergraph.getUnionID(atom);
        int[] atnos = atom.getList();
        MolAtom[] a = new MolAtom[atnos.length];
        for (int i = 0; i < atnos.length; ++i) {
            a[i] = (MolAtom)atom.clone();
            a[i].setList(null);
            a[i].setAtno(atnos[i]);
            Supergraph.setUnionID(a[i], uid);
            this.store(a[i], pid, atnos[i]);
            this.add(a[i]);
            for (int j = 0; j < sgroups.length; ++j) {
                this.setSgroupParent(a[i], sgroups[j], true);
            }
        }
        this.copyBonds(atom, a);
        this.isolate(atom);
    }

    private void expandBonds() {
        this.initBonds();
        while (!this.bondStack.empty()) {
            MolBond bond = this.bondStack.pop();
            if (this.indexOf(bond) == -1) continue;
            this.expandQueryBond(bond);
        }
        this.bondStack = null;
    }

    private void initBonds() {
        this.bondStack = new Stack();
        for (int i = this.getBondCount() - 1; i >= 0; --i) {
            MolBond bond = this.getBond(i);
            if (bond == null || !this.isExpandable(bond)) continue;
            this.bondStack.push(bond);
        }
    }

    private void expandQueryBond(MolBond bond) {
        int i;
        MolAtom atom1 = bond.getAtom1();
        MolAtom atom2 = bond.getAtom2();
        int[] types = BOND_TYPES[bond.getType()];
        MolAtom[] a = new MolAtom[types.length];
        this.isolate(bond);
        if (!atom2.hasQueryBonds() && (atom1.hasQueryBonds() || atom1.getBondCount() > atom2.getBondCount())) {
            MolAtom atom = atom1;
            atom1 = atom2;
            atom2 = atom;
        }
        MulticenterSgroup[] sgroups = ExpansionUtil.getMulticenterSgroups(this, atom1);
        for (int i2 = 0; i2 < sgroups.length; ++i2) {
            sgroups[i2].removeAtom(atom1);
        }
        int pid = Supergraph.getID(atom1);
        int bid = Supergraph.getID(atom2);
        for (i = 0; i < a.length; ++i) {
            a[i] = (MolAtom)atom1.clone();
            this.store(a[i], pid, Supergraph.getBondCode(bid, types[i]));
            this.add(a[i]);
            for (int j = 0; j < sgroups.length; ++j) {
                this.setSgroupParent(a[i], sgroups[j], true);
            }
        }
        this.copyBonds(atom1, a);
        this.isolate(atom1);
        for (i = 0; i < a.length; ++i) {
            MolBond bondi = new MolBond(a[i], atom2);
            bondi.setFlags(types[i], 15);
            this.add(bondi);
        }
    }

    private void expandMultiBonds() {
        MulticenterSgroup msg = null;
        while ((msg = this.findMulticenterSgroup()) != null) {
            boolean hasCoordinateBond = false;
            MolAtom ca = msg.getCentralAtom();
            for (int j = ca.getBondCount() - 1; j >= 0; --j) {
                MolBond mb = ca.getBond(j);
                if (mb.getType() != 9) {
                    MolAtom oa = mb.getOtherAtom(ca);
                    if (oa.getAtno() != 137) {
                        this.expandMultiBond(msg, oa, mb);
                        continue;
                    }
                    MulticenterSgroup m = this.findContainingMulticenterSgroup(oa);
                    this.expandMultiBond(msg, m, mb);
                    continue;
                }
                hasCoordinateBond = true;
            }
            if (hasCoordinateBond) continue;
            this.ungroupSgroup(msg);
        }
    }

    private void expandMultiBond(MulticenterSgroup msg, MolAtom oa, MolBond mb) {
        this.expandMultiBond(msg, oa, mb, null);
    }

    private MolAtom[] expandMultiBond(MulticenterSgroup msg, MolAtom oa, MolBond mb, MulticenterSgroup excl) {
        MulticenterSgroup[] sgroups = ExpansionUtil.getMulticenterSgroups(this, oa);
        int pid = Supergraph.getID(oa);
        int type = mb.getType();
        MolAtom[] a = new MolAtom[msg.getAtomCount()];
        for (int i = 0; i < a.length; ++i) {
            a[i] = (MolAtom)oa.clone();
            MolAtom ma = msg.getAtom(i);
            int mid = Supergraph.getID(ma);
            this.store(a[i], pid, Supergraph.getMultiBondCode(mid, type));
            this.add(a[i]);
            this.add(mb.cloneBond(ma, a[i]));
            for (int j = 0; j < sgroups.length; ++j) {
                if (sgroups[j] == excl) continue;
                this.setSgroupParent(a[i], sgroups[j], true);
            }
        }
        this.isolate(mb);
        this.copyBonds(oa, a);
        this.isolate(oa);
        return a;
    }

    private void expandMultiBond(MulticenterSgroup msg, MulticenterSgroup m, MolBond mb) {
        MolAtom ca = m.getCentralAtom();
        MolAtom[] a = this.expandMultiBond(msg, ca, mb, m);
        int mcount = m.getAtomCount();
        for (int i = 0; i < a.length; ++i) {
            MulticenterSgroup mclone = (MulticenterSgroup)m.cloneSgroup(this, null);
            SelectionMolecule mgraph = m.cloneStructure();
            mclone.setSgroupGraph(mgraph);
            mclone.setCentralAtom(a[i]);
            this.fuse(mgraph, false);
            int id = Supergraph.getID(a[i]);
            int pid = this.parentIDArray.get(id);
            int group = this.groupArray.get(id);
            for (int j = 0; j < mcount; ++j) {
                MolAtom ma = mclone.getAtom(j);
                this.store(ma, pid, group);
                this.setSgroupParent(ma, mclone, true);
                this.copyBonds(m.getAtom(j), ma);
            }
        }
        this.isolate(m);
    }

    private MulticenterSgroup findMulticenterSgroup() {
        for (int i = this.getSgroupCount() - 1; i >= 0; --i) {
            MulticenterSgroup msg;
            Sgroup sg = this.getSgroup(i);
            if (sg.getType() != 14 || !(msg = (MulticenterSgroup)sg).hasNonCoordinateBond()) continue;
            return msg;
        }
        return null;
    }

    private static int getBondCode(int id, int type) {
        return id * 16 + type;
    }

    private static int getMultiBondCode(int id, int type) {
        return -(Supergraph.getBondCode(id, type) + 16);
    }

    private static int getBondAtomID(int code) {
        if (code < 0) {
            code = -code - 16;
        }
        return code / 16;
    }

    private static int getBondType(int code) {
        if (code < 0) {
            code = -code - 16;
        }
        return code % 16;
    }

    private void addHelperAtoms(ArrayList<MolAtom> helperAtomList) throws SupergraphException {
        this.addHelperAtoms(this, helperAtomList);
        if (this.origRgMol != null) {
            for (int i = this.origRgMol.getRgroupCount() - 1; i >= 0; --i) {
                for (int j = this.origRgMol.getRgroupMemberCount(i) - 1; j >= 0; --j) {
                    this.addHelperAtoms(this.origRgMol.getRgroupMember(i, j), null);
                }
            }
        }
    }

    private void addHelperAtoms(Molecule mol, ArrayList<MolAtom> helperAtomList) throws SupergraphException {
        Hashtable<MolBond, RepeatingUnitSgroup> xbond2repunit = new Hashtable<MolBond, RepeatingUnitSgroup>();
        Hashtable<MolBond, RepeatingUnitSgroup> mbond2repunit = new Hashtable<MolBond, RepeatingUnitSgroup>();
        for (int i = mol.getSgroupCount() - 1; i >= 0; --i) {
            RepeatingUnitSgroup repunit;
            MolBond[] xbonds;
            Sgroup sgroup = mol.getSgroup(i);
            if (!(sgroup instanceof RepeatingUnitSgroup) || PolymerUtil.isPolymerSgroup(sgroup) || (xbonds = (repunit = (RepeatingUnitSgroup)sgroup).findCrossingBonds()).length <= 1 || xbonds.length > 4) continue;
            for (MolBond xbond : xbonds) {
                xbond2repunit.put(xbond, repunit);
            }
            SelectionMolecule graph = sgroup.getSgroupGraph();
            for (int j = graph.getBondCount() - 1; j >= 0; --j) {
                MolBond bond = graph.getBond(j);
                if (!this.isExpandable(bond.getAtom1()) || !this.isExpandable(bond.getAtom2())) continue;
                mbond2repunit.put(bond, repunit);
            }
        }
        MolBond[] bonds = mol.getBondArray();
        for (int i = bonds.length - 1; i >= 0; --i) {
            this.addHelperAtoms(mol, bonds[i], xbond2repunit, mbond2repunit, helperAtomList);
        }
    }

    private void addHelperAtoms(Molecule mol, MolBond bond, Hashtable<MolBond, RepeatingUnitSgroup> xbond2repunit, Hashtable<MolBond, RepeatingUnitSgroup> mbond2repunit, ArrayList<MolAtom> helperAtomList) {
        RepeatingUnitSgroup repunit = xbond2repunit.get(bond);
        MolBond nextBond = null;
        while (bond != null) {
            nextBond = null;
            MolAtom atom1 = bond.getAtom1();
            MolAtom atom2 = bond.getAtom2();
            boolean expandable1 = this.isExpandable(atom1);
            boolean expandable2 = this.isExpandable(atom2);
            if (expandable1 && expandable2 || repunit != null && (expandable1 || expandable2)) {
                boolean attach2;
                int atno1 = atom1.getAtno();
                int atno2 = atom2.getAtno();
                MolAtom[] link1Atoms = Supergraph.getLinksIncludingLigand(atom1, atom2);
                int minrep1 = atom1.getMinRepetitions();
                int maxrep1 = atom1.getMaxRepetitions();
                MolAtom[] link2Atoms = Supergraph.getLinksIncludingLigand(atom2, atom1);
                int minrep2 = atom2.getMinRepetitions();
                int maxrep2 = atom2.getMaxRepetitions();
                boolean attach1 = atno1 == 134 || link1Atoms != null;
                boolean bl = attach2 = atno2 == 134 || link2Atoms != null;
                if (attach1 && atno2 == 128) {
                    attach2 = true;
                } else if (attach2 && atno1 == 128) {
                    attach1 = true;
                }
                if (attach1 && attach2 || repunit != null) {
                    RepeatingUnitSgroup mrepunit;
                    MolAtom helper = Supergraph.createHelperAtom();
                    mol.add(helper);
                    Supergraph.setID(helper, -1 * mol.indexOf(helper));
                    MolBond[] head = null;
                    MolBond[] tail = null;
                    if (repunit != null) {
                        head = repunit.getHeadCrossingBonds();
                        tail = repunit.getTailCrossingBonds();
                        if (expandable1 && repunit.hasAtom(atom1) || expandable2 && repunit.hasAtom(atom2)) {
                            repunit.add(helper);
                        }
                    }
                    if ((mrepunit = mbond2repunit.get(bond)) != null) {
                        mrepunit.add(helper);
                    }
                    this.splitBond(mol, bond, helper);
                    if (link1Atoms != null) {
                        Supergraph.updateLinkNode(atom1, helper, link1Atoms, minrep1, maxrep1);
                    }
                    if (link2Atoms != null) {
                        Supergraph.updateLinkNode(atom2, helper, link2Atoms, minrep2, maxrep2);
                    }
                    if (repunit != null) {
                        Supergraph.updateRepeatingUnit(repunit, helper, bond, head, tail);
                        if (expandable1 && expandable2) {
                            MolBond molBond = nextBond = repunit.hasAtom(atom1) ? atom2.getBondTo(helper) : atom1.getBondTo(helper);
                        }
                    }
                    if (helperAtomList != null) {
                        helperAtomList.add(helper);
                    }
                }
            }
            bond = nextBond;
        }
    }

    private void removeHelperAtoms(ArrayList<MolAtom> helperAtomList) {
        ArrayList<MolAtom> list1 = new ArrayList<MolAtom>();
        ArrayList<MolAtom> list2 = new ArrayList<MolAtom>();
        for (int i = helperAtomList.size() - 1; i >= 0; --i) {
            MolAtom atom = helperAtomList.get(i);
            this.removeHelperAtom(atom, list1, list2);
        }
    }

    private void removeHelperAtom(MolAtom atom, ArrayList<MolAtom> list1, ArrayList<MolAtom> list2) {
        if (this.indexOf(atom) == -1) {
            return;
        }
        MolBond bond = atom.getBond(0);
        for (int j = atom.getBondCount() - 1; j >= 0; --j) {
            MolBond bondj = atom.getBond(j);
            MolAtom ligandj = atom.getLigand(j);
            int set = bondj.getSetSeq();
            if (set == 0) {
                set = j == 0 ? 3 - atom.getBond(1).getSetSeq() : 3 - atom.getBond(0).getSetSeq();
            }
            assert (set == 1 || set == 2);
            if (set == 1) {
                list1.add(ligandj);
                continue;
            }
            if (set == 2) {
                list2.add(ligandj);
                continue;
            }
            System.err.println("removeHelperAtoms() - panic: illegal bond set: " + set);
        }
        this.isolate(atom);
        bond.setSetSeq(0);
        int size1 = list1.size();
        int size2 = list2.size();
        MolAtom atom1 = null;
        MolAtom atom2 = null;
        for (int j1 = 0; j1 < size1; ++j1) {
            atom1 = list1.get(j1);
            for (int j2 = 0; j2 < size2; ++j2) {
                atom2 = list2.get(j2);
                this.add(bond.cloneBond(atom1, atom2));
            }
        }
        list1.clear();
        list2.clear();
    }

    private void removeHelperAtoms(Molecule mol) {
        if (mol instanceof RgMolecule) {
            RgMolecule rgm = (RgMolecule)mol;
            this.removeHelperAtoms0(rgm.getRoot());
            for (int i = 0; i < rgm.getRgroupCount(); ++i) {
                for (int j = 0; j < rgm.getRgroupMemberCount(i); ++j) {
                    this.removeHelperAtoms0(rgm.getRgroupMember(i, j));
                }
            }
        }
    }

    private void removeHelperAtoms0(Molecule mol) {
        for (int i = mol.getAtomCount() - 1; i >= 0; --i) {
            MolAtom atom = mol.getAtom(i);
            if (!Supergraph.isHelperAtom(atom)) continue;
            this.removeHelperAtom(mol, atom);
        }
    }

    private void removeHelperAtom(Molecule mol, MolAtom atom) {
        if (mol.indexOf(atom) == -1) {
            return;
        }
        this.unsplitBond(mol, atom);
    }

    private static MolAtom[] getLinksIncludingLigand(MolAtom atom, MolAtom ligand) {
        if (atom.isLinkNode()) {
            MolAtom link1 = atom.getLigand(atom.getLinkNodeOuterAtom(0));
            MolAtom link2 = atom.getLigand(atom.getLinkNodeOuterAtom(1));
            if (link1 == ligand || link2 == ligand) {
                return new MolAtom[]{link1, link2};
            }
        }
        return null;
    }

    private static void updateLinkNode(MolAtom atom, MolAtom link, MolAtom[] oldLinks, int minrep, int maxrep) {
        atom.setLinkNodeOuterAtom(0, -1);
        atom.setLinkNodeOuterAtom(1, -1);
        int linkIndex = -1;
        for (int i = atom.getBondCount() - 1; i >= 0; --i) {
            MolAtom ligand = atom.getLigand(i);
            if (ligand == oldLinks[0]) {
                atom.setLinkNodeOuterAtom(0, i);
                continue;
            }
            if (ligand == oldLinks[1]) {
                atom.setLinkNodeOuterAtom(1, i);
                continue;
            }
            if (ligand != link) continue;
            linkIndex = i;
        }
        if (atom.getLinkNodeOuterAtom(0) == -1) {
            atom.setLinkNodeOuterAtom(0, linkIndex);
        } else if (atom.getLinkNodeOuterAtom(1) == -1) {
            atom.setLinkNodeOuterAtom(1, linkIndex);
        }
        atom.setMinRepetitions(minrep);
        atom.setMaxRepetitions(maxrep);
    }

    private static void updateRepeatingUnit(RepeatingUnitSgroup repunit, MolAtom helper, MolBond xbond, MolBond[] head, MolBond[] tail) {
        int i;
        MolAtom atom1 = xbond.getAtom1();
        MolAtom atom2 = xbond.getAtom2();
        if (head != null) {
            for (i = 0; i < head.length; ++i) {
                if (head[i] != xbond) continue;
                head[i] = repunit.hasAtom(atom1) ? atom1.getBondTo(helper) : atom2.getBondTo(helper);
                repunit.setHeadCrossingBonds(head);
                return;
            }
        }
        if (tail != null) {
            for (i = 0; i < tail.length; ++i) {
                if (tail[i] != xbond) continue;
                tail[i] = repunit.hasAtom(atom1) ? atom1.getBondTo(helper) : atom2.getBondTo(helper);
                repunit.setTailCrossingBonds(tail);
                return;
            }
        }
    }

    public final int getID(int i) {
        if (this.atomIds == null) {
            return Supergraph.getID(this.getAtom(i));
        }
        return this.atomIds[i];
    }

    public boolean isCompatible(int i1, int i2) {
        return this.isCompatibleByID(this.getID(i1), this.getID(i2));
    }

    public boolean isCompatibleByID(int id1, int id2) {
        int bid2;
        int r1 = id1;
        int r2 = id2;
        int c1 = r1;
        int c2 = r2;
        int bid1 = this.groups[c1] < -1 ? Supergraph.getBondAtomID(this.groups[c1]) : -1;
        int n = bid2 = this.groups[c2] < -1 ? Supergraph.getBondAtomID(this.groups[c2]) : -1;
        while (r1 != r2) {
            if (r1 < r2) {
                c2 = r2;
                r2 = this.parentIDs[r2];
                if (bid2 != -1 || this.groups[c2] >= -1) continue;
                bid2 = Supergraph.getBondAtomID(this.groups[c2]);
                continue;
            }
            c1 = r1;
            r1 = this.parentIDs[r1];
            if (bid1 != -1 || this.groups[c1] >= -1) continue;
            bid1 = Supergraph.getBondAtomID(this.groups[c1]);
        }
        if (this.groups[c1] == this.groups[c2]) {
            r1 = bid1 == -1 ? id1 : bid1;
            int n2 = r2 = bid2 == -1 ? id2 : bid2;
            if (r1 != id1 || r2 != id2) {
                return this.isCompatibleByID(r1, r2);
            }
        }
        return this.groups[c1] == this.groups[c2];
    }

    public int getParentID(int i) {
        return this.getParentIDByID(this.getID(i));
    }

    public int getParentIDByID(int id) {
        return this.parentIDs[id];
    }

    public int getGroup(int i) {
        return this.getGroupByID(this.getID(i));
    }

    public int getGroupByID(int id) {
        return this.groups[id];
    }

    public int getOrigAtomIndex(int i) {
        return this.homologyConverter.getMap()[this.getHomologyConvertedIndex(i)];
    }

    public int getOrigAtomIndexByID(int id) {
        int convIndex = this.getHomologyConvertedIndexByID0(id);
        return convIndex < 0 ? -1 : this.homologyConverter.getMap()[convIndex];
    }

    public int getHomologyConvertedIndex(int i) {
        return this.getHomologyConvertedIndexByID0(this.getID(i));
    }

    public int getHomologyConvertedIndexByID0(int id) {
        int uid = this.unionIDs[id];
        if (uid >= this.origUnionAtomCount) {
            int sgindex = uid - this.origUnionAtomCount;
            MolAtom sgatom = this.origSgroups[sgindex].getAtom(0);
            uid = this.origUnion.indexOf(sgatom);
        }
        return uid;
    }

    public int getRootAtomIndex(int i) {
        return this.getRootAtomIndexByID(this.getID(i));
    }

    public int getRootAtomIndexByID(int id) {
        int r;
        int c = r = id;
        while ((r = this.parentIDs[r]) != -1) {
            c = r;
        }
        return c;
    }

    private MolAtom getOrigAtom(MolAtom atom) {
        return this.origUnion.getAtom(this.unionIDs[Supergraph.getID(atom)]);
    }

    private MolBond getOrigBond(MolBond bond) {
        return this.getOrigAtom(bond.getAtom1()).getBondTo(this.getOrigAtom(bond.getAtom2()));
    }

    public int[] getGraphUnionWalk(int i) {
        return this.getGraphUnionWalkByID(this.getID(i));
    }

    public int[] getGraphUnionWalkByID(int id) {
        int r = id;
        IntVector walk = new IntVector();
        walk.add(this.unionIDs[r]);
        while ((r = this.parentIDs[r]) != -1) {
            if (this.isSgroupMarkerID(r)) continue;
            walk.add(this.unionIDs[r]);
        }
        walk.reverse();
        return walk.toArray();
    }

    private boolean isSgroupMarkerID(int id) {
        return this.unionIDs[id] >= this.origUnionAtomCount;
    }

    private boolean isExpandable(MolAtom atom) {
        return this.expansionHelper.isExpandable(atom) || Supergraph.isRepeatingUnitMarker(atom);
    }

    private boolean isExpandable(MolBond bond) {
        return this.expansionHelper.isExpandable(bond);
    }

    private static void setUnionID(MolAtom atom, int id) {
        atom.putProperty(SG_UNION_ATOMID, id);
    }

    private static int getUnionID(MolAtom atom) {
        return atom.containsPropertyKey(SG_UNION_ATOMID) ? (Integer)atom.getProperty(SG_UNION_ATOMID) : -1;
    }

    private int setID(MolAtom atom) {
        atom.putProperty(SG_ATOMID, this.nextID);
        return this.nextID++;
    }

    private static void setID(MolAtom atom, int id) {
        atom.putProperty(SG_ATOMID, id);
    }

    protected static int getID(MolAtom atom) {
        return atom.containsPropertyKey(SG_ATOMID) ? (Integer)atom.getProperty(SG_ATOMID) : -1;
    }

    private static MolAtom createHelperAtom() {
        return ExpansionUtil.createHelperAtom();
    }

    private static boolean isHelperAtom(MolAtom atom) {
        return ExpansionUtil.isHelperAtom(atom);
    }

    private void splitBond(Molecule mol, MolBond bond, MolAtom middle) {
        ExpansionUtil.splitBond(mol, bond, middle, this.origRgMol);
    }

    private void unsplitBond(Molecule mol, MolAtom middle) {
        ExpansionUtil.unsplitBond(mol, middle, this.origRgMol);
    }

    private void copyBonds(MolAtom atom, MolAtom a) {
        this.copyBonds(atom, new MolAtom[]{a});
    }

    private void copyBonds(MolAtom atom, MolAtom[] a) {
        for (int j = atom.getBondCount() - 1; j >= 0; --j) {
            MolBond bond = atom.getBond(j);
            MolAtom ligand = atom.getLigand(j);
            boolean expandable = this.isExpandable(bond);
            for (int i = 0; i < a.length; ++i) {
                if (a[i].isBoundTo(ligand)) continue;
                MolBond bondi = bond.cloneBond(a[i], ligand);
                this.add(bondi);
                if (!expandable || this.bondStack == null) continue;
                this.bondStack.push(bondi);
            }
        }
    }

    @Override
    public int getParity(int i) {
        return this.origUnion.getParity(this.getHomologyConvertedIndex(i));
    }

    @Override
    public int getLocalParity(int i) {
        return this.origUnion.getLocalParity(this.getHomologyConvertedIndex(i));
    }

    @Override
    public int getChirality(int i) {
        return this.origUnion.getChirality(this.getHomologyConvertedIndex(i));
    }

    @Override
    public boolean isAbsStereo() {
        return this.origMol.isAbsStereo();
    }

    @Override
    public void setAbsStereo(boolean c) {
        this.origMol.setAbsStereo(c);
    }

    @Override
    public void valenceCheck() {
        this.origMol.valenceCheck();
    }

    public void valenceCheck(Vector<MolAtom> v) {
        this.origMol.valenceCheck(this.getOrigAtoms(v));
    }

    public void qpropCheck(Vector<MolAtom> v) {
        this.origMol.qpropCheck(this.getOrigAtoms(v));
    }

    private Vector<MolAtom> getOrigAtoms(Vector<MolAtom> v) {
        if (v == null) {
            return null;
        }
        int size = v.size();
        Vector<MolAtom> w = new Vector<MolAtom>(size);
        for (int i = 0; i < size; ++i) {
            MolAtom atom = v.get(i);
            w.add(this.getOrigAtom(atom));
        }
        return w;
    }

    @Override
    public int getStereo2(int i1, int i2, int i3, int i4) {
        return this.getStereo2ByID(this.getID(i1), this.getID(i2), this.getID(i3), this.getID(i4));
    }

    public int getStereo2ByID(int id1, int id2, int id3, int id4) {
        int[][] origatoms = new int[][]{this.getGraphUnionWalkByID(id1), this.getGraphUnionWalkByID(id2), this.getGraphUnionWalkByID(id3), this.getGraphUnionWalkByID(id4)};
        BackTrack bt = new BackTrack(origatoms);
        while (bt.hasMoreElements()) {
            int[] oi = (int[])bt.nextElement();
            if (!this.inDihedral(oi, origatoms)) continue;
            return this.origUnion.getStereo2(origatoms[0][oi[0]], origatoms[1][oi[1]], origatoms[2][oi[2]], origatoms[3][oi[3]]);
        }
        return -1;
    }

    private boolean inDihedral(int[] oi, int[][] origatoms) {
        BondTable btab = this.origUnion.getBondTable();
        int a1 = origatoms[0][oi[0]];
        int a2 = origatoms[1][oi[1]];
        int a3 = origatoms[2][oi[2]];
        int a4 = origatoms[3][oi[3]];
        return btab.getBondIndex(a1, a2) != -1 && btab.getBondIndex(a2, a3) != -1 && btab.getBondIndex(a3, a4) != -1;
    }

    @Override
    public int getStereo2(MolAtom a1, int i2, int i3, MolAtom a4) {
        return this.origUnion.getStereo2(this.getOrigAtom(a1), this.getHomologyConvertedIndex(i2), this.getHomologyConvertedIndex(i3), this.getOrigAtom(a4));
    }

    @Override
    public int getStereo2(MolBond b, MolAtom a1, MolAtom a4) {
        return this.origUnion.getStereo2(this.getOrigBond(b), this.getOrigAtom(a1), this.getOrigAtom(a4));
    }

    @Override
    public int getStereo2(MolBond b, MolAtom a1, MolAtom a4, boolean grcheck) {
        return this.origUnion.getStereo2(this.getOrigBond(b), this.getOrigAtom(a1), this.getOrigAtom(a4), grcheck);
    }

    @Override
    public int getStereo2(MolBond b) {
        return this.origUnion.getStereo2(this.getOrigBond(b));
    }

    @Override
    public boolean canBeCT(int i2, int i3) {
        return this.origUnion.canBeCT(this.getHomologyConvertedIndex(i2), this.getHomologyConvertedIndex(i3));
    }

    @Override
    public boolean canBeCT(int i2, int i3, boolean grcheck) {
        return this.origUnion.canBeCT(this.getHomologyConvertedIndex(i2), this.getHomologyConvertedIndex(i3), grcheck);
    }

    @Override
    public boolean isRingBond(int idx) {
        return this.origUnion.isRingBond(this.origUnion.indexOf(this.getOrigBond(this.getBond(idx))));
    }

    @Override
    public void clonecopy(MoleculeGraph g) {
        super.clonecopy(g);
        if (g instanceof Supergraph) {
            Supergraph s = (Supergraph)g;
            s.origMol = this.origMol;
            s.origRgMol = this.origRgMol;
            s.origMarkush = this.origMarkush;
            s.origUnion = this.origUnion;
            s.origUnionAtomCount = this.origUnionAtomCount;
            s.origSgroups = this.origSgroups;
            s.homologyConverter = this.homologyConverter;
            if (this.parentIDs != null) {
                s.parentIDs = new int[this.parentIDs.length];
                s.groups = new int[this.groups.length];
                s.unionIDs = new int[this.unionIDs.length];
                s.firstChildIDs = new int[this.firstChildIDs.length];
                s.nextSiblingIDs = new int[this.nextSiblingIDs.length];
                s.atomIds = new int[this.atomIds.length];
                System.arraycopy(this.parentIDs, 0, s.parentIDs, 0, this.parentIDs.length);
                System.arraycopy(this.groups, 0, s.groups, 0, this.groups.length);
                System.arraycopy(this.unionIDs, 0, s.unionIDs, 0, this.unionIDs.length);
                System.arraycopy(this.firstChildIDs, 0, s.firstChildIDs, 0, this.firstChildIDs.length);
                System.arraycopy(this.nextSiblingIDs, 0, s.nextSiblingIDs, 0, this.nextSiblingIDs.length);
                System.arraycopy(this.atomIds, 0, s.atomIds, 0, this.atomIds.length);
                s.hasAAInfo = this.hasAAInfo;
                s.isAAAtom = this.isAAAtom;
                s.isAABond = this.isAABond;
            } else {
                s.parentIDs = null;
                s.groups = null;
                s.unionIDs = null;
                s.firstChildIDs = null;
                s.nextSiblingIDs = null;
                s.atomIds = null;
                s.hasAAInfo = -1;
                s.isAAAtom = null;
                s.isAABond = null;
            }
        }
    }

    @Override
    public void clonelesscopy(MoleculeGraph g) {
        super.clonelesscopy(g);
        if (g instanceof Supergraph) {
            Supergraph s = (Supergraph)g;
            s.parentIDs = this.parentIDs;
            s.groups = this.groups;
            s.unionIDs = this.unionIDs;
        }
    }

    @Override
    public Molecule cloneMolecule() {
        Supergraph m = new Supergraph();
        this.clonecopy(m);
        return m;
    }

    @Override
    public MoleculeGraph newInstance() {
        Supergraph m = new Supergraph();
        this.makeItSimilar(m);
        return m;
    }

    public final boolean isAmbiguousAromaticAtom(int a) {
        if (this.hasAAInfo == 0) {
            return this.isAAAtom[a];
        }
        if (this.hasAAInfo == -1) {
            int[] graphUnionWalk = this.getGraphUnionWalk(a);
            for (int i = 0; i < graphUnionWalk.length; ++i) {
                MolAtom ma;
                if (graphUnionWalk[i] >= this.origUnionAtomCount || !AromUtil.hasAmbigAromRingMarker(ma = this.origUnion.getAtom(graphUnionWalk[i]))) continue;
                return true;
            }
        }
        return false;
    }

    public final boolean isAmbiguousAromaticBond(int a1, int a2) {
        boolean r = false;
        switch (this.hasAAInfo) {
            case 0: {
                MolAtom ma1 = this.getAtom(a1);
                MolAtom ma2 = this.getAtom(a2);
                MolBond b = ma1.getBondTo(ma2);
                int bIndex = this.indexOf(b);
                r = this.isAABond[bIndex];
                break;
            }
            case -1: {
                if (!this.isAmbiguousAromaticAtom(a1) || !this.isAmbiguousAromaticAtom(a2)) {
                    r = false;
                    break;
                }
                r = this.isAmbiguousAromaticBond0(a1, a2);
                break;
            }
            case 1: {
                break;
            }
        }
        return r;
    }

    private final boolean isAmbiguousAromaticBond0(int a1, int a2) {
        int[] origConnAtoms = this.getOrigConnectedAtoms(a1, a2);
        if (origConnAtoms == null) {
            return false;
        }
        MolAtom ma1 = this.origUnion.getAtom(origConnAtoms[0]);
        MolAtom ma2 = this.origUnion.getAtom(origConnAtoms[1]);
        return AromUtil.haveCommonAmbigAromRingMarker(ma1, ma2);
    }

    public final boolean isAmbiguousAromaticBond(MolBond b) {
        boolean r = false;
        switch (this.hasAAInfo) {
            case 0: {
                int bIndex = this.indexOf(b);
                r = this.isAABond[bIndex];
                break;
            }
            case -1: {
                int a1 = this.indexOf(b.getAtom1());
                int a2 = this.indexOf(b.getAtom2());
                r = this.isAmbiguousAromaticBond(a1, a2);
                break;
            }
            case 1: {
                break;
            }
        }
        return r;
    }

    public final boolean isAmbiguousAromaticBond(int bIndex) {
        boolean r = false;
        switch (this.hasAAInfo) {
            case 0: {
                r = this.isAABond[bIndex];
                break;
            }
            case -1: {
                MolBond b = this.getBond(bIndex);
                int a1 = this.indexOf(b.getAtom1());
                int a2 = this.indexOf(b.getAtom2());
                r = this.isAmbiguousAromaticBond(a1, a2);
                break;
            }
            case 1: {
                break;
            }
        }
        return r;
    }

    public int getLogicalAromaticity(MolAtom ma_a1, MolAtom ma_a2, MolAtom ma_a3, MolBond b12, MolBond b23) {
        if (!this.isAmbiguousAromaticBond(b23)) {
            return 0;
        }
        if (b23.getType() == 4) {
            return 0;
        }
        if (!this.isAmbiguousAromaticBond(b12)) {
            return 1;
        }
        if (b12.getType() == 4 && b12.getSetSeq() != 63) {
            return 1;
        }
        int a1 = this.indexOf(ma_a1);
        int a2 = this.indexOf(ma_a2);
        int a3 = this.indexOf(ma_a3);
        int[] origConnAtoms12 = this.getOrigConnectedAtoms(a1, a2);
        int[] origConnAtoms23 = this.getOrigConnectedAtoms(a2, a3);
        if (origConnAtoms12 == null || origConnAtoms23 == null) {
            return 0;
        }
        MolAtom ma1 = this.origUnion.getAtom(origConnAtoms12[0]);
        MolAtom ma2 = this.origUnion.getAtom(origConnAtoms12[1]);
        if (ma2 != this.origUnion.getAtom(origConnAtoms23[0])) {
            return 0;
        }
        MolAtom ma3 = this.origUnion.getAtom(origConnAtoms23[1]);
        Set<String> m1 = AromUtil.getAmbigAromRingMarkers(ma1);
        Set<String> m2 = AromUtil.getAmbigAromRingMarkers(ma2);
        if (!m1.isEmpty() && !m2.isEmpty() && AromUtil.hasAmbigAromRingMarker(ma3)) {
            boolean s1Cs2 = m2.containsAll(m1);
            boolean s2Cs1 = m1.containsAll(m2);
            if (s1Cs2 && s2Cs1) {
                return 2;
            }
            if (s1Cs2) {
                return 3;
            }
            if (s2Cs1) {
                return 4;
            }
            return 1;
        }
        return 0;
    }

    public int[] getOrigConnectedAtoms(int a1, int a2) {
        int[] guw1 = this.getGraphUnionWalk(a1);
        int[] guw2 = this.getGraphUnionWalk(a2);
        int[][] walks = new int[][]{guw1, guw2};
        BackTrack bt = new BackTrack(walks);
        while (bt.hasMoreElements()) {
            int aa2;
            int[] idx = (int[])bt.nextElement();
            int aa1 = walks[0][idx[0]];
            if (!this.isConnectedInUnion(aa1, aa2 = walks[1][idx[1]])) continue;
            return new int[]{aa1, aa2};
        }
        return null;
    }

    private boolean isConnectedInUnion(int i, int i1) {
        if (i >= this.origUnionAtomCount || i1 >= this.origUnionAtomCount) {
            return false;
        }
        if (i == i1 && this.origUnion.getAtom(i).isLinkNode()) {
            return true;
        }
        MolAtom ma1 = this.origUnion.getAtom(i);
        MolAtom ma2 = this.origUnion.getAtom(i1);
        return ma1.isBoundTo(ma2);
    }

    @Override
    protected void isolate(MolAtom node) {
        MolAtom atom = node;
        MulticenterSgroup[] sgroups = ExpansionUtil.getMulticenterSgroups(this, atom);
        for (int i = 0; i < sgroups.length; ++i) {
            sgroups[i].removeAtom(atom);
        }
        super.isolate(node);
    }

    private void isolate(MulticenterSgroup m) {
        MolAtom[] atoms = m.getAtomArray();
        m.setCentralAtom(null);
        this.ungroupSgroup(m);
        for (int i = 0; i < atoms.length; ++i) {
            this.isolate(atoms[i]);
        }
    }

    public static boolean isMarkushMolecule(Molecule m) {
        return Supergraph.containsMarkushFeatures(m);
    }

    public static boolean containsMarkushFeatures(Molecule m) {
        if (m instanceof Supergraph) {
            return true;
        }
        return ExpansionUtil.isMarkushMolecule(m) || SearchUtil.hasMarkushQueryAtom(m);
    }

    public static boolean canSupergraphCreated(Molecule m) {
        return Supergraph.errorInSupergraphRepresentation(m) == null;
    }

    public static String errorInSupergraphRepresentation(Molecule m) {
        String ret = null;
        if (m instanceof RgMolecule) {
            RgMolecule rg = (RgMolecule)m;
            ret = Supergraph.errorInSupergraphRepresentation(rg.getRoot());
            if (ret != null) {
                return ret;
            }
            int rgroupCount = rg.getRgroupCount();
            for (int i = 0; i < rgroupCount; ++i) {
                int rgroupMemberCount = rg.getRgroupMemberCount(i);
                for (int j = 0; j < rgroupMemberCount; ++j) {
                    ret = Supergraph.errorInSupergraphRepresentation(rg.getRgroupMember(i, j));
                    if (ret == null) continue;
                    return ret;
                }
            }
        }
        int atomCount = m.getAtomCount();
        for (int i = 0; i < atomCount; ++i) {
            MolAtom ma = m.getAtom(i);
            int at = ma.getAtno();
            switch (at) {
                case 131: {
                    return "Any atom";
                }
                case 132: {
                    return "Hetero atom";
                }
                case 129: {
                    return "NOT list";
                }
            }
        }
        return ExpansionCounter.getMarkushErrorMessage(m);
    }

    @Override
    @Deprecated
    public int[][] getBtab() {
        if (this.allAtomsAreInRoot(false)) {
            return this.getBondTable().getMatrixArray();
        }
        throw new IllegalArgumentException("btab creation not allowed for supergraph targets");
    }

    @Override
    public int[][] getCtab() {
        if (this.allAtomsAreInRoot(false)) {
            return super.getCtab();
        }
        throw new IllegalArgumentException("ctab creation not allowed for supergraph targets");
    }

    @Override
    public int[][] createBHtab() {
        if (this.allAtomsAreInRoot(false)) {
            return super.createBHtab();
        }
        throw new IllegalArgumentException("bhtab creation not allowed for supergraph targets");
    }

    @Override
    public int[][] createCHtab() {
        if (this.allAtomsAreInRoot(false)) {
            return super.createCHtab();
        }
        throw new IllegalArgumentException("chtab creation not allowed for supergraph targets");
    }

    private boolean allAtomsAreInRoot(boolean checkConvertedHomologies) {
        for (int id = 0; id < this.parentIDs.length; ++id) {
            int parentID = this.parentIDs[id];
            if (parentID == -1 || !checkConvertedHomologies && this.getOrigAtomIndexByID(id) != this.getHomologyConvertedIndexByID0(id)) continue;
            return false;
        }
        return true;
    }

    public boolean hasExpandedMarkushFeature(boolean checkConvertedHomologies) {
        return !this.allAtomsAreInRoot(checkConvertedHomologies);
    }

    public Molecule getOrigMarkush() {
        return this.origMarkush;
    }

    public int getMarkushUnionIndex(int index) {
        return this.getMarkushUnionIndex(this.getAtom(index));
    }

    public int getMarkushUnionIndex(MolAtom atom) {
        int markushIndex = this.getOrigAtomIndexByID(Supergraph.getID(atom));
        Object indexProp = this.markush.getAtom(markushIndex).getProperty(ORIG_ATOM_INDEX);
        return indexProp instanceof Integer ? (Integer)indexProp : -1;
    }

    private ExpansionRecord createRgroupExpansionRecord(MolAtom atom, int rgindex, int memberIndex) {
        return new ExpansionRecord(6, this.getMarkushUnionIndex(atom), rgindex, memberIndex);
    }

    private ExpansionRecord createLinkNodeExpansionRecord(MolAtom atom, int repetitions) {
        return new ExpansionRecord(3, this.getMarkushUnionIndex(atom), repetitions);
    }

    private ExpansionRecord createRepeatingUnitExpansionRecord(RepeatingUnitSgroup repunit, int repetitions) {
        return new ExpansionRecord(5, this.findSgroupIndex(repunit), repetitions);
    }

    private ExpansionRecord createAtomListExpansionRecord(MolAtom atom, int atno) {
        return new ExpansionRecord(1, this.getMarkushUnionIndex(atom), atno);
    }

    private ExpansionRecord createBondListExpansionRecord(MolAtom atom1, MolAtom atom2, int bondType) throws SupergraphException {
        int index1 = this.getMarkushUnionIndex(atom1);
        int index2 = this.getMarkushUnionIndex(atom2);
        MolBond bond = this.getUnionBond(this.origMarkush.getAtom(index1), this.origMarkush.getAtom(index2));
        int bondIndex = -1;
        if (bond != null) {
            bondIndex = this.origMarkush.indexOf(bond);
        } else {
            for (ExpansionRecord record : this.expansionRecords) {
                int index;
                if (record.getFeature().getType() != 4 || index1 != (index = record.getValue()) && index2 != index) continue;
                bondIndex = record.getFeature().getData()[0];
                break;
            }
            if (bondIndex == -1) {
                throw new SupergraphException("panic: bond index not found\nmolecule name: " + this.markush.getName());
            }
        }
        return new ExpansionRecord(2, bondIndex, bondType);
    }

    private ExpansionRecord createMultiBondExpansionRecord(MulticenterSgroup msg, MolAtom inner, MolAtom outer) {
        MolAtom ca = msg.getCentralAtom();
        int index1 = this.getMarkushUnionIndex(ca);
        int index2 = this.getMarkushUnionIndex(outer);
        assert (index1 != -1);
        assert (index2 != -1);
        MolBond bond = this.getUnionBond(this.origMarkush.getAtom(index1), this.origMarkush.getAtom(index2));
        int bondIndex = this.origMarkush.indexOf(bond);
        assert (bondIndex != -1);
        return new ExpansionRecord(4, bondIndex, this.findSgroupIndex(msg), this.getMarkushUnionIndex(inner));
    }

    private MolBond getUnionBond(MolAtom atom1, MolAtom atom2) {
        MolBond bond = atom1.getBondTo(atom2);
        if (bond != null || !(this.origMarkush instanceof RgMolecule)) {
            return bond;
        }
        RgMolecule rgmol = (RgMolecule)this.origMarkush;
        int id1 = rgmol.rgroupIdOf(atom1);
        int id2 = rgmol.rgroupIdOf(atom2);
        if (id1 == -1 && id2 == -1) {
            return null;
        }
        if (id1 != -1 && id2 == -1) {
            return Supergraph.getBondToR(atom2, id1);
        }
        if (id1 == -1 && id2 != -1) {
            return Supergraph.getBondToR(atom1, id2);
        }
        bond = Supergraph.getBondToR(atom2, id1);
        if (bond == null) {
            bond = Supergraph.getBondToR(atom1, id2);
        }
        return bond;
    }

    private static MolBond getBondToR(MolAtom atom, int rid) {
        for (int i = atom.getBondCount() - 1; i >= 0; --i) {
            MolAtom ligand = atom.getLigand(i);
            if (ligand.getAtno() != 134 || ligand.getRgroup() != rid) continue;
            return atom.getBond(i);
        }
        return null;
    }

    private int findSgroupIndex(Sgroup sgroup) {
        int type = sgroup.getType();
        int count = sgroup.getAtomCount();
        IntVector v = new IntVector(count);
        for (int i = 0; i < count; ++i) {
            MolAtom atom = sgroup.getAtom(i);
            if (Supergraph.isHelperAtom(atom)) continue;
            v.add(this.getMarkushUnionIndex(sgroup.getAtom(i)));
        }
        int[] uids = v.toArray();
        Arrays.sort(uids);
        count = uids.length;
        int[] muids = new int[count];
        for (int i = this.origMarkush.getSgroupCount() - 1; i >= 0; --i) {
            Sgroup sg = this.origMarkush.getSgroup(i);
            if (sg.getType() != type || sg.getAtomCount() != count) continue;
            for (int j = 0; j < count; ++j) {
                muids[j] = this.origMarkush.indexOf(sg.getAtom(j));
            }
            Arrays.sort(muids);
            if (!Arrays.equals(uids, muids)) continue;
            return i;
        }
        assert (false);
        return -1;
    }

    public static void main(String[] args) throws Exception {
        int k;
        if (args.length == 0) {
            System.out.println(HELP);
            return;
        }
        String opts = args[0].indexOf(".") == -1 ? args[0].toUpperCase() : "";
        boolean outputID = opts.indexOf("I") != -1;
        boolean outputParentID = opts.indexOf("P") != -1;
        boolean outputGroup = opts.indexOf("G") != -1;
        boolean outputRootIndex = opts.indexOf("R") != -1;
        boolean outputOrigIndex = opts.indexOf("O") != -1;
        boolean outputWalk = opts.indexOf("W") != -1;
        boolean outputSaveStructure = opts.indexOf("S") != -1;
        boolean markushArom = opts.indexOf("A") != -1;
        MolExporter exporter = new MolExporter(System.out, "mrv");
        int n = k = opts.length() == 0 ? 0 : 1;
        while (k < args.length) {
            Molecule[] mols;
            int n2 = 0;
            for (Molecule mol : mols = ConfigTools.getTargetMolecules(args[k])) {
                ++n2;
                if (markushArom) {
                    MarkushAromata aromata = new MarkushAromata();
                    aromata.aromatize(mol);
                }
                try {
                    Supergraph graph = new Supergraph(mol);
                    int count = graph.getAtomCount();
                    for (int i = 0; i < count; ++i) {
                        int r;
                        MolAtom atom = graph.getAtom(i);
                        StringBuffer label = new StringBuffer();
                        if (outputID) {
                            r = graph.getID(i);
                            label.append("I:" + r + " ");
                        }
                        if (outputParentID) {
                            r = graph.getParentID(i);
                            label.append("P:" + r + " ");
                        }
                        if (outputGroup) {
                            r = graph.getGroup(i);
                            label.append("G:" + r + " ");
                        }
                        if (outputRootIndex) {
                            r = graph.getRootAtomIndex(i);
                            label.append("R:" + (r + 1) + " ");
                        }
                        if (outputOrigIndex) {
                            r = graph.getOrigAtomIndex(i);
                            label.append("O:" + (r + 1) + " ");
                        }
                        if (outputWalk) {
                            int[] w = graph.getGraphUnionWalk(i);
                            label.append("W:");
                            for (int j = 0; j < w.length; ++j) {
                                if (j > 0) {
                                    label.append(",");
                                }
                                label.append(w[j] + 1);
                            }
                        }
                        atom.putProperty("SG", new String(label));
                    }
                    exporter.write(outputSaveStructure ? graph.saveStructure() : graph);
                }
                catch (Throwable e) {
                    System.err.println("Error for input: " + args[k] + ": molceule: " + n2 + "\n" + e.getMessage());
                    e.printStackTrace();
                }
            }
            ++k;
        }
        exporter.close();
    }
}

