/*
 * Decompiled with CFR 0.152.
 */
package chemaxon.fragmenter;

import chemaxon.formats.MolFormatException;
import chemaxon.struc.MolAtom;
import chemaxon.struc.Molecule;
import chemaxon.util.CLQ;
import chemaxon.util.MolHandler;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.LineNumberReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.StringTokenizer;

public class FragmentStatistics {
    private static final String lineSep = System.getProperty("line.separator");
    private static final String helptext = lineSep + "FragmentStatistics, (C) 1999-2012 ChemAxon Ltd." + lineSep + "Creates statistics from fragment CXSMILES file." + lineSep + "Statistical category measure and fragment count are read from two SMILES fields" + lineSep + "written by Fragmenter when run with the -s, --statistics option." + lineSep + "Usage:" + lineSep + "  fragstat [options] [input file]" + lineSep + lineSep + "Options: " + lineSep + "  -h, --help                      this help message" + lineSep + "  -s, --help-scoring              help on fragment scoring" + lineSep + "  -c, --cutoffs <cutoffs>         category classes with continuous range" + lineSep + "                                  cutoffs: a list of cutoff values" + lineSep + "                                  defining activity intervals" + lineSep + "                                  (e.g. \"0.5 2.3 5.3\")" + lineSep + "  -r, --range <range>             category classes with discrete range" + lineSep + "                                  range: a list of all possible values" + lineSep + "                                  in ascending activity order" + lineSep + "                                  (e.g. \"0 1 2 3\")" + lineSep + "  -t, --order-type <order>        category activity order" + lineSep + "                                  a: ascending  (the smaller the more active)" + lineSep + "                                  d: descending (the bigger the more active)" + lineSep + "                                  (default: d)" + lineSep + "  -p, --attachment-point <N|S|A>  fragment attachment points:" + lineSep + "                                  N:  no marker atoms" + lineSep + "                                  S:  denoted by any-atom (star) markers" + lineSep + "                                  A:  denoted by Al and Ar atom markers" + lineSep + "                                      (Al: aliphatic, Ar: aromatic)" + lineSep + "                                  (default: N)" + lineSep + "  -a, --all-fragments             display all fragments in output" + lineSep + "                                  (default: only actives in highest category)" + lineSep + "  -m, --min-atom-count            minimum number of heavy atoms" + lineSep + "                                  required in a fragment" + lineSep + "                                  (default: 0 - no restriction)" + lineSep + "  -e, --default-activity <value>  default activity value set for fragments" + lineSep + "                                  with no activity value" + lineSep + "                                  (default: skip these fragments)" + lineSep + "  -d, --display-header            display table header in output" + lineSep + "                                  (otherwise displays only cxsmiles)" + lineSep + "  -o, --output <filepath>         output file path (default: stdout)" + lineSep + lineSep + "Examples:" + lineSep + "  fragstat -c \"4.7 6.8\" -t a -a -o stat.txt fragments.cxsmiles" + lineSep + "  fragstat -r \"C1 C2 C3\" -d -o stat.txt fragments.cxsmiles" + lineSep;
    private static final String helpscoring = lineSep + "Fragment scoring:" + lineSep + lineSep + "  Fragments are sorted by the following scoring function:" + lineSep + "  ac^x*(w1*c1 + w2*c2 + ... + wN*cN)" + lineSep + "  where:" + lineSep + "  ac is the heavy atom count," + lineSep + "  w1, w2, ..., wN are the category weights in descending order," + lineSep + "                  (see the -w, --weight parameter below)" + lineSep + "  c1, c2, ..., cN are the fragment counts in each category," + lineSep + "                  in descending activity order" + lineSep + "  x is an exponent (see the -x, --exponent parameter below)" + lineSep + lineSep + "Scoring Options:" + lineSep + lineSep + "  -w, --weights <weights>     a list of all category weights" + lineSep + "                              (default: from +1 to -1, equidistant, e.g." + lineSep + "                              \"1 -1\" for two, \"1 0 -1\" for three categories)" + lineSep + "  -x, --exponent <x>          the exponent of the atom count" + lineSep + "                              in the scoring function (see above)" + lineSep + "                              (default: 1)" + lineSep + lineSep + "Example:" + lineSep + "  fragstat -c \"4.7 6.8\" -w \"1 -0.5 -1.8\" -x 1.3 -o stat.txt fragments.cxsmiles" + lineSep;
    public static final int ASCENDING = 0;
    public static final int DESCENDING = 1;
    private double[] cutoffs = null;
    private String[] categories = null;
    private int minAtomCount = 0;
    private String defaultActivity = null;
    private double[] weights = null;
    private double x = 1.0;
    private int order = 1;
    private int attachmentPointType = 0;
    private boolean all = false;
    private boolean header = false;

    public FragmentStatistics() {
        this.setDefaultWeights();
    }

    public FragmentStatistics(double[] cutoffs) {
        int n = cutoffs.length;
        this.cutoffs = new double[n];
        System.arraycopy(cutoffs, 0, this.cutoffs, 0, n);
        Arrays.sort(this.cutoffs);
        this.categories = new String[n + 1];
        this.categories[0] = "<" + this.cutoffs[0];
        for (int i = 1; i < n; ++i) {
            this.categories[i] = "[" + this.cutoffs[i - 1] + ", " + this.cutoffs[i] + ")";
        }
        this.categories[n] = ">=" + this.cutoffs[n - 1];
        this.setDefaultWeights();
    }

    public FragmentStatistics(String[] range) {
        this.categories = new String[range.length];
        System.arraycopy(range, 0, this.categories, 0, range.length);
        this.setDefaultWeights();
    }

    private void setDefaultWeights() {
        if (this.categories == null || this.categories.length == 1) {
            this.weights = new double[]{1.0};
            return;
        }
        int n = this.categories.length - 1;
        double d = 2.0 / (double)n;
        this.weights = new double[this.categories.length];
        this.weights[0] = -1.0;
        for (int i = 1; i < n; ++i) {
            this.weights[i] = this.weights[i - 1] + d;
        }
        this.weights[n] = 1.0;
    }

    public void setScoringWeights(double[] weights) {
        if (this.categories == null) {
            throw new IllegalArgumentException("Weights are only valid with categories - no categories have been sepcified");
        }
        if (weights.length != this.categories.length) {
            throw new IllegalArgumentException("The number of weights (" + weights.length + ") should equal to the number of categories (" + this.categories.length + ").");
        }
        this.weights = new double[weights.length];
        System.arraycopy(weights, 0, this.weights, 0, weights.length);
        Arrays.sort(this.weights);
    }

    public void setScoringExponent(double x) {
        this.x = x;
    }

    public void setMinAtomCount(int minAtomCount) {
        this.minAtomCount = minAtomCount;
    }

    public void setAttachmentPointType(int attachmentPointType) {
        this.attachmentPointType = attachmentPointType;
    }

    public void setDefaultActivity(String defaultActivity) {
        this.defaultActivity = defaultActivity;
    }

    public void setDefaultActivity(double defaultActivityValue) {
        this.defaultActivity = String.valueOf(defaultActivityValue);
    }

    public void setOrder(int order) {
        this.order = order;
    }

    public void setAllFragments(boolean all) {
        this.all = all;
    }

    public void setHeader(boolean header) {
        this.header = header;
    }

    public void analyze(InputStream in, OutputStream out) throws IOException, MolFormatException {
        FragmentRecord[] fragments = this.read(in);
        StatisticsRecord[] statistics = this.stat(fragments);
        this.write(statistics, out);
    }

    private FragmentRecord[] read(InputStream in) throws IOException {
        LineNumberReader reader = new LineNumberReader(new BufferedReader(new InputStreamReader(in)));
        ArrayList<FragmentRecord> list = new ArrayList<FragmentRecord>();
        String line = null;
        while ((line = reader.readLine()) != null) {
            if (line.startsWith("#CxSMILES")) continue;
            StringTokenizer st = new StringTokenizer(line, " \t");
            int n = st.countTokens();
            if (st.countTokens() < 2) {
                System.err.println("Error in line (missing field): " + reader.getLineNumber());
                continue;
            }
            String smiles = st.nextToken();
            String labels = st.nextToken();
            String uid = "";
            if (n > 3 || n == 3 && this.categories == null) {
                uid = st.nextToken();
            }
            String value = null;
            if (this.categories != null) {
                value = st.nextToken();
            }
            list.add(new FragmentRecord(smiles, uid, labels, value));
        }
        reader.close();
        FragmentRecord[] fragments = new FragmentRecord[list.size()];
        list.toArray(fragments);
        return fragments;
    }

    private StatisticsRecord[] stat(FragmentRecord[] fragments) throws MolFormatException {
        Arrays.sort(fragments);
        ArrayList list = new ArrayList();
        String id = null;
        StatisticsRecord sr = null;
        int m = this.categories != null && this.order == 1 ? this.categories.length - 1 : 0;
        block2: for (int i = 0; i < fragments.length; ++i) {
            if (id == null || !id.equals(fragments[i].id)) {
                this.store(sr, m, list);
                id = fragments[i].id;
                Molecule mol = new MolHandler(fragments[i].smiles).getMolecule();
                int ac = this.getHeavyAtomCount(mol);
                if (ac < this.minAtomCount) {
                    id = null;
                    sr = null;
                    continue;
                }
                sr = new StatisticsRecord(fragments[i].smiles, fragments[i].labels, ac);
            }
            if (this.categories != null) {
                String value = fragments[i].value;
                if (value == null || value.trim().equals("-")) {
                    value = this.defaultActivity;
                }
                if (value == null) continue;
                if (this.cutoffs != null) {
                    try {
                        int t;
                        double x = Double.parseDouble(value);
                        for (t = 0; t < this.cutoffs.length && x >= this.cutoffs[t]; ++t) {
                        }
                        int n = t;
                        sr.counts[n] = sr.counts[n] + 1;
                    }
                    catch (NumberFormatException e) {}
                    continue;
                }
                for (int t = 0; t < this.categories.length; ++t) {
                    if (!this.categories[t].equals(value)) continue;
                    int n = t;
                    sr.counts[n] = sr.counts[n] + 1;
                    continue block2;
                }
                continue;
            }
            sr.counts[0] = sr.counts[0] + 1;
        }
        this.store(sr, m, list);
        Object[] statistics = new StatisticsRecord[list.size()];
        list.toArray(statistics);
        Arrays.sort(statistics);
        return statistics;
    }

    private int getHeavyAtomCount(Molecule mol) {
        int ac = mol.getAtomCount() - mol.getExplicitHcount();
        if (this.attachmentPointType == 0) {
            return ac;
        }
        for (int i = mol.getAtomCount() - 1; i >= 0; --i) {
            if (!this.isAttachmentAtom(mol.getAtom(i))) continue;
            --ac;
        }
        return ac;
    }

    private boolean isAttachmentAtom(MolAtom atom) {
        int atno = atom.getAtno();
        if (this.attachmentPointType == 2) {
            return atno == 13 || atno == 18;
        }
        if (this.attachmentPointType == 1) {
            return atno == 131;
        }
        return false;
    }

    private void store(StatisticsRecord sr, int m, ArrayList list) {
        if (sr != null) {
            int count = 0;
            if (this.all) {
                for (int j = 0; j < sr.counts.length; ++j) {
                    count += sr.counts[j];
                }
            } else {
                count = sr.counts[m];
            }
            if (count > 0) {
                sr.calcScore();
                list.add(sr);
            }
        }
    }

    private void write(StatisticsRecord[] statistics, OutputStream out) throws IOException {
        BufferedWriter writer = new BufferedWriter(new OutputStreamWriter(out));
        DecimalFormat df = new DecimalFormat();
        df.setMaximumFractionDigits(2);
        df.setGroupingUsed(false);
        if (this.header) {
            if (this.categories != null) {
                writer.write("SMILES AtomLabels\tAtomCount");
                int n = this.categories.length - 1;
                for (int i = 0; i < this.categories.length; ++i) {
                    int t = this.order == 1 ? n - i : i;
                    writer.write("\t" + this.categories[t]);
                }
                writer.write("\tScore");
            } else {
                writer.write("SMILES AtomLabels\tAtomCount\tFragmentCount\tScore");
            }
            writer.write(lineSep);
        }
        for (int i = 0; i < statistics.length; ++i) {
            writer.write(statistics[i].smiles);
            writer.write(" " + statistics[i].labels);
            writer.write("\t" + statistics[i].atomcount);
            if (this.categories != null) {
                int n = this.categories.length - 1;
                for (int j = 0; j < this.categories.length; ++j) {
                    int t = this.order == 1 ? n - j : j;
                    writer.write("\t" + statistics[i].counts[t]);
                }
            } else {
                writer.write("\t" + statistics[i].counts[0]);
            }
            writer.write("\t" + df.format(statistics[i].score));
            writer.write(lineSep);
        }
        ((Writer)writer).flush();
        if (out != System.out) {
            ((Writer)writer).close();
        }
    }

    public static void main(String[] args) throws Exception {
        int i;
        Object[] values;
        StringTokenizer st;
        CLQ.Parameter pMinAtomCount;
        String str;
        double x;
        CLQ clq = new CLQ(args, null);
        if (clq.lookup("-h", "--help", "", 1, false, false) != null) {
            System.out.println(helptext);
            return;
        }
        if (clq.lookup("-s", "--help-scoring", "", 1, false, false) != null) {
            System.out.println(helpscoring);
            return;
        }
        CLQ.Parameter pCutoffs = clq.lookup("-c", "--cutoffs", "", 2, false, false);
        String cutoffs = pCutoffs != null ? pCutoffs.getString() : null;
        CLQ.Parameter pRange = clq.lookup("-r", "--range", "", 2, false, false);
        String range = pRange != null ? pRange.getString() : null;
        CLQ.Parameter pWeights = clq.lookup("-w", "--weights", "", 2, false, false);
        String weights = pWeights != null ? pWeights.getString() : null;
        CLQ.Parameter pExponent = clq.lookup("-x", "--exponent", "", 2, false, false);
        double d = x = pExponent != null ? Double.parseDouble(pExponent.getString()) : Double.NaN;
        if (cutoffs != null && range != null) {
            System.out.println("Parameters -c, --cutoffs and -r, --range are mutually exclusive.");
            return;
        }
        CLQ.Parameter pAttach = clq.lookup("-p", "--attachment-point", "", 2, false, false);
        String attach = pAttach != null ? pAttach.getString().toUpperCase() : "N";
        int attachmentPointType = 0;
        if (attach.startsWith("S")) {
            attachmentPointType = 1;
        } else if (attach.startsWith("A")) {
            attachmentPointType = 2;
        }
        CLQ.Parameter pOrder = clq.lookup("-t", "--order-type", "", 2, false, false);
        int order = 1;
        String string = str = pOrder != null ? pOrder.getString() : null;
        if (str != null) {
            if (str.toLowerCase().startsWith("a")) {
                order = 0;
            } else if (!str.toLowerCase().startsWith("d")) {
                System.out.println("Invalid order type: " + str + " valid types are 'a' and 'd'.");
                return;
            }
        }
        int minAtomCount = (pMinAtomCount = clq.lookup("-m", "--min-atom-count", "", 2, false, false)) != null ? pMinAtomCount.getInt() : 0;
        CLQ.Parameter pDefAct = clq.lookup("-e", "--default-activity", "", 2, false, false);
        String defact = pDefAct != null ? pDefAct.getString() : null;
        CLQ.Parameter pAll = clq.lookup("-a", "--all-fragments", "", 1, false, false);
        CLQ.Parameter pHeader = clq.lookup("-d", "--display-header", "", 1, false, false);
        CLQ.Parameter pOutput = clq.lookup("-o", "--output", "", 2, false, false);
        String instr = clq.notUsed();
        InputStream in = instr != null ? new FileInputStream(instr) : System.in;
        FragmentStatistics fs = null;
        if (cutoffs != null) {
            st = new StringTokenizer(cutoffs, " \t");
            values = new double[st.countTokens()];
            for (i = 0; i < values.length; ++i) {
                values[i] = Double.parseDouble(st.nextToken());
            }
            fs = new FragmentStatistics((double[])values);
        } else if (range != null) {
            st = new StringTokenizer(range, " \t");
            values = new String[st.countTokens()];
            for (i = 0; i < values.length; ++i) {
                values[i] = (double)st.nextToken();
            }
            fs = new FragmentStatistics((String[])values);
        } else {
            fs = new FragmentStatistics();
        }
        if (weights != null) {
            st = new StringTokenizer(weights, " \t");
            values = new double[st.countTokens()];
            for (i = 0; i < values.length; ++i) {
                values[i] = Double.parseDouble(st.nextToken());
            }
            fs.setScoringWeights((double[])values);
        }
        if (!Double.isNaN(x)) {
            fs.setScoringExponent(x);
        }
        fs.setAttachmentPointType(attachmentPointType);
        fs.setMinAtomCount(minAtomCount);
        fs.setDefaultActivity(defact);
        fs.setOrder(order);
        fs.setAllFragments(pAll != null);
        fs.setHeader(pHeader != null);
        OutputStream out = null;
        out = pOutput != null ? new FileOutputStream(pOutput.getString()) : System.out;
        fs.analyze(in, out);
    }

    class StatisticsRecord
    implements Comparable {
        String smiles = null;
        String labels = null;
        int[] counts = null;
        int atomcount = 0;
        double score = 0.0;

        StatisticsRecord(String smiles, String labels, int atomcount) {
            this.smiles = smiles;
            this.labels = labels;
            this.counts = FragmentStatistics.this.categories != null ? new int[FragmentStatistics.this.categories.length] : new int[1];
            this.atomcount = atomcount;
        }

        public void calcScore() {
            this.score = 0.0;
            int n = this.counts.length - 1;
            for (int i = 0; i < this.counts.length; ++i) {
                int t = FragmentStatistics.this.order == 1 ? n - i : i;
                this.score += FragmentStatistics.this.weights[n - i] * (double)this.counts[t];
            }
            if (FragmentStatistics.this.x > -0.001 && FragmentStatistics.this.x < 0.001) {
                return;
            }
            this.score = FragmentStatistics.this.x < 0.999 || FragmentStatistics.this.x > 1.001 ? (this.score *= Math.exp(FragmentStatistics.this.x * Math.log(this.atomcount))) : (this.score *= (double)this.atomcount);
        }

        public int compareTo(Object o) {
            double d = this.score - ((StatisticsRecord)o).score;
            return d > 0.0 ? -1 : (d < 0.0 ? 1 : 0);
        }
    }

    static class FragmentRecord
    implements Comparable {
        String smiles = null;
        String uid = null;
        String labels = null;
        String value = null;
        String id = null;

        FragmentRecord(String smiles, String uid, String labels, String value) {
            this.smiles = smiles;
            this.uid = uid;
            this.labels = labels;
            this.value = value;
            this.id = smiles + " " + uid;
        }

        public int compareTo(Object o) {
            return this.id.compareTo(((FragmentRecord)o).id);
        }
    }
}

