/*
 * Decompiled with CFR 0.152.
 */
package chemaxon.marvin.io.formats.pdb;

import chemaxon.core.util.PDBResidues;
import chemaxon.formats.MolInputStream;
import chemaxon.struc.MacroMolecule;
import chemaxon.struc.Molecule;
import chemaxon.struc.PeriodicSystem;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.util.Hashtable;
import java.util.StringTokenizer;

public class PDBReader {
    private static final int PDB_RECORD_TYPE_COUNT = 53;
    private static final String PDB_RECORD_TYPES = "HEADER:S:M:+:+:OBSLTE:SC:10:O:-:-:TITLE:SC:10:M:-:+:CAVEAT:SC:10:O:-:-:COMPND:SC:10:M:+:+:SOURCE:SC:10:M:-:+:KEYWDS:SC:10:M:-:+:EXPDTA:SC:10:M:-:+:AUTHOR:SC:10:M:-:+:REVDAT:M:M:-:+:SPRSDE:SC:10:O:-:-:JRNL:O:O:-:-:REMARK:O:O:-:-:DBREF:M:O:-:+:SEQADV:M:O:-:-:SEQRES:M:O:+:+:MODRES:M:O:+:+:FTNOTE:MC:10:O:-:-:HET:M:O:+:+:HETNAM:MC:10:O:+:+:HETSYN:M:O:-:-:FORMUL:MC:18:O:-:-:HELIX:M:O:-:+:SHEET:M:O:-:+:TURN:M:O:-:+:SSBOND:M:O:-:-:LINK:M:O:-:-:HYDBND:M:O:-:-:SLTBRG:M:O:-:-:CISPEP:M:O:-:-:SITE:M:O:-:-:CRYST1:S:M:-:-:ORIGX1:S:M:-:-:ORIGX2:S:M:-:-:ORIGX3:S:M:-:-:SCALE1:S:M:-:-:SCALE2:S:M:-:-:SCALE3:S:M:-:-:MTRIX1:M:O:-:-:MTRIX2:M:O:-:-:MTRIX3:M:O:-:-:TVECT:M:O:-:-:MODEL:G:O:+:+:ATOM:M:O:+:+:SIGATM:M:O:-:-:ANISOU:M:O:-:-:SIGUIJ:M:O:-:-:TER:G:O:+:+:HETATM:M:O:+:+:ENDMDL:G:O:-:+:CONECT:M:O:+:+:MASTER:S:M:-:-:END:S:M:+:+:";
    private static final String NL = System.getProperty("line.separator");
    private static final String DEFAULT_CHAIN_ID = "A";
    private boolean consistencyCheck = false;
    private boolean verbose = false;
    private boolean suppressErrors = true;
    private boolean headerProcessed = false;
    private int[] fixedHydrogenCount = new int[100];
    private static final PDBRecordType[] pdbRecordTypes = new PDBRecordType[53];
    private static final Hashtable pdbRecordTypeNameToId = new Hashtable();
    private int modelCount = 0;
    private int modelSerial = 1;
    private boolean noChainDefinition = true;
    private Hashtable chains = new Hashtable();
    private int[] residueCountPerChain = null;
    private int[] atomCountPerChain = null;
    private Hashtable heteroGroups = new Hashtable();
    private int[] atomCountPerHeteroGroup = null;
    private static final String[] WATER_RESIDUE_NAME = new String[]{"HOH", "DOD", "TIP", "TIP3"};
    private int[] waterOCount = new int[WATER_RESIDUE_NAME.length];
    private int[] waterHCount = new int[WATER_RESIDUE_NAME.length];
    private boolean endOfCurrentMolecule = false;
    private String lastChainId = " ";
    private int lastResSeqNo = 0;
    private String lastResName = null;
    private char lastICode = (char)32;
    private Hashtable modresSerialToIndex = new Hashtable();
    private Hashtable modresSerialToResTypeId = new Hashtable();
    private Line line = null;
    private int pass = 0;
    private boolean inOldFormat = false;
    private String idCode = null;
    private String nextChainId = "A";
    private boolean firstATOM = true;
    private int recordTypeId = -1;
    private int lastRecordTypeId = -1;
    private int maxRecordTypeId = -1;
    private int continuation;
    private int lastContinuation;
    private MacroMolecule mm;
    private boolean named = false;
    private int hydrogenizeMode = 0;
    private boolean fixBonds = true;
    private boolean fixBondTypes = true;
    private boolean omitConnect = false;

    public static boolean isPDBRecord(String keyword) {
        return pdbRecordTypeNameToId.containsKey(keyword);
    }

    public void setInput(MolInputStream is) {
        this.line = new Line(is);
    }

    public void setHydrogenize(int hydrogenizeMode) {
        this.hydrogenizeMode = hydrogenizeMode;
        if (hydrogenizeMode != 0) {
            this.fixBonds = true;
        }
    }

    public void setOmitConnect(boolean omitConnect) {
        this.omitConnect = omitConnect;
    }

    public void setFixBondTypes(boolean fixBondTypes) {
        this.fixBondTypes = fixBondTypes;
    }

    public boolean skipMol() throws IOException {
        this.recordTypeId = this.line.getType();
        while (this.line.read() != 1 && !pdbRecordTypes[this.recordTypeId].getType().equals("END")) {
            this.recordTypeId = this.line.getType();
        }
        return true;
    }

    public MacroMolecule read() throws IOException, MacroMolecule.BadMoleculeException {
        long fp = this.getFilePointer();
        this.pass = 1;
        this.endOfCurrentMolecule = false;
        if (!this.readPass()) {
            return null;
        }
        if (!this.headerProcessed) {
            this.inOldFormat = true;
        }
        if (this.modelCount == 0) {
            this.modelCount = 1;
        }
        this.mm = new MacroMolecule(this.modelCount, this.chains, this.residueCountPerChain, this.atomCountPerChain, this.heteroGroups, this.atomCountPerHeteroGroup);
        this.addWater();
        this.pass = 2;
        this.endOfCurrentMolecule = false;
        this.line.reset(fp);
        this.nextChainId = DEFAULT_CHAIN_ID;
        this.firstATOM = true;
        this.readPass();
        if (!this.endOfCurrentMolecule) {
            this.processEND();
        }
        return this.mm;
    }

    public void convert(MacroMolecule mm, Molecule mol) {
        mol.clearForImport("pdb");
        mm.toMolecule(mol);
    }

    public long getFilePointer() {
        return this.line.is.getFilePointer();
    }

    private boolean readPass() throws IOException {
        int lineOk;
        boolean notEmpty = false;
        while (!this.endOfCurrentMolecule && (lineOk = this.line.read()) != 1) {
            block7: {
                if (lineOk == 2 && this.verbose) {
                    System.err.println("PDB Error: " + this.line.lineAsString + " in line " + this.line.lineCount + " is ignored.");
                    continue;
                }
                notEmpty = true;
                this.recordTypeId = this.line.getType();
                if (this.recordTypeId == -1) {
                    if (!this.verbose) continue;
                    System.err.println("PDB Error: " + this.line.lineAsString + " in line " + this.line.lineCount + " is ignored.");
                    continue;
                }
                if (pdbRecordTypes[this.recordTypeId].continued()) {
                    this.continuation = this.line.getContinuation(pdbRecordTypes[this.recordTypeId].continuationAt());
                }
                this.checkConsistency();
                try {
                    this.processRecord();
                }
                catch (MacroMolecule.BadMoleculeException e) {
                    if (this.verbose) {
                        System.err.println("PDB Error: " + e.getMessage());
                    }
                    if (this.suppressErrors) break block7;
                    throw new IOException(e.getMessage());
                }
            }
            this.lastRecordTypeId = this.recordTypeId;
            this.lastContinuation = this.continuation;
            if (this.lastRecordTypeId <= this.maxRecordTypeId) continue;
            this.maxRecordTypeId = this.lastRecordTypeId;
        }
        return notEmpty;
    }

    private void processRecord() throws MacroMolecule.BadMoleculeException {
        if (!pdbRecordTypes[this.recordTypeId].inPass(this.pass)) {
            return;
        }
        try {
            Method proc = this.getClass().getMethod(PDBReader.pdbRecordTypes[this.recordTypeId].processMethodName, null);
            proc.invoke((Object)this, (Object[])null);
        }
        catch (NoSuchMethodException e) {
            e.printStackTrace();
            System.err.println("Internal error in PDBImport.java. \nCurrent record: " + PDBReader.pdbRecordTypes[this.recordTypeId].processMethodName);
        }
        catch (IllegalAccessException e) {
            e.printStackTrace();
            System.err.println("Internal error in PDBImport.java.\nCurrent record: " + PDBReader.pdbRecordTypes[this.recordTypeId].processMethodName);
        }
        catch (InvocationTargetException e) {
            e.printStackTrace();
            throw new RuntimeException("Internal error  in PDBImport, caused by " + e.getMessage() + "\nCurrent record: " + PDBReader.pdbRecordTypes[this.recordTypeId].processMethodName);
        }
    }

    public void processHEADER() {
        this.idCode = this.line.getString(62, 65);
        this.inOldFormat = this.idCode == null || this.idCode.equals(this.line.getStringNoStrip(72, 75));
        this.headerProcessed = true;
        if (this.pass == 2) {
            String classification = this.line.getString(10, 49);
            String depData = this.line.getString(50, 58);
            this.mm.addHeader(classification, depData, this.idCode);
        }
    }

    public void processTITLE() {
        this.mm.addTitle(this.line.getString(10, 69));
    }

    public void processSOURCE() {
        this.mm.addSource(this.line.getString(10, 69));
    }

    public void processKEYWDS() {
        this.mm.addKeywords(this.line.getString(10, 69));
    }

    public void processEXPDTA() {
        this.mm.addExpData(this.line.getString(10, 69));
    }

    public void processAUTHOR() {
        this.mm.addAuthor(this.line.getString(10, 69));
    }

    public void processREVDAT() {
    }

    public void processCOMPND() throws IOException, MacroMolecule.BadMoleculeException {
        switch (this.pass) {
            case 1: {
                this.processCOMPND1();
                break;
            }
            case 2: {
                this.processCOMPND2();
            }
        }
    }

    public void processCOMPND1() throws IOException, MacroMolecule.BadMoleculeException {
        int lastPos = this.line.strCompare("CHAIN:", 10);
        if (lastPos != -1) {
            String chainId = this.line.getToken(lastPos + 1, ",; ");
            while (chainId != null && chainId.length() > 0) {
                this.chains.put(chainId.equalsIgnoreCase("null") ? DEFAULT_CHAIN_ID : chainId, new Integer(this.chains.size()));
                chainId = this.line.getToken();
            }
        }
    }

    public void processCOMPND2() throws IOException, MacroMolecule.BadMoleculeException {
        int lastPos = this.line.strCompare("MOL_ID:", 10);
        if (lastPos != -1) {
            this.mm.addMolId(this.line.getInt(lastPos + 1, ';'));
            return;
        }
        lastPos = this.line.strCompare("MOLECULE:", 10);
        if (lastPos != -1) {
            this.mm.addCompound(this.line.getString(lastPos + 1, ';'));
        } else {
            this.mm.addMolName(this.line.getString(10, 71));
        }
        lastPos = this.line.strCompare("CHAIN:", 10);
        if (lastPos != -1) {
            this.noChainDefinition = false;
            String chainId = this.line.getToken(lastPos + 1, ",; ");
            while (chainId != null && chainId.length() > 0) {
                this.mm.addChain(chainId.equalsIgnoreCase("null") ? DEFAULT_CHAIN_ID : chainId);
                chainId = this.line.getToken();
            }
        }
    }

    public void processDBREF() throws MacroMolecule.BadMoleculeException {
        String chainId = this.line.getStringNoStrip(12, 12);
        if (this.isEmpty(chainId)) {
            chainId = this.nextChainId;
        }
        this.guessNumberOfChainsIfNeeded(chainId);
        int seqBegin = this.line.getInt(14, 17);
        int seqEnd = this.line.getInt(20, 23);
    }

    public void processSEQRES() throws IOException, MacroMolecule.BadMoleculeException {
        String chainId = this.line.getStringNoStrip(11, 11);
        int serNum = this.line.getInt(8, 9);
        if (serNum == 1) {
            this.nextChainId = DEFAULT_CHAIN_ID;
        }
        if (this.isEmpty(chainId)) {
            chainId = this.nextChainId;
        }
        if (this.pass == 1) {
            this.guessNumberOfChainsIfNeeded(chainId);
            return;
        }
        String resName = this.line.getToken(19, " ");
        int chainNo = (Integer)this.chains.get(chainId);
        if (this.residueCountPerChain == null || this.residueCountPerChain[chainNo] == 0) {
            return;
        }
        int numRes = this.line.getInt(13, 16);
        if (!this.mm.addSeqres(chainId, numRes, resName)) {
            return;
        }
        while (resName != null && (!this.heteroGroups.containsKey(resName) || PDBResidues.isModRes(resName))) {
            int resTypeId = PDBResidues.getResidueTypeId(resName);
            if (resTypeId == -1) {
                resTypeId = PDBResidues.addModRes(resName);
            }
            this.mm.addSeqres(resTypeId);
            resName = this.line.getToken(69);
        }
    }

    public void processMODRES() {
        if (this.pass == 1) {
            String resName = this.line.getString(12, 14);
            String stdRes = this.line.getString(24, 26);
            String comment = this.line.getString(29, 69);
            if (PDBResidues.isStandardResidue(stdRes)) {
                PDBResidues.addModResName(resName, stdRes);
            }
        }
    }

    public void processHET() {
        String hetId = this.line.getString(7, 9);
        if (PDBResidues.isModRes(hetId)) {
            return;
        }
        switch (this.pass) {
            case 1: {
                if (this.heteroGroups.containsKey(hetId)) break;
                this.heteroGroups.put(hetId, new Integer(this.heteroGroups.size()));
                break;
            }
            case 2: {
                String chainId = this.line.getStringNoStrip(12, 12);
                int numHetAtoms = this.line.getInt(20, 24);
                int seqNum = this.line.getInt(13, 16);
                char iCode = this.line.getChar(17);
                this.mm.addHet(hetId, chainId, seqNum, iCode, numHetAtoms);
            }
        }
    }

    public void processHETNAM() {
        int id;
        Molecule residue;
        String resName;
        String hetId = this.line.getString(11, 13);
        String name = this.line.getString(15, '\u0000');
        if (this.pass == 2 && !PDBResidues.isModRes(hetId)) {
            this.mm.addHetNam(hetId, name);
        }
        if (this.pass == 1 && PDBResidues.isModRes(hetId) && ((resName = (residue = PDBResidues.getResidue(id = PDBResidues.getResidueTypeId(hetId))).getName()) == null || resName.length() == 0)) {
            residue.setName(name);
        }
    }

    public void processHELIX() {
        int serNum = this.line.getInt(7, 9);
        if (serNum == 1) {
            this.nextChainId = DEFAULT_CHAIN_ID;
        }
        String helixId = this.line.getString(11, 13);
        String chainId = this.line.getStringNoStrip(19, 19);
        if (this.isEmpty(chainId)) {
            chainId = this.nextChainId;
        }
        int initSeqNum = this.line.getInt(21, 24);
        char initICode = this.line.getChar(25);
        int endSeqNum = this.line.getInt(33, 36);
        char endICode = this.line.getChar(37);
        int helixClass = this.line.getInt(38, 39);
        int length = this.line.getInt(71, 75);
        this.mm.addHelix(serNum, helixId, chainId, initSeqNum, initICode, endSeqNum, endICode, helixClass, length);
    }

    public void processSHEET() {
        int serNum = this.line.getInt(7, 9);
        if (serNum == 1) {
            this.nextChainId = DEFAULT_CHAIN_ID;
        }
        String sheetId = this.line.getString(11, 13);
        int numStrands = this.line.getInt(14, 15);
        String chainId = this.line.getStringNoStrip(21, 21);
        if (this.isEmpty(chainId)) {
            chainId = this.nextChainId;
        }
        int initSeqNum = this.line.getInt(22, 25);
        char initICode = this.line.getChar(26);
        int endSeqNum = this.line.getInt(33, 36);
        char endICode = this.line.getChar(37);
        int sense = this.line.getInt(38, 39);
        this.mm.addSheet(serNum, sheetId, chainId, initSeqNum, initICode, endSeqNum, endICode, sense, numStrands);
    }

    public void processTURN() {
        int seq = this.line.getInt(7, 9);
        if (seq == 1) {
            this.nextChainId = DEFAULT_CHAIN_ID;
        }
        String turnId = this.line.getString(11, 13);
        String chainId = this.line.getStringNoStrip(19, 19);
        if (this.isEmpty(chainId)) {
            chainId = this.nextChainId;
        }
        int initSeqNum = this.line.getInt(20, 23);
        char initICode = this.line.getChar(24);
        int endSeqNum = this.line.getInt(31, 34);
        char endICode = this.line.getChar(35);
        this.mm.addTurn(seq, turnId, chainId, initSeqNum, initICode, endSeqNum, endICode);
    }

    public void processMODEL() throws IOException, MacroMolecule.BadMoleculeException {
        int m = this.line.getInt(10, 13);
        if (m <= 0) {
            throw new IOException("PDB Error: Bad model serial number in line " + this.line.lineCount + ".");
        }
        this.modelSerial = m;
        if (this.pass == 1) {
            ++this.modelCount;
        } else {
            this.mm.beginModel(m);
        }
        this.firstATOM = true;
    }

    public void processENDMDL() throws IOException, MacroMolecule.BadMoleculeException {
        if (this.pass == 2) {
            this.mm.endModel();
        }
    }

    public void processATOM() throws MacroMolecule.BadMoleculeException {
        this.processATOM(false);
    }

    public void processHETATM() throws MacroMolecule.BadMoleculeException {
        this.processATOM(true);
    }

    public void processATOM(boolean het) throws MacroMolecule.BadMoleculeException {
        int charge;
        boolean unknownResidue;
        if (this.firstATOM) {
            this.nextChainId = DEFAULT_CHAIN_ID;
            this.firstATOM = false;
        }
        int serial = this.line.getInt(6, 10);
        String name = this.line.getString(12, 15);
        if (name.endsWith("XT")) {
            return;
        }
        char altLoc = this.line.getChar(16);
        String resName = this.line.getStringNoStrip(17, 19);
        boolean bl = unknownResidue = resName.equals("   ") || resName.equals("UNK");
        if (!resName.equals("   ")) {
            resName = resName.trim();
        }
        boolean modRes = PDBResidues.isModRes(resName);
        String chainId = this.line.getStringNoStrip(21, 21);
        int resSeqNo = this.line.getInt(22, 25);
        boolean noStdChainId = this.isEmpty(chainId);
        if (noStdChainId && (!het || modRes)) {
            chainId = this.nextChainId;
        }
        char iCode = this.line.getChar(26);
        float x = this.line.getFloat(30, 37);
        float y = this.line.getFloat(38, 45);
        float z = this.line.getFloat(46, 53);
        float bfactor = this.line.getFloat(60, 65);
        String segId = this.line.getString(72, 75);
        if (noStdChainId && segId != null && segId.length() != 0 && this.charmmChainId(segId)) {
            chainId = segId;
        }
        String elementAtTheEnd = this.line.getString(76, 77);
        String chargeStr = this.line.getString(78, 79);
        if (this.inOldFormat || elementAtTheEnd == null) {
            chargeStr = null;
        }
        String element = this.trimNonAlpha(name);
        if (elementAtTheEnd != null && elementAtTheEnd.length() > 0 && elementAtTheEnd.charAt(0) != ' ' && elementAtTheEnd.contains("[0-9]") && name.contains(elementAtTheEnd)) {
            element = this.trimNonAlpha(elementAtTheEnd);
        }
        if (this.hydrogenizeMode == 16 && this.canBeHydrogen(element)) {
            return;
        }
        if (this.pass == 2 && this.canBeHydrogen(element) && PDBResidues.isStandardResidue(resName) && !modRes) {
            if (this.isNewResidue(chainId, resSeqNo, resName, iCode)) {
                this.updateLastResidueProperties(chainId, resSeqNo, resName, iCode);
                for (int i = 0; i < this.fixedHydrogenCount.length; ++i) {
                    this.fixedHydrogenCount[i] = 0;
                }
            }
            name = this.fixNonStandardHydrogen(resName, name, x, y, z);
        }
        int n = chargeStr != null ? (chargeStr.charAt(0) != ' ' ? chargeStr.charAt(0) - 48 : 0) : (charge = 0);
        if (charge != 0 && charge < 8) {
            charge *= chargeStr.charAt(1) != '-' ? -1 : 1;
        }
        switch (this.pass) {
            case 1: {
                if (this.isWater(resName)) {
                    this.updateWater(resName, element);
                    break;
                }
                if (het || unknownResidue) {
                    this.processHETATM1(resName, name, element, serial, chainId, resSeqNo);
                    break;
                }
                this.processATOM1(chainId, resName, resSeqNo, iCode, name, element, serial, x, y, z);
                break;
            }
            case 2: {
                this.mm.addAtom(serial, name, altLoc, resName, chainId, resSeqNo, iCode, x, y, z, element, charge, het && !modRes, bfactor);
            }
        }
    }

    private String trimNonAlpha(String in) {
        int copyTo;
        int copyFrom;
        String ret = new String();
        for (copyFrom = 0; copyFrom < in.length() && !Character.isLetter(in.charAt(copyFrom)); ++copyFrom) {
        }
        for (copyTo = copyFrom + 1; copyTo < in.length() && Character.isLetter(in.charAt(copyTo)); ++copyTo) {
        }
        return in.substring(copyFrom, copyTo);
    }

    private boolean isWater(String resName) {
        return this.waterResidueTypeIndex(resName) != -1;
    }

    private boolean charmmChainId(String segId) {
        if (segId.equals(this.idCode)) {
            return false;
        }
        return segId.startsWith("PRO") || segId.charAt(0) == 'C' && segId.length() > 3 && Character.isDigit(segId.charAt(3)) || segId.startsWith("DNA") || segId.startsWith("RNA");
    }

    private int waterResidueTypeIndex(String resName) {
        for (int i = 0; i < WATER_RESIDUE_NAME.length; ++i) {
            if (!WATER_RESIDUE_NAME[i].equals(resName)) continue;
            return i;
        }
        return -1;
    }

    private void updateWater(String resName, String element) {
        int wri = this.waterResidueTypeIndex(resName);
        if (element.charAt(0) == 'O') {
            int n = wri;
            this.waterOCount[n] = this.waterOCount[n] + 1;
        }
        if (element.charAt(0) == 'H') {
            int n = wri;
            this.waterHCount[n] = this.waterHCount[n] + 1;
        }
    }

    private boolean canBeHydrogen(String element) {
        char ch0 = element.charAt(0);
        if (ch0 == 'H' || ch0 == 'D') {
            return true;
        }
        return '0' <= ch0 && ch0 <= '9' && element.charAt(1) == 'H';
    }

    private String fixNonStandardHydrogen(String resName, String elementName, float x, float y, float z) {
        int resAtomId;
        int resTypeId = PDBResidues.getResidueTypeId(resName);
        int hId = PDBResidues.getResidueAtomIndex(resTypeId, elementName);
        if (hId != -1) {
            return elementName;
        }
        int n = resAtomId = this.mm.findParentAtom(x, y, z);
        int n2 = this.fixedHydrogenCount[n];
        this.fixedHydrogenCount[n] = n2 + 1;
        String ha = PDBResidues.getHydrogenOfAtom(resTypeId, resAtomId, n2);
        return ha;
    }

    public void processATOM1(String chainId, String resName, int resSeqNo, char iCode, String atomName, String element, int serial, float x, float y, float z) throws MacroMolecule.BadMoleculeException {
        if (this.modelSerial == 1) {
            this.allocAtomCountArrays(chainId);
        }
        int chainNo = (Integer)this.chains.get(chainId);
        if (this.isNewResidue(chainId, resSeqNo, resName, iCode)) {
            int resTypeId = PDBResidues.getResidueTypeId(resName);
            if (resTypeId == -1) {
                resTypeId = PDBResidues.addModRes(resName);
            }
            if (this.modelSerial == 1) {
                int n = chainNo;
                this.residueCountPerChain[n] = this.residueCountPerChain[n] + 1;
                int n2 = chainNo;
                this.atomCountPerChain[n2] = this.atomCountPerChain[n2] + PDBResidues.getResidueAtomCount(resTypeId);
            }
            this.updateLastResidueProperties(chainId, resSeqNo, resName, iCode);
        }
        if (PDBResidues.isModRes(resName)) {
            int index;
            int n = index = PDBResidues.isStdResKnown(resName) ? PDBResidues.addModResAtom(resName, atomName, element) : PDBResidues.addModResAtom(resName, atomName, element, x, y, z);
            if (index != -1) {
                Integer s = new Integer(serial);
                this.modresSerialToIndex.put(s, new Integer(index));
                this.modresSerialToResTypeId.put(s, resName);
            }
            this.allocAtomCountArrays(chainId);
            if (!this.lastChainId.equals(chainId) || this.lastResSeqNo != resSeqNo || !this.lastResName.equals(resName)) {
                int n3 = chainNo;
                this.residueCountPerChain[n3] = this.residueCountPerChain[n3] + 1;
                this.lastChainId = chainId;
                this.lastResSeqNo = resSeqNo;
                this.lastResName = resName;
            }
            int n4 = chainNo;
            this.atomCountPerChain[n4] = this.atomCountPerChain[n4] + 1;
        }
    }

    private void allocAtomCountArrays(String chainId) {
        if (!this.chains.containsKey(chainId)) {
            this.chains.put(chainId, new Integer(this.chains.size()));
        }
        int cc = this.chains.size();
        if (this.residueCountPerChain == null) {
            this.residueCountPerChain = new int[cc];
            this.atomCountPerChain = new int[cc];
        }
        if (this.residueCountPerChain.length < cc) {
            int[] temp = new int[cc];
            System.arraycopy(this.atomCountPerChain, 0, temp, 0, this.atomCountPerChain.length);
            this.atomCountPerChain = temp;
            temp = new int[cc];
            System.arraycopy(this.residueCountPerChain, 0, temp, 0, this.residueCountPerChain.length);
            this.residueCountPerChain = temp;
        }
    }

    private boolean isNewResidue(String chainId, int resSeqNo, String resName, char iCode) {
        return !this.lastChainId.equals(chainId) || this.lastResSeqNo != resSeqNo || !this.lastResName.equals(resName) || this.lastICode != iCode;
    }

    private void updateLastResidueProperties(String chainId, int resSeqNo, String resName, char iCode) {
        this.lastChainId = chainId;
        this.lastResSeqNo = resSeqNo;
        this.lastResName = resName;
        this.lastICode = iCode;
    }

    public void processHETATM1(String resName, String atomName, String element, int serial, String chainId, int resSeqNo) throws MacroMolecule.BadMoleculeException {
        if (this.modelSerial != 1) {
            return;
        }
        if (PDBResidues.isModRes(resName)) {
            int index = PDBResidues.addModResAtom(resName, atomName, element);
            if (index != -1) {
                Integer s = new Integer(serial);
                this.modresSerialToIndex.put(s, new Integer(index));
                this.modresSerialToResTypeId.put(s, resName);
            }
            this.allocAtomCountArrays(chainId);
            int chainNo = (Integer)this.chains.get(chainId);
            if (!this.lastChainId.equals(chainId) || this.lastResSeqNo != resSeqNo || !this.lastResName.equals(resName)) {
                int n = chainNo;
                this.residueCountPerChain[n] = this.residueCountPerChain[n] + 1;
                this.lastChainId = chainId;
                this.lastResSeqNo = resSeqNo;
                this.lastResName = resName;
            }
            int n = chainNo;
            this.atomCountPerChain[n] = this.atomCountPerChain[n] + 1;
        } else {
            int htid;
            if (!this.heteroGroups.containsKey(resName)) {
                this.heteroGroups.put(resName, new Integer(this.heteroGroups.size()));
            }
            this.allocateAtomCountPerHeteroGroup();
            int n = htid = ((Integer)this.heteroGroups.get(resName)).intValue();
            this.atomCountPerHeteroGroup[n] = this.atomCountPerHeteroGroup[n] + 1;
            this.lastChainId = chainId;
        }
    }

    private void allocateAtomCountPerHeteroGroup() {
        if (this.atomCountPerHeteroGroup == null) {
            this.atomCountPerHeteroGroup = new int[2 * this.heteroGroups.size()];
        }
        if (this.atomCountPerHeteroGroup.length < this.heteroGroups.size()) {
            int[] na = new int[this.atomCountPerHeteroGroup.length * 2];
            System.arraycopy(this.atomCountPerHeteroGroup, 0, na, 0, this.atomCountPerHeteroGroup.length);
            this.atomCountPerHeteroGroup = na;
        }
    }

    private String getElementSymbol(String elemString) {
        String e;
        if (elemString.length() == 0) {
            return elemString;
        }
        if (elemString.length() >= 2 && elemString.charAt(1) == 'H') {
            return elemString.substring(1, 2);
        }
        if (elemString.charAt(0) == 'H') {
            return elemString.substring(0, 1);
        }
        if (elemString.charAt(0) != ' ' && (e = this.findElementWithPrefix(elemString)) != null) {
            return e;
        }
        return elemString.substring(elemString.charAt(0) != ' ' ? 0 : 1, elemString.length());
    }

    private String findElementWithPrefix(String elemString) {
        int i;
        for (i = 1; i < 110; ++i) {
            if (!elemString.equals(PeriodicSystem.getSymbol(i))) continue;
            return PeriodicSystem.getSymbol(i);
        }
        elemString = elemString.substring(1);
        for (i = 1; i < 110; ++i) {
            if (!elemString.startsWith(PeriodicSystem.getSymbol(i))) continue;
            return PeriodicSystem.getSymbol(i);
        }
        return null;
    }

    public void processTER() {
        if (this.pass == 2) {
            this.mm.addTer();
        }
        char cid = this.nextChainId.charAt(0);
        this.nextChainId = new String(new char[]{cid = (char)(cid + '\u0001')});
    }

    public void processCONECT() throws MacroMolecule.BadMoleculeException {
        if (this.omitConnect) {
            return;
        }
        int[] atomIndex = new int[]{this.line.getInt(6, 10), this.line.getInt(11, 15), this.line.getInt(16, 20), this.line.getInt(21, 25), this.line.getInt(26, 30)};
        if (this.pass == 1) {
            Integer as = new Integer(atomIndex[0]);
            Integer ai = (Integer)this.modresSerialToIndex.get(as);
            String resName = (String)this.modresSerialToResTypeId.get(as);
            if (resName != null) {
                for (int i = 1; i < 5; ++i) {
                    Integer ba = (Integer)this.modresSerialToIndex.get(new Integer(atomIndex[i]));
                    if (ba == null) continue;
                    PDBResidues.addModResBond(resName, ai, ba);
                }
            }
        } else {
            this.mm.addBond(atomIndex);
        }
    }

    public void processEND() {
        if (this.pass == 1) {
            PDBResidues.completeModifiedResidues();
        } else {
            if (this.fixBonds) {
                this.mm.end(this.hydrogenizeMode, this.fixBondTypes);
            }
            PDBResidues.completeModifiedResidues();
        }
        this.endOfCurrentMolecule = true;
    }

    private void addWater() {
        for (int i = 0; i < WATER_RESIDUE_NAME.length; ++i) {
            if (this.waterOCount[i] == 0) continue;
            this.mm.addWater(WATER_RESIDUE_NAME[i], this.waterOCount[i], this.waterHCount[i]);
        }
    }

    private void checkConsistency() throws IOException {
        if (!this.consistencyCheck) {
            return;
        }
        String error = null;
        if (this.recordTypeId == this.lastRecordTypeId) {
            if (pdbRecordTypes[this.recordTypeId].single() && !pdbRecordTypes[this.recordTypeId].continued()) {
                error = error + "Inconsistent PDB file, error in line " + this.line.lineCount + ": " + PDBReader.pdbRecordTypes[this.lastRecordTypeId].type + " should be single no continued." + NL;
            }
            if (pdbRecordTypes[this.recordTypeId].continued() && this.continuation != 1 && this.continuation != this.lastContinuation + 1) {
                error = error + "Inconsistent PDB file, error in line " + this.line.lineCount + ": wrong continuation." + NL;
            }
        }
        for (int i = this.maxRecordTypeId + 1; i < this.recordTypeId; ++i) {
            if (!pdbRecordTypes[i].mandatory()) continue;
            error = error + "Inconsistent PDB file, error in line " + this.line.lineCount + ": Mandatory record type " + PDBReader.pdbRecordTypes[i].typeAsString + " is missing." + NL;
        }
        if (error != null) {
            if (!this.suppressErrors) {
                throw new IOException(error);
            }
            if (this.verbose) {
                System.err.println(error);
            }
        }
    }

    private void guessNumberOfChainsIfNeeded(String chainId) {
        if (this.pass != 1 || !this.noChainDefinition) {
            return;
        }
        if (!this.chains.containsKey(chainId)) {
            this.chains.put(chainId, new Integer(this.chains.size()));
        }
    }

    private boolean isEmpty(String s) {
        return s == null || s.length() == 0 || s.trim().length() == 0;
    }

    void dump() {
    }

    private static void asser(boolean cond) {
        if (!cond) {
            throw new RuntimeException("Assertion failed.");
        }
    }

    public static void main(String[] args) {
        PDBReader p = new PDBReader();
        try {
            p.setInput(new MolInputStream((InputStream)new FileInputStream(args[0]), "PDB"));
            long t0 = System.currentTimeMillis();
            MacroMolecule m = p.read();
            System.out.println("READ " + (System.currentTimeMillis() - t0) + " ms");
            System.out.println("name = " + m.getName());
            System.out.println("atom count = " + m.getAtomCount());
            System.out.println("defined atom count = " + m.getDefinedAtomCount());
            System.out.println("hydrogen count = " + m.getHydrogenCount());
            System.out.println("header = " + m.getHeader());
            System.out.println("title = " + m.getTitle());
            System.out.println("source = " + m.getSource());
            System.out.println("keywords = " + m.getKeywords());
            System.out.println("exp. data = " + m.getExpData());
            System.out.println("author = " + m.getAuthor());
            NumberFormat decimalForm = NumberFormat.getInstance();
            if (decimalForm instanceof DecimalFormat) {
                decimalForm.setMinimumFractionDigits(3);
                decimalForm.setMaximumFractionDigits(3);
            }
            t0 = System.currentTimeMillis();
            MacroMolecule.ComponentIterator ci = m.getIterator();
            while (ci.hasNext()) {
                MacroMolecule.Component c = ci.next();
                if (c instanceof MacroMolecule.HeteroComponent) {
                    System.out.println("hetId=" + ((MacroMolecule.HeteroComponent)c).getHetId() + " name=" + c.getName());
                } else {
                    System.out.println(c.getName());
                }
                if (!(c instanceof MacroMolecule.Polymer)) continue;
                MacroMolecule.Polymer poly = (MacroMolecule.Polymer)c;
                MacroMolecule.Polymer.BondIterator bi = (MacroMolecule.Polymer.BondIterator)poly.getBondIterator(true);
                System.out.println("bi.bondCount=" + bi.getCount());
                System.out.println("bi.getModelCount()" + bi.getModelCount());
                for (int model = 1; model <= bi.getModelCount(); ++model) {
                    bi.setModel(model);
                    System.out.println("Model " + model);
                    int bc = 0;
                    bi.reset();
                    while (bi.hasNext()) {
                        System.out.println(PDBResidues.getResidueName(bi.getResidueType(1)) + " " + bi.getAtomType(1) + " " + bi.getBondType() + " " + bi.getAtomType(2) + " " + PDBResidues.getResidueName(bi.getResidueType(2)) + " (" + bi.getX(1) + "," + bi.getY(1) + "," + bi.getZ(1) + ") -" + " (" + bi.getX(2) + "," + bi.getY(2) + "," + bi.getZ(2) + ") " + " neighbour (" + bi.getX(0) + "," + bi.getY(0) + "," + bi.getZ(0) + ") ");
                        float bl = (float)Math.sqrt(Math.pow(bi.getX(2) - bi.getX(1), 2.0) + Math.pow(bi.getY(2) - bi.getY(1), 2.0) + Math.pow(bi.getZ(2) - bi.getZ(1), 2.0));
                        System.out.println(" bl = " + bl);
                        ++bc;
                        bi.next();
                    }
                    System.out.println("bc = " + bc);
                }
            }
            System.out.println("ITERA " + (System.currentTimeMillis() - t0) + " ms");
        }
        catch (Exception e) {
            if (p.line != null) {
                System.out.println("line=" + p.line.lineAsString + " linecount=" + p.line.lineCount);
            }
            e.printStackTrace();
        }
    }

    static {
        StringTokenizer st = new StringTokenizer(PDB_RECORD_TYPES, ":");
        int rti = 0;
        while (st.hasMoreTokens()) {
            PDBReader.pdbRecordTypes[rti] = new PDBRecordType(st);
            pdbRecordTypeNameToId.put(PDBReader.pdbRecordTypes[rti].typeAsString, new Integer(rti));
            ++rti;
        }
    }

    private class Line {
        MolInputStream is;
        private char[] line = new char[100];
        private String lineAsString = null;
        private int lineLen = 0;
        private int lineCount = 0;
        private int type = -1;
        private int tokenPos = -1;
        private String tokenTerm = null;

        public Line(MolInputStream is) {
            this.is = is;
            this.is.mark(Integer.MAX_VALUE);
        }

        public void reset() throws IOException {
            this.is.reset();
            this.lineCount = 0;
        }

        public void reset(long filePosition) throws IOException {
            this.is.reset();
            this.is.skip(filePosition);
            this.lineCount = 0;
        }

        public int read() throws IOException {
            Integer t;
            this.tokenPos = -1;
            this.tokenTerm = null;
            this.lineAsString = this.is.readLine();
            if (this.lineAsString == null) {
                return 1;
            }
            ++this.lineCount;
            this.lineLen = this.lineAsString.length();
            this.lineAsString.getChars(0, this.lineLen, this.line, 0);
            if (this.lineAsString.length() < 6) {
                this.lineAsString = this.lineAsString.concat("      ");
            }
            if ((t = (Integer)pdbRecordTypeNameToId.get(this.lineAsString.substring(0, 6))) == null) {
                return 2;
            }
            this.type = t;
            return 0;
        }

        public int getType() {
            return this.type;
        }

        public int getContinuation(int continuationAt) {
            if (this.line[continuationAt] == ' ') {
                return 1;
            }
            if (this.line[continuationAt - 1] == ' ') {
                return this.line[continuationAt] - 48;
            }
            return (this.line[continuationAt - 1] - 48) * 10 + (this.line[continuationAt] - 48);
        }

        public int strCompare(String substr, int pos) {
            int i;
            while (pos < this.lineLen && this.line[pos] == ' ') {
                ++pos;
            }
            if (pos == this.lineLen) {
                return -1;
            }
            for (i = 0; i < substr.length() && pos + i < this.lineLen; ++i) {
                if (this.line[pos + i] == substr.charAt(i)) continue;
                return -1;
            }
            return pos + i;
        }

        public int getInt(int pos, char term) {
            int i;
            for (i = pos; i < this.lineLen && (this.line[i] < '0' || this.line[i] > '9') && this.line[i] != '-'; ++i) {
            }
            if (i == this.lineLen) {
                return -1;
            }
            char ch = this.line[i];
            int sign = 1;
            if (ch == '-') {
                sign = -1;
                ch = this.line[++i];
            }
            int r = 0;
            while (i != this.lineLen && '0' <= ch && ch <= '9' && ch != term) {
                r = r * 10 + (ch - 48);
                ch = this.line[++i];
            }
            return sign * r;
        }

        public int getInt(int from, int to) {
            int i;
            for (i = from; i < this.lineLen && i <= to && this.line[i] == ' '; ++i) {
            }
            if (i == this.lineLen || i > to) {
                return -1;
            }
            char ch = this.line[i];
            int sign = 1;
            if (ch == '-') {
                sign = -1;
                ch = this.line[++i];
            }
            int r = 0;
            while ('0' <= ch && ch <= '9' && i <= to) {
                r = r * 10 + (ch - 48);
                ch = this.line[++i];
            }
            return sign * r;
        }

        public String getString(int pos, char terminator) {
            int lp;
            if (pos >= this.lineLen) {
                return null;
            }
            for (lp = this.lineLen - 1; lp >= 0 && this.line[lp] != terminator; --lp) {
            }
            return lp == -1 ? this.substring(pos) : this.substring(pos, lp);
        }

        public String getString(int from, int to) {
            if (from >= this.lineLen) {
                return null;
            }
            return to >= this.lineLen ? this.substring(from) : this.substring(from, to + 1);
        }

        public String getStringNoStrip(int from, int to) {
            if (from >= this.lineLen) {
                return null;
            }
            return to >= this.lineLen ? new String(this.line, from, this.lineLen - from) : new String(this.line, from, to + 1 - from);
        }

        private String substring(int from, int to) {
            while (from < this.lineLen && from <= to && this.line[from] == ' ') {
                ++from;
            }
            while (to <= this.lineLen && to > from && this.line[to - 1] == ' ') {
                --to;
            }
            return from == this.lineLen || from > to ? null : new String(this.line, from, to - from);
        }

        private String substring(int from) {
            return this.substring(from, this.lineLen);
        }

        public String getToken(int pos, String term) {
            this.tokenPos = pos - 1;
            this.tokenTerm = term;
            return this.getToken();
        }

        public String getToken() {
            ++this.tokenPos;
            while (this.tokenPos < this.lineLen && this.line[this.tokenPos] == ' ') {
                ++this.tokenPos;
            }
            if (this.tokenPos == this.lineLen) {
                return null;
            }
            int firstPos = this.tokenPos;
            while (this.tokenPos < this.lineLen && this.tokenTerm.indexOf(this.line[this.tokenPos]) == -1) {
                ++this.tokenPos;
            }
            return this.tokenPos == this.lineLen ? null : new String(this.line, firstPos, this.tokenPos - firstPos);
        }

        public String getToken(int maxPos) {
            if (maxPos > this.lineLen) {
                return null;
            }
            ++this.tokenPos;
            while (this.tokenPos <= maxPos && this.line[this.tokenPos] == ' ') {
                ++this.tokenPos;
            }
            if (this.tokenPos > maxPos) {
                return null;
            }
            int firstPos = this.tokenPos;
            while (this.tokenPos <= maxPos && this.tokenTerm.indexOf(this.line[this.tokenPos]) == -1) {
                ++this.tokenPos;
            }
            return this.tokenPos - 1 > maxPos ? null : new String(this.line, firstPos, this.tokenPos - firstPos);
        }

        public char getChar(int pos) {
            return this.line[pos];
        }

        public float getFloat(int from, int to) {
            while (from < this.lineLen && from <= to && this.line[from] == ' ') {
                ++from;
            }
            if (from == this.lineLen || from > to) {
                return 0.0f;
            }
            char ch = this.line[from];
            float e = 0.0f;
            int sign = 1;
            if (ch == '-') {
                sign = -1;
                ch = this.line[++from];
            }
            int dec = -1;
            while (from < this.lineLen && from <= to) {
                ch = this.line[from];
                if (ch == '.') {
                    dec = 1;
                } else if ('0' <= ch && ch <= '9') {
                    e = e * 10.0f + (float)(ch - 48);
                    if (dec != -1) {
                        dec *= 10;
                    }
                }
                ++from;
            }
            return (float)sign * e / (float)dec;
        }
    }

    private static class PDBRecordType {
        private static final int SINGLE = 1;
        private static final int MULTIPLE = 2;
        private static final int CONTINUED = 4;
        private static final int GROUPING = 8;
        private static final int OTHER = 16;
        private static final int MANDATORY = 32;
        private static final int FIRST_PASS = 64;
        private static final int SECOND_PASS = 128;
        public char[] type = new char[7];
        public String typeAsString = null;
        public String processMethodName = null;
        public int properties = 0;
        private int continuationAt = -1;

        public PDBRecordType(StringTokenizer st) {
            String mano;
            this.typeAsString = st.nextToken();
            this.typeAsString.getChars(0, this.typeAsString.length(), this.type, 0);
            this.processMethodName = "process" + this.typeAsString;
            while (this.typeAsString.length() < 6) {
                this.typeAsString = this.typeAsString + " ";
            }
            String mul = st.nextToken();
            boolean mayCont = false;
            switch (mul.charAt(0)) {
                case 'S': {
                    this.properties |= 1;
                    mayCont = true;
                    break;
                }
                case 'M': {
                    this.properties |= 2;
                    mayCont = true;
                    break;
                }
                case 'G': {
                    this.properties |= 8;
                    break;
                }
                case 'O': {
                    this.properties |= 0x10;
                    break;
                }
                default: {
                    PDBReader.asser(false);
                }
            }
            if (mayCont && mul.length() == 2) {
                PDBReader.asser(mul.charAt(1) == 'C');
                this.properties |= 4;
                String contAt = st.nextToken();
                this.continuationAt = Integer.parseInt(contAt) - 1;
            }
            PDBReader.asser((mano = st.nextToken()).length() == 1 && (mano.charAt(0) == 'M' || mano.charAt(0) == 'O'));
            this.properties |= mano.charAt(0) == 'M' ? 32 : 0;
            String impl = st.nextToken();
            PDBReader.asser(impl.length() == 1 && (impl.charAt(0) == '-' || impl.charAt(0) == '+'));
            this.properties |= impl.charAt(0) == '+' ? 64 : 0;
            impl = st.nextToken();
            PDBReader.asser(impl.length() == 1 && (impl.charAt(0) == '-' || impl.charAt(0) == '+'));
            this.properties |= impl.charAt(0) == '+' ? 128 : 0;
        }

        public String getType() {
            return this.typeAsString;
        }

        public boolean single() {
            return (this.properties & 1) != 0;
        }

        public boolean multiple() {
            return (this.properties & 2) != 0;
        }

        public boolean continued() {
            return (this.properties & 4) != 0;
        }

        public boolean grouping() {
            return (this.properties & 8) != 0;
        }

        public boolean other() {
            return (this.properties & 0x10) != 0;
        }

        public boolean mandatory() {
            return (this.properties & 0x20) != 0;
        }

        public boolean inPass(int pass) {
            return pass == 1 ? (this.properties & 0x40) != 0 : (this.properties & 0x80) != 0;
        }

        public int continuationAt() {
            return this.continuationAt;
        }

        void dump() {
            System.out.print(this.type + ":");
            System.out.print((this.properties & 1) != 0 ? "S" : "");
            System.out.print((this.properties & 2) != 0 ? "M" : "");
            System.out.print((this.properties & 4) != 0 ? "C" : "");
            System.out.print((this.properties & 8) != 0 ? "G" : "");
            System.out.print((this.properties & 0x10) != 0 ? "O" : "");
            System.out.print((this.properties & 0x20) != 0 ? ":M:" : ":O:");
            System.out.print((this.properties & 0x40) != 0 ? "+:" : "-:");
            System.out.println((this.properties & 0x80) != 0 ? "+:" : "-:");
        }
    }
}

