/*
 * Decompiled with CFR 0.152.
 */
package chemaxon.calculations.pka;

import chemaxon.calculations.PolarGroups;
import chemaxon.calculations.pka.AcidicPKa;
import chemaxon.calculations.pka.AtomPolarizabilityPKa;
import chemaxon.calculations.pka.IntrinsicIonization;
import chemaxon.struc.Molecule;

class BasicPKa {
    private static final double e = 2.718282;
    private IntrinsicIonization pka;
    protected Molecule mol;
    private int[][] pkaBasicType;
    private int[][] cHtab;
    double[] Qsum;
    private double[] Qpi;
    private double[] cyanoInc;
    private int[] aza;
    private PolarGroups pg;
    private int actHeteroCount;

    protected BasicPKa(IntrinsicIonization pka) {
        this.pka = pka;
        this.mol = pka.mol;
        this.pkaBasicType = pka.pkaBasicType;
        this.Qsum = pka.Qsum;
        this.cyanoInc = pka.cyanoInc;
        this.cHtab = pka.cHtab;
        this.aza = pka.getAza();
        this.pg = pka.getPolarGroups();
        this.Qpi = new double[this.mol.getAtomCount() + this.mol.getImplicitHcount()];
        for (int i = 0; i < this.Qpi.length; ++i) {
            this.Qpi[i] = pka.charge.getPiCharge(i);
        }
    }

    protected double calculate(int atom) {
        if (atom < 0 || atom >= this.mol.getAtomCount() || this.mol.getAtom(atom).getCharge() != 0) {
            return Double.NaN;
        }
        switch (this.getAtno(atom)) {
            case 7: {
                return this.getNitrogenBPKa(atom);
            }
            case 6: {
                return this.getCarbonBPKa(atom);
            }
            case 8: {
                return this.getOxigenBPka(atom);
            }
            case 16: {
                return this.getSulfurBPKa(atom);
            }
            case 15: {
                return this.getPhosphorBPKa(atom);
            }
        }
        return Double.NaN;
    }

    private double getNitrogenBPKa(int atom) {
        int baseType = this.pkaBasicType[atom][0];
        int subBaseType = this.pkaBasicType[atom][1];
        if (baseType == 200) {
            this.setDeprotonationState(atom);
            double pkaVal = -47.59 * this.Qsum[atom] - 17.26 * (this.getBondCharge2(atom) - this.Qsum[atom]) - 1.0 * this.getHBondCorrection(atom) - 5.97;
            if (subBaseType == 20) {
                pkaVal += -1.9;
            } else if (subBaseType == 10) {
                pkaVal += 0.52;
            }
            return pkaVal += this.cyanoInc[atom];
        }
        if (baseType == 500 || baseType == 600) {
            double bc = this.getHeteroAtomCharge(atom);
            if (this.pkaBasicType[atom][1] == 125) {
                bc += -0.3497 * bc - 0.1;
            }
            double pkaVal = -107.30000000000001 * bc - 22.207;
            pkaVal += this.getHBond3(atom, bc);
            if (bc < -0.2904) {
                pkaVal -= -50.0 * bc - 14.52;
            }
            if (baseType == 600) {
                pkaVal = 0.83 * pkaVal + 3.02;
            }
            if (subBaseType == 222) {
                if (!this.isOtherNChg(atom)) {
                    pkaVal += 1.8;
                }
            } else if (subBaseType == 322) {
                pkaVal = !this.isOtherNChg(atom) ? (pkaVal += -1.2) : (pkaVal += -2.2);
            } else if (subBaseType == 422) {
                pkaVal = !this.isOtherNChg(atom) ? (pkaVal += -1.2) : (pkaVal += -2.2);
            } else if (subBaseType == 522) {
                pkaVal = !this.isOtherNChg(atom) ? (pkaVal += 1.2) : (pkaVal += -0.5);
            }
            return pkaVal += this.cyanoInc[atom];
        }
        if (baseType == 1) {
            return this.getHeteroAnilinepKa(atom, false, this.cHtab[atom].length == 2);
        }
        if (baseType == 0) {
            int nha;
            double pkaVal = -14.227 * this.getBondCharge2(atom) - 0.3171 - 0.7;
            if (this.aza[atom] != -1) {
                pkaVal -= 5.0;
            }
            if ((nha = this.pka.pkaAcidType[atom][0]) == 22 || nha == 23) {
                pkaVal -= 2.2;
            }
            return pkaVal;
        }
        if (baseType == 2) {
            return this.getNHCOBasicpKa(atom);
        }
        if (baseType == 3) {
            return this.getGuaAmidinepKa(atom);
        }
        if (baseType == 1000) {
            return this.getAliphaticBasicpKa(atom, false);
        }
        if (baseType == 5) {
            return this.getAliphaticImidepKa(atom);
        }
        return Double.NaN;
    }

    private boolean isOtherNChg(int n) {
        for (int i = 0; i < this.cHtab[n].length; ++i) {
            int nb = this.cHtab[n][i];
            if (nb >= this.mol.getAtomCount() || !(this.getBondCharge2(nb) > 0.13)) continue;
            return true;
        }
        return false;
    }

    private double getCarbonBPKa(int atom) {
        int baseType = this.pkaBasicType[atom][0];
        if (baseType == 101) {
            double pka = -504.88 * this.Qsum[atom] - 22.23;
            if (pka > 0.6) {
                pka = 0.6;
            }
            return pka;
        }
        return Double.NaN;
    }

    private double getOxigenBPka(int atom) {
        int baseType = this.pkaBasicType[atom][0];
        int subBaseType = this.pkaBasicType[atom][1];
        int alphaAtom = this.pkaBasicType[atom][2];
        int nitrogen = this.pkaBasicType[atom][3];
        int[] v = new int[4];
        double pkaVal = Double.NaN;
        switch (baseType) {
            case 2000: {
                pkaVal = -114.3 * this.Qsum[atom] - 13.26 * this.getBondCharge2(nitrogen) - 27.67;
                break;
            }
            case 2500: {
                if (subBaseType == 100) {
                    pkaVal = -1.8;
                    break;
                }
                pkaVal = 0.4374 * new AcidicPKa(this.pka).getAlcoholicAcidicpKa(atom) - 9.3598;
                break;
            }
            case 2600: {
                pkaVal = -33.7 * this.Qsum[atom] - 15.6;
                break;
            }
            case 2700: {
                pkaVal = -6.76 * this.Qsum[atom] - 6.7;
                break;
            }
            case 2800: {
                pkaVal = -33.7 * this.Qsum[atom] - 15.6;
                break;
            }
            case 2900: {
                pkaVal = -33.7 * this.Qsum[atom] - 15.6;
                break;
            }
            case 3000: {
                double s = this.getBondCharge11(nitrogen) - this.Qsum[atom];
                pkaVal = -35.0 * s;
                pkaVal += this.setNitrosoCorrection(nitrogen, atom);
                break;
            }
            case 4000: {
                if (subBaseType == 100) {
                    pkaVal = -114.3 * this.Qsum[atom] - 13.26 * this.getBondCharge2(nitrogen) - 27.67;
                    break;
                }
                double s = this.getBondCharge11(nitrogen) - this.Qsum[atom];
                pkaVal = -35.0 * s;
                break;
            }
            case 4100: {
                double s = this.getBondCharge11(nitrogen);
                if (s > 0.0) {
                    pkaVal = -3.96 * s - 17.56;
                    break;
                }
                pkaVal = -46.0 * s - 18.0;
                break;
            }
            case 4200: {
                double r = this.pka.getAvarageAromaticRingCharge(alphaAtom);
                this.getDelIsland(atom, 10);
                pkaVal = -10.0 - 12.5 * (0.15 * r + this.Qsum[atom]) - 0.2 * (double)this.getSphereSize();
                break;
            }
            case 4300: {
                pkaVal = -56.57 * this.Qsum[atom] - 24.287;
                if (subBaseType != 300 && subBaseType != 415) break;
                double bc = 0.0;
                for (int i = 0; i < 4; ++i) {
                    v[i] = this.pkaBasicType[atom][11 - i];
                    if (v[i] == -1) continue;
                    bc += this.getBondCharge12(v[i]);
                    bc += -this.Qsum[v[i]];
                }
                pkaVal += -7.25 * bc + 2.75;
                break;
            }
            case 4350: {
                int alpha = this.cHtab[atom][0];
                this.getAtomDistances(alpha, 4);
                double y = this.getPureCarbCountInc();
                if (subBaseType == 100 || subBaseType == 150) {
                    int n1 = this.pkaBasicType[atom][2];
                    int n2 = this.pkaBasicType[atom][3];
                    int chg1 = this.mol.getAtom(n1).getCharge();
                    int chg2 = this.mol.getAtom(n2).getCharge();
                    double p = this.getBondCharge2(alpha) + this.Qsum[n1] + this.Qsum[n2];
                    pkaVal = chg1 < 0 && chg2 == 0 ? -42.3 * p - 6.667 - 25.0 + y : (chg2 < 0 && chg1 == 0 ? -42.3 * p - 6.667 - 25.0 + y : (chg1 < 0 && chg2 < 0 ? -42.3 * p - 6.667 - 56.0 + y : -42.3 * p - 6.667 + y - 4.5));
                } else if (subBaseType == 200 || subBaseType == 220) {
                    boolean u = this.isAlphaCNegative(alpha);
                    int n1 = this.pkaBasicType[atom][2];
                    int chg1 = this.mol.getAtom(n1).getCharge();
                    double p = this.getBondCharge24(alpha) + this.Qsum[n1];
                    pkaVal = chg1 < 0 ? -42.3 * p - 3.667 - 18.0 + y : (u ? -42.3 * p / 6.0 - 3.667 + y : -42.3 * p - 3.667 + y);
                } else if (subBaseType == 175) {
                    int n1 = this.pkaBasicType[atom][2];
                    int chg1 = this.mol.getAtom(n1).getCharge();
                    double p = this.getBondCharge24(alpha) + 0.25 * this.Qsum[n1];
                    pkaVal = chg1 < 0 ? -32.3 * p - 2.0 - 18.0 + y : (this.aza[n1] != -1 ? -42.3 * p - 6.0 + y : -32.3 * p - 2.0 + y);
                }
                if (this.pg.getGroupProperty()[atom] != 410) break;
                pkaVal += 3.0;
                break;
            }
        }
        return pkaVal;
    }

    private double getSulfurBPKa(int atom) {
        int baseType = this.pkaBasicType[atom][0];
        int nitrogen = this.pkaBasicType[atom][3];
        switch (baseType) {
            case 2000: {
                return -114.3 * this.Qsum[atom] - 13.26 * this.getBondCharge2(nitrogen) - 27.67;
            }
            case 2500: 
            case 2600: 
            case 2700: 
            case 2800: 
            case 2900: {
                return -33.7 * this.Qsum[atom] - 15.6;
            }
            case 3000: {
                return -35.0 * (this.getBondCharge11(nitrogen) - this.Qsum[atom]);
            }
            case 4000: {
                if (this.aza[nitrogen] != -1) {
                    return -114.3 * this.Qsum[atom] - 13.26 * this.getBondCharge2(nitrogen) - 27.67;
                }
                return -35.0 * (this.getBondCharge11(nitrogen) - this.Qsum[atom]);
            }
            case 4350: {
                return -3.0;
            }
        }
        return Double.NaN;
    }

    private double getPhosphorBPKa(int atom) {
        int baseType = this.pkaBasicType[atom][0];
        int subBaseType = this.pkaBasicType[atom][1];
        if (baseType == 2000) {
            return this.getAliphaticBasicpKa(atom, true);
        }
        if (baseType == 1 && subBaseType < 900) {
            return this.getHeteroAnilinepKa(atom, true, false);
        }
        return Double.NaN;
    }

    private void setDeprotonationState(int atom) {
        this.pka.deprotonation = false;
        int[] rInfo = this.pka.charge.findLargestRing(atom);
        if (rInfo[0] != -1 && rInfo.length > 0 && rInfo[0] < this.pka.aAtoms.length) {
            int size = this.pka.aAtoms[rInfo[0]].length;
            int rIndex = rInfo[0];
            for (int i = 0; i < size; ++i) {
                if (this.mol.getAtom(this.pka.aAtoms[rIndex][i]).getCharge() != -1) continue;
                this.pka.deprotonation = true;
                return;
            }
        }
    }

    private double getHeteroAtomCharge(int atom) {
        double sum = this.Qsum[atom];
        for (int i = 0; i < this.cHtab[atom].length; ++i) {
            int nb = this.cHtab[atom][i];
            int vs = this.pka.getValenceState(nb);
            if (vs == 9 || this.getAtno(nb) != 7) continue;
            if (vs == 12) {
                sum += 0.4259 * this.Qsum[nb] + 0.0747;
                continue;
            }
            sum += 0.25 * this.Qsum[nb];
        }
        return sum;
    }

    private double getHBondCorrection(int atom) {
        double HBondIncrement = 0.0;
        for (int i = 0; i < this.cHtab[atom].length; ++i) {
            int nb = this.cHtab[atom][i];
            if (this.cHtab[nb].length != 3 || this.getAtno(nb) != 7) continue;
            HBondIncrement = -15.96 * this.Qsum[atom] + 0.4528 + 0.6;
        }
        return HBondIncrement;
    }

    private double getHBond3(int ArN, double ncharge) {
        int[][] HBT = this.pka.HBT;
        int[] hbtAIndex = this.pka.hbtAIndex;
        double inc = 0.0;
        if (HBT != null && HBT[hbtAIndex[ArN]][11] != -1) {
            inc = this.mol.getAtom(HBT[hbtAIndex[ArN]][11]).getCharge() == -1 ? 10.65 * ncharge + 3.12 : -5.38 * ncharge + 1.93;
        }
        return inc;
    }

    private double getHeteroAnilinepKa(int atom, boolean phosphine, boolean araImine) {
        int subBaseType = this.pkaBasicType[atom][1];
        int commonAtom = this.pkaBasicType[atom][2];
        int aat = this.pkaBasicType[atom][3];
        double pkaVal = Double.NaN;
        double chg = this.getBondCharge2(atom);
        if (subBaseType == 11) {
            pkaVal = araImine || phosphine || this.pg.getCarboxy()[atom] == 1820 ? -51.05 * chg - 26.24 * this.Qpi[atom] + 59.61677 * this.Qsum[atom] - 2.69601 : -61.05 * chg - 26.24 * this.Qpi[atom] + 59.61677 * this.Qsum[atom] + 2.69601;
            if (aat == 2 && this.isDiPhenyl(atom)) {
                pkaVal += -1.5;
            }
        } else if (subBaseType == 901 || subBaseType == 244) {
            int t;
            pkaVal = -31.09 * this.getBondCharge2(atom) - 1.22 * this.Qsum[atom] + 2.99 * (this.setAtomSphere(commonAtom, 4, 0.75) / (double)this.getAtomCount()) - 6.58;
            if (this.aza[atom] != -1 && (t = this.pg.getCarboxy()[this.aza[atom]]) == -1) {
                if (this.mol.getAtom(this.aza[atom]).hasAromaticBond()) {
                    pkaVal += -2.9;
                } else if (this.pkaBasicType[this.aza[atom]][1] == 901) {
                    pkaVal += -2.9;
                } else if (this.pkaBasicType[this.aza[atom]][1] == 1000) {
                    pkaVal += 0.45;
                }
            }
            if (subBaseType == 244) {
                pkaVal += 1.5;
            }
            if (aat == 2 && this.isDiPhenyl(atom)) {
                pkaVal += -1.5;
            }
        } else if (subBaseType == 902) {
            pkaVal = -87.56 * this.Qsum[atom] - 22.28 * this.getBondCharge7(commonAtom, atom) - 14.85;
        } else if (subBaseType == 903) {
            pkaVal = -107.30000000000001 * this.Qsum[atom] - 23.6;
            pkaVal += 0.78 * this.getAliphaticPolInc(atom, 4, 0.75);
        } else if (subBaseType == 904) {
            pkaVal = -107.30000000000001 * this.Qsum[atom] - 23.207;
            pkaVal += 0.8 * this.getAliphaticPolInc(atom, 4, 0.75);
        } else if (subBaseType == 905) {
            pkaVal = -107.30000000000001 * this.Qsum[atom] - 23.207;
            pkaVal += 0.8 * this.getAliphaticPolInc(atom, 4, 0.75);
            pkaVal += -2.8;
        } else {
            pkaVal = this.getAnilinepKa(atom, phosphine);
            if (aat == 2 && this.isDiPhenyl(atom)) {
                pkaVal += -1.5;
            }
        }
        if (pkaVal > 15.0) {
            return 15.0;
        }
        return pkaVal;
    }

    private boolean isDiPhenyl(int a) {
        int c = 0;
        for (int i = 0; i < this.cHtab[a].length; ++i) {
            int nb = this.cHtab[a][i];
            if (nb >= this.mol.getAtomCount() || !this.mol.getAtom(nb).hasAromaticBond()) continue;
            ++c;
        }
        return c == 2;
    }

    private double getBondCharge7(int atom, int n) {
        double sum = this.Qsum[atom];
        for (int j = 0; j < this.cHtab[atom].length; ++j) {
            int nb = this.cHtab[atom][j];
            if (nb == n || nb == atom || atom >= this.mol.getAtomCount()) continue;
            sum += this.getBondCharge2(nb);
            sum -= this.Qsum[atom];
        }
        return sum;
    }

    private double getAliphaticPolInc(int atom, int shellCount, double dFact) {
        AtomPolarizabilityPKa atomPol = new AtomPolarizabilityPKa(this.pka.charge);
        int MAXSPHERE = shellCount;
        int actAtom = atom;
        int s = 0;
        int hCount = 0;
        int MOLATOMS = this.mol.getAtomCount();
        int[] sAtom = new int[MOLATOMS];
        int[] c = new int[MOLATOMS];
        for (int i = 0; i < MOLATOMS; ++i) {
            c[i] = -1;
        }
        int distance = 1;
        boolean sChange = true;
        double hInc = 0.0;
        int i = 0;
        while (distance < MAXSPHERE + 1 & sChange) {
            sChange = false;
            if (actAtom != -1) {
                int length = this.cHtab[actAtom].length;
                for (int j = 0; j < length; ++j) {
                    int sphereAtom = this.cHtab[actAtom][j];
                    if (sphereAtom < MOLATOMS) {
                        if (!(sAtom[sphereAtom] == 0 & sphereAtom != atom) || this.mol.getAtom(sphereAtom).hasAromaticBond()) continue;
                        sAtom[sphereAtom] = distance;
                        c[s] = sphereAtom;
                        ++s;
                        continue;
                    }
                    ++hCount;
                    hInc += 0.314 * Math.pow(dFact, distance - 1);
                }
            }
            if (c[i] != -1) {
                actAtom = c[i];
                distance = sAtom[actAtom] + 1;
                sChange = true;
            } else {
                sChange = false;
            }
            if (++i != MOLATOMS) continue;
            sChange = false;
        }
        double sum1 = 0.0;
        double sum2 = 0.0;
        double[] res = new double[]{0.0, 0.0, 0.0};
        double apol = 0.0;
        for (i = 0; i < s; ++i) {
            int a = c[i];
            distance = sAtom[a];
            res = atomPol.getPolarizabilityParameter(a);
            sum1 += Math.pow(dFact, distance - 1) * res[0];
            sum2 += res[1];
        }
        sum1 = Math.pow(sum1 + hInc, 2.0);
        apol = 4.0 * sum1 / (sum2 += (double)hCount);
        return apol;
    }

    private double getAnilinepKa(int atom, boolean phosphine) {
        int subBaseType = this.pkaBasicType[atom][1];
        int aat = this.pkaBasicType[atom][3];
        double pkaVal = Double.NaN;
        if (aat == 1) {
            pkaVal = -98.068 * this.getBondCharge2(atom) - 306.723 * this.Qsum[atom] + 703.057 * this.Qpi[atom] + 0.912 * (this.setAtomSphere(atom, 10, 1.0) / (double)this.getAtomCount()) - 166.93;
            pkaVal += this.addsAminCorrection(atom, 1);
            pkaVal += this.cyanoInc[atom];
        } else if (aat == 2) {
            if (subBaseType == 16 || subBaseType == 17) {
                pkaVal = -40.67 * this.getBondCharge2(atom) + 0.96 * this.setAtomSphere(atom, 4, 0.75) - 9.95;
                pkaVal += subBaseType == 16 ? 3.5 : 0.5;
            } else if (subBaseType == 15) {
                pkaVal = -40.67 * this.getBondCharge2(atom) + 0.96 * this.setAtomSphere(atom, 4, 0.75) - 9.95;
                pkaVal += -15.05 * this.Qpi[atom];
                pkaVal += this.addsAminCorrection(atom, 2);
                pkaVal += this.cyanoInc[atom];
            } else {
                pkaVal = phosphine ? -3.34 * this.getBondCharge6(atom) - 25.64 * this.getAromaticPiBondCharge(atom) + 0.28 * this.getAliphaticPolInc(atom, 10, 0.75) - 2.0 : -3.34 * this.getBondCharge6(atom) - 45.64 * this.getAromaticPiBondCharge(atom) + 0.28 * this.getAliphaticPolInc(atom, 10, 0.75) + 7.73;
                pkaVal += this.getBetaCorrection(atom);
                pkaVal += this.addsAminCorrection(atom, 2);
                pkaVal += this.cyanoInc[atom];
            }
        } else if (aat == 3) {
            if (subBaseType == 16 || subBaseType == 17) {
                pkaVal = -40.67 * this.getBondCharge2(atom) + 0.96 * this.setAtomSphere(atom, 4, 0.75) - 9.95;
                pkaVal += subBaseType == 16 ? 2.5 : -0.5;
                if (this.aza[atom] != -1) {
                    pkaVal += -1.7;
                }
            } else if (subBaseType == 15) {
                if (phosphine) {
                    pkaVal = -20.67 * this.getBondCharge2(atom) + 0.96 * this.setAtomSphere(atom, 4, 0.75) - 10.95;
                } else {
                    pkaVal = -40.67 * this.getBondCharge2(atom) + 0.96 * this.setAtomSphere(atom, 4, 0.75) - 9.95;
                    pkaVal += -15.05 * this.Qpi[atom];
                }
                pkaVal += this.addsAminCorrection(atom, 3);
                pkaVal += this.cyanoInc[atom];
            } else {
                pkaVal = subBaseType == 155 ? (phosphine ? -40.8 * this.getBondCharge6(atom) + 19.36 * this.getAromaticPiBondCharge(atom) + 0.76 * this.getAliphaticPolInc(atom, 10, 0.75) - 9.3 : -88.8 * this.getBondCharge6(atom) + 19.36 * this.getAromaticPiBondCharge(atom) + 0.76 * this.getAliphaticPolInc(atom, 10, 0.75) - 9.3) : (subBaseType == 144 ? (phosphine ? -30.8 * this.getBondCharge6(atom) + 19.36 * this.getAromaticPiBondCharge(atom) - 10.0 : -88.8 * this.getBondCharge6(atom) + 19.36 * this.getAromaticPiBondCharge(atom) - 10.0) : (phosphine ? -40.67 * this.getBondCharge2(atom) + 0.96 * this.setAtomSphere(atom, 4, 0.75) - 7.95 : -40.67 * this.getBondCharge2(atom) + 0.96 * this.setAtomSphere(atom, 4, 0.75) - 9.95));
                pkaVal += this.getBetaCorrection(atom);
                pkaVal += this.addsAminCorrection(atom, 3);
                pkaVal += this.cyanoInc[atom];
            }
        }
        if (pkaVal > 17.0) {
            return 17.0;
        }
        return pkaVal;
    }

    private double addsAminCorrection(int atom, int aOrder) {
        if (this.pka.HBT != null && this.pka.HBT[this.pka.hbtAIndex[atom]][5] != -1) {
            return this.getAlphaAminInc(atom);
        }
        return 0.0;
    }

    private double getAlphaAminInc(int basicAtom) {
        double sum = 0.0;
        if (this.pka.HBT != null) {
            int y = (this.pka.HBT[this.pka.hbtAIndex[basicAtom]][5] + 1) / 100;
            if (y == 1) {
                sum = 1.8;
            } else if (y == 2) {
                sum = 4.4;
            } else if (y == 3) {
                sum = 5.4;
            }
        }
        return sum;
    }

    private double getAromaticPiBondCharge(int atom) {
        double sum = this.Qpi[atom];
        for (int j = 0; j < this.cHtab[atom].length; ++j) {
            if (this.cHtab[atom][j] >= this.mol.getAtomCount() || !this.mol.getAtom(this.cHtab[atom][j]).hasAromaticBond()) continue;
            sum += this.Qpi[this.cHtab[atom][j]];
        }
        if (this.aza[atom] != -1) {
            sum += 0.2;
        }
        return sum;
    }

    private double getBetaCorrection(int atom) {
        double sum = 0.0;
        for (int i = 0; i < this.cHtab[atom].length; ++i) {
            if (this.getAtno(this.cHtab[atom][i]) == 1 || !(this.Qsum[this.cHtab[atom][i]] > 0.0) || this.mol.getAtom(this.cHtab[atom][i]).hasAromaticBond()) continue;
            sum += -36.5 * this.Qsum[this.cHtab[atom][i]] + 0.125;
        }
        return sum;
    }

    private double getNHCOBasicpKa(int atom) {
        int subBaseType = this.pkaBasicType[atom][1];
        double s = this.getBondCharge10(atom);
        double pka = Double.NaN;
        if (s > 0.009) {
            pka = -1.4455 * Math.log(s) - 8.27;
        } else if (this.Qsum[atom] > 0.0) {
            pka = -543.48 * this.Qsum[atom] * this.Qsum[atom] - 38.08 * this.Qsum[atom] + 0.42;
            if (subBaseType == 201) {
                pka -= 1.4;
            }
        } else {
            pka = -5.38 * this.Qsum[atom] + 0.25 * this.setAtomSphere(atom, 4, 0.75) - 2.26;
            if (subBaseType == 201) {
                pka -= 1.4;
            }
        }
        if (pka > 5.0) {
            return 5.0;
        }
        return pka;
    }

    private double getBondCharge10(int atom) {
        double sum = this.Qsum[atom];
        for (int j = 0; j < this.cHtab[atom].length; ++j) {
            int nb = this.cHtab[atom][j];
            int atno = this.getAtno(nb);
            if (atno == 6 && this.mol.getAtom(nb).getCharge() != -1) {
                sum += this.Qsum[nb];
                continue;
            }
            if (atno == 1) continue;
            sum += 0.5 * this.Qsum[nb];
        }
        return sum;
    }

    private double getGuaAmidinepKa(int atom) {
        int subBaseType = this.pkaBasicType[atom][1];
        int commonAtom = this.pkaBasicType[atom][2];
        int hx = this.pkaBasicType[atom][5];
        double pkaVal = Double.NaN;
        if (subBaseType == 1 || subBaseType == 11 || subBaseType == 51 || subBaseType == 55) {
            this.getDelIsland(commonAtom, 15);
            int sSize = this.getSphereSize();
            pkaVal = -11.162 * this.getBondCharge4(commonAtom) + 0.372 * this.setAtomSphere(commonAtom, 10, 0.75) - 0.138 * (double)sSize + 3.71;
            pkaVal += this.setGuanidinAzaCorrection(this.aza[atom]);
            if (hx != -1) {
                if (this.isCommonHetero(commonAtom)) {
                    pkaVal += -11.26 * this.Qsum[commonAtom] * 3.0;
                }
                pkaVal -= 3.26;
            }
            if (this.aza[atom] != -1 && this.pkaBasicType[this.aza[atom]][0] == 1) {
                pkaVal += 1.5;
            }
            pkaVal = subBaseType == 11 ? (pkaVal += -3.5) : (subBaseType == 51 ? (pkaVal += -1.5) : (subBaseType == 55 ? (pkaVal += -2.0) : (pkaVal += this.cyanoInc[atom])));
        } else if (subBaseType == 12 || subBaseType == 14 || subBaseType == 17 || subBaseType == 18) {
            this.getDelIsland(commonAtom, 15);
            int sSize = this.getSphereSize();
            pkaVal = -11.162 * this.getBondCharge4(commonAtom) + 0.372 * this.setAtomSphere(commonAtom, 10, 0.75) - 0.138 * (double)sSize + 3.71;
            pkaVal += this.setGuanidinAzaCorrection(this.aza[atom]);
            if (subBaseType == 14) {
                pkaVal += -1.2;
            }
            if (subBaseType == 17) {
                pkaVal += -1.7;
            }
            if (subBaseType == 18) {
                pkaVal += -4.0;
            }
            if (hx != -1) {
                pkaVal -= 3.26;
            }
            pkaVal += this.cyanoInc[atom];
        } else if (subBaseType == 127 || subBaseType == 128) {
            if (subBaseType == 128) {
                this.getDelIsland(commonAtom, 15);
                int sSize = this.getSphereSize();
                pkaVal = -11.162 * this.getBondCharge4(commonAtom) + 0.372 * this.setAtomSphere(commonAtom, 10, 0.75) - 0.138 * (double)sSize + 3.71;
                pkaVal += -2.0;
            } else if (subBaseType == 127) {
                pkaVal = -38.52 * this.getBondCharge2(atom) + 9.5 * (this.setAtomSphere(commonAtom, 4, 0.75) / (double)this.getAtomCount()) + 3.2 * this.getBondCharge2(this.pkaBasicType[atom][11]) - 7.6;
                pkaVal += -2.0;
            }
            pkaVal += this.setAmidinAzaCorrection(this.aza[atom]);
            if (hx != -1) {
                pkaVal -= 3.26;
            }
            pkaVal += this.cyanoInc[atom];
        } else if (subBaseType == 125 || subBaseType == 126) {
            if (subBaseType == 126) {
                this.getDelIsland(commonAtom, 15);
                int sSize = this.getSphereSize();
                pkaVal = -11.162 * this.getBondCharge4(commonAtom) + 0.372 * this.setAtomSphere(commonAtom, 10, 0.75) - 0.138 * (double)sSize + 3.71;
                pkaVal += -5.5;
            } else if (subBaseType == 125) {
                pkaVal = -38.52 * this.getBondCharge2(atom) + 9.5 * (this.setAtomSphere(commonAtom, 4, 0.75) / (double)this.getAtomCount()) + 3.2 * this.getBondCharge2(this.pkaBasicType[atom][11]) - 7.6;
                pkaVal += -5.5;
            }
            pkaVal += this.setAmidinAzaCorrection(this.aza[atom]);
            if (hx != -1) {
                pkaVal -= 3.26;
            }
            pkaVal += this.cyanoInc[atom];
        } else if (subBaseType == 72 || subBaseType == 75) {
            pkaVal = -31.09 * this.getBondCharge2(atom) - 1.22 * this.Qsum[atom] + 2.99 * (this.setAtomSphere(atom, 10, 1.0) / (double)this.getAtomCount()) - 5.18;
            if (subBaseType == 75) {
                pkaVal += -4.7;
            }
            pkaVal += this.setAmidinAzaCorrection(this.aza[atom]);
            if (hx != -1) {
                pkaVal -= 3.26;
            }
            pkaVal += this.cyanoInc[atom];
        } else if (subBaseType == 2 || subBaseType == 130 || subBaseType == 22) {
            this.getDelIsland(commonAtom, 2);
            int sSize = this.getSphereSize();
            pkaVal = -22.08 * this.getBondCharge4(commonAtom) + 0.625 * this.setAtomSphere(commonAtom, 2, 0.75) - 0.49 * (double)sSize - 0.068;
            int r = this.isAmidinImide(commonAtom, atom);
            if (r != -1) {
                switch (r) {
                    case 1: {
                        pkaVal += -3.5;
                        break;
                    }
                    case 3: {
                        pkaVal += -1.5;
                        break;
                    }
                    case 4: {
                        if (subBaseType != 2) {
                            pkaVal += 2.4;
                            break;
                        }
                    }
                    case 5: {
                        pkaVal += 0.0;
                    }
                }
            }
            if (subBaseType == 22 && this.pkaBasicType[atom][4] != 1) {
                pkaVal += -1.5;
            }
            if (subBaseType == 130) {
                pkaVal += -2.0;
            }
            pkaVal += this.setAmidinAzaCorrection(this.aza[atom]);
            if (hx != -1) {
                pkaVal -= 3.26;
            }
            pkaVal += this.cyanoInc[atom];
        } else if (subBaseType == 120 || subBaseType == 121 || subBaseType == 122 || subBaseType == 135 || subBaseType == 136) {
            if (subBaseType == 120 || subBaseType == 121) {
                pkaVal = -38.52 * this.getBondCharge2(atom) + 9.5 * (this.setAtomSphere(commonAtom, 4, 0.75) / (double)this.getAtomCount()) + 3.2 * this.getBondCharge2(this.pkaBasicType[atom][11]) - 7.6;
                if (subBaseType == 121) {
                    pkaVal += -1.5;
                }
            } else if (subBaseType == 136 || subBaseType == 135) {
                pkaVal = -38.52 * this.getBondCharge2(atom) + 9.5 * (this.setAtomSphere(commonAtom, 4, 0.75) / (double)this.getAtomCount()) + 3.2 * this.getBondCharge2(this.pkaBasicType[atom][11]) - 7.6;
                if (subBaseType == 136) {
                    pkaVal += -2.0;
                } else if (subBaseType == 135) {
                    pkaVal += -3.5;
                }
            } else if (subBaseType == 122) {
                this.getDelIsland(commonAtom, 15);
                int sSize = this.getSphereSize();
                pkaVal = -11.162 * this.getBondCharge4(commonAtom) + 0.372 * this.setAtomSphere(commonAtom, 10, 0.75) - 0.138 * (double)sSize + 3.71;
            }
            pkaVal += this.setAmidinAzaCorrection(this.aza[atom]);
            if (hx != -1) {
                pkaVal -= 3.26;
            }
            pkaVal += this.cyanoInc[atom];
        }
        if (pkaVal > 15.0) {
            return 15.0;
        }
        return pkaVal;
    }

    private double getBondCharge4(int commonAtom) {
        double sum = 0.0;
        for (int i = 0; i < this.cHtab[commonAtom].length; ++i) {
            int nb = this.cHtab[commonAtom][i];
            if (this.getAtno(nb) == 7 && this.mol.getAtom(nb).getCharge() != -1) {
                sum += this.getBondCharge2(nb) - this.Qsum[commonAtom];
                continue;
            }
            if (this.getAtno(nb) != 6) continue;
            if (this.mol.getAtom(nb).getCharge() != -1) {
                sum += this.Qsum[nb];
                continue;
            }
            if (this.pka.pkaAcidType[nb][5] != 4) continue;
            sum += 0.35;
        }
        return sum += this.Qsum[commonAtom];
    }

    private double setGuanidinAzaCorrection(int azaAtom) {
        if (azaAtom == -1) {
            return 0.0;
        }
        int groupType = this.pg.getGroupType(azaAtom);
        if (groupType == 10 || groupType == 12) {
            return -4.0;
        }
        int carboxy = this.pg.getCarboxy()[azaAtom];
        if (carboxy == 200 || carboxy == 600) {
            return -7.0;
        }
        return -2.0;
    }

    private boolean isCommonHetero(int c) {
        return this.getAtno(c) != 6;
    }

    private double setAmidinAzaCorrection(int azaAtom) {
        if (azaAtom == -1) {
            return 0.0;
        }
        int t = this.pg.getCarboxy()[azaAtom];
        if (t == 200 || t == 600 || t == 500) {
            return -6.5;
        }
        return -3.0;
    }

    private int isAmidinImide(int common, int amidineN) {
        int nb;
        int i;
        int guanidine = 1;
        int amidine = 2;
        int imide = 3;
        int vinilImi = 5;
        for (i = 0; i < this.cHtab[common].length; ++i) {
            nb = this.cHtab[common][i];
            if (nb >= this.mol.getAtomCount()) continue;
            if (this.pkaBasicType[nb][0] == 3 && nb != amidineN) {
                int st = this.pkaBasicType[nb][1];
                if (st == 1 || st == 12 || st == 14) {
                    return guanidine;
                }
                return amidine;
            }
            if (this.pkaBasicType[nb][0] != 5 || nb == amidineN) continue;
            if (this.pkaBasicType[nb][1] == 40) {
                return vinilImi;
            }
            return imide;
        }
        for (i = 0; i < this.cHtab[amidineN].length; ++i) {
            nb = this.cHtab[amidineN][i];
            if (nb >= this.mol.getAtomCount() && nb == common) continue;
            for (int j = 0; j < this.cHtab[nb].length; ++j) {
                int nnb = this.cHtab[nb][j];
                if (nnb >= this.mol.getAtomCount() || this.pkaBasicType[nnb][0] != 5 || nnb == amidineN) continue;
                return 4;
            }
        }
        return -1;
    }

    private double getAliphaticImidepKa(int atom) {
        double iinc;
        int subBaseType = this.pkaBasicType[atom][1];
        int commonAtom = this.pkaBasicType[atom][2];
        int hx = this.pkaBasicType[atom][5];
        double pkaVal = Double.NaN;
        double chg = this.getBondCharge2(atom);
        double d = iinc = chg < -0.288 ? 20.12 * chg + 5.79 : 0.0;
        if (this.pg.getCarboxy()[atom] == 1800) {
            if (this.pka.isInRing[commonAtom]) {
                int ri = this.pkaBasicType[atom][5];
                double ac = this.getAvarageAliphaticRingCharge(ri);
                iinc = -112.9 * ac - 0.92;
                double damp = -74.26 * ac + 2.03;
                damp = damp < 1.0 ? 1.0 : damp;
                iinc /= damp;
            } else {
                iinc = 0.0;
            }
            double d2 = -1.0 * chg - this.Qsum[atom];
            d2 = 2.58 * d2 + 0.019;
            d2 = d2 < 1.0 ? 1.0 : d2;
            pkaVal = (-31.09 * chg - 1.22 * this.Qsum[atom]) / d2 + 2.99 * (this.setAtomSphere(atom, 10, 1.0) / (double)this.getAtomCount()) - 7.18 + iinc;
            return pkaVal;
        }
        pkaVal = -31.09 * chg - 1.22 * this.Qsum[atom] + 2.99 * (this.setAtomSphere(atom, 10, 1.0) / (double)this.getAtomCount()) - 5.18 + iinc;
        if (this.pka.sAra[atom] == 1) {
            int ri = this.pkaBasicType[atom][5];
            double ac = this.getAvarageAliphaticSideRingCharge(ri);
            pkaVal += -115.65 * ac + 0.9035;
        } else if (subBaseType == -1 && this.pka.isInRing[atom]) {
            int ri = this.pkaBasicType[atom][5];
            double ac = this.getAvarageAliphaticRingChargePlus(ri);
            if (this.actHeteroCount >= 2) {
                pkaVal += -30.6 * ac + 3.2;
            }
        }
        if (this.aza[atom] != -1) {
            pkaVal = this.pkaBasicType[this.aza[atom]][0] == 1 ? (pkaVal -= 3.0) : (this.pkaBasicType[this.aza[atom]][0] == 1000 ? (pkaVal -= 5.0) : (pkaVal -= 5.55));
        }
        if (subBaseType == 40) {
            pkaVal += this.getVinilPathIncrement(atom);
        } else if (subBaseType == 402) {
            pkaVal += this.getVinilPathBreakIncrement(atom);
        } else if (subBaseType == 45) {
            pkaVal += 1.3;
        } else if (subBaseType == 50) {
            pkaVal += 1.1;
        } else if (subBaseType == 51) {
            pkaVal += -2.1;
        } else if (subBaseType == 60) {
            pkaVal += 0.0;
        } else if (subBaseType == 11) {
            pkaVal += -4.5;
        }
        if (hx == 10) {
            pkaVal += -2.2;
            if (chg < -0.17) {
                pkaVal -= -40.0 * chg - 6.8;
            }
        } else if (hx == 20) {
            pkaVal += -1.0;
        }
        if ((pkaVal += this.cyanoInc[atom]) > 15.0) {
            return 15.0;
        }
        return pkaVal;
    }

    private double getAvarageAliphaticRingChargePlus(int rind) {
        double sum = 0.0;
        this.actHeteroCount = 0;
        int L = this.pka.nonAring[rind].length;
        for (int i = 0; i < L; ++i) {
            int ra = this.pka.nonAring[rind][i];
            if (this.getAtno(ra) == 6) {
                sum += this.Qsum[ra];
                continue;
            }
            if (this.mol.getAtom(ra).getCharge() == 1) {
                sum += this.Qsum[ra];
                if (!this.pka.isDBonded(ra)) continue;
                ++this.actHeteroCount;
                continue;
            }
            sum += 0.35 * this.Qsum[ra];
            if (!this.pka.isDBonded(ra)) continue;
            ++this.actHeteroCount;
        }
        return sum / (double)L;
    }

    private double getAvarageAliphaticRingCharge(int rind) {
        double sum = 0.0;
        int L = this.pka.nonAring[rind].length;
        for (int i = 0; i < L; ++i) {
            int ra = this.pka.nonAring[rind][i];
            sum += this.getAtno(ra) == 6 ? this.Qsum[ra] : 0.35 * this.Qsum[ra];
        }
        return sum / (double)L;
    }

    private double getAvarageAliphaticSideRingCharge(int rind) {
        int c = 0;
        double sum = 0.0;
        int L = this.pka.nonAring[rind].length;
        for (int i = 0; i < this.pka.nonAring[rind].length; ++i) {
            int ra = this.pka.nonAring[rind][i];
            sum += this.getAtno(ra) == 6 ? this.Qsum[ra] : 0.35 * this.Qsum[ra];
            for (int j = 0; j < this.cHtab[ra].length; ++j) {
                int nb = this.cHtab[ra][j];
                if (this.getAtno(nb) != 6 || this.pka.isTheSameRing(rind, nb)) continue;
                sum += this.Qsum[nb];
                ++c;
            }
        }
        return sum / (double)(c + L);
    }

    private double getVinilPathIncrement(int atom) {
        double sum = 0.0;
        boolean addedYet = false;
        int ind = this.pka.getNind(atom);
        if (ind == -1) {
            return 0.0;
        }
        for (int i = 0; i < 12; ++i) {
            int nb = this.pka.vinilPath[ind][i];
            if (nb == -1 || this.pkaBasicType[nb][0] == -1) continue;
            if (this.mol.getAtom(nb).getCharge() == 1) {
                if (this.pka.mpk0 != null && this.pka.mpk0[nb] > 3.0) {
                    sum += -2.5;
                    continue;
                }
                sum += -1.5;
                continue;
            }
            if (this.pkaBasicType[nb][0] == 3) {
                sum += -2.5;
                continue;
            }
            if (this.pkaBasicType[nb][0] == 1000) {
                if (this.pkaBasicType[nb][10] == 31 || this.pkaBasicType[nb][10] == 32 || addedYet) continue;
                sum = this.pka.sAra[atom] != 1 ? (sum += 3.0) : (sum += 1.0);
                addedYet = true;
                continue;
            }
            if (this.pkaBasicType[nb][0] != 1 || addedYet) continue;
            if (this.pkaBasicType[nb][1] == 15) {
                sum += 4.0;
                addedYet = true;
                continue;
            }
            if (this.pkaBasicType[nb][1] == 155) {
                sum += 2.0;
                addedYet = true;
                continue;
            }
            sum = this.pka.sAra[atom] != 1 ? (sum += 4.0) : (sum += 1.0);
            addedYet = true;
        }
        return sum;
    }

    private double getVinilPathBreakIncrement(int atom) {
        int ind = this.pka.getNind(atom);
        if (ind == -1) {
            return 0.0;
        }
        double sum = 0.0;
        for (int i = 0; i < 12; ++i) {
            int nb = this.pka.vinilPath[ind][i];
            if (nb == -1) continue;
            sum += this.pka.getVinilBreakerInc(nb);
        }
        if (sum != 0.0) {
            sum = 2.2E-4 * Math.pow(sum, 3.0) + 0.02598 * Math.pow(sum, 2.0) + 1.04816 * sum - 0.09189;
        }
        return sum;
    }

    private double getBondCharge11(int atom) {
        double sum = this.Qsum[atom];
        for (int i = 0; i < this.cHtab[atom].length; ++i) {
            int nb = this.cHtab[atom][i];
            if (this.getAtno(nb) == 1) continue;
            if (this.mol.getAtom(nb).getCharge() == 0) {
                sum += this.Qsum[nb];
                continue;
            }
            return this.Qsum[atom];
        }
        return sum;
    }

    private double setNitrosoCorrection(int nit, int ox) {
        for (int i = 0; i < this.cHtab[nit].length; ++i) {
            int nb = this.cHtab[nit][i];
            if (nb >= this.mol.getAtomCount() || nb == ox) continue;
            int t = this.pg.getCarboxy()[nb];
            switch (t) {
                case 2: 
                case 300: {
                    return -1.5;
                }
                case 500: 
                case 600: 
                case 700: 
                case 1200: {
                    return -2.0;
                }
                case 1: 
                case 200: 
                case 800: {
                    return -2.5;
                }
                case 100: 
                case 400: 
                case 1300: {
                    return -3.5;
                }
            }
        }
        return 0.0;
    }

    private double getPureCarbCountInc() {
        double inc = -0.4;
        block10: for (int i = 0; i < this.getSphereSize(); ++i) {
            int a = this.getSphere(i);
            int d = this.pka.dist[a];
            if (this.getAtno(a) != 6 || this.pg.getCarboxy()[a] != -1) continue;
            if (!this.mol.getAtom(a).hasAromaticBond()) {
                switch (d) {
                    case 1: {
                        inc += 0.6;
                        break;
                    }
                    case 2: {
                        inc += 0.7;
                        break;
                    }
                    case 3: 
                    case 4: {
                        inc += 0.3;
                    }
                }
                continue;
            }
            switch (d) {
                case 1: {
                    inc += -1.6;
                    continue block10;
                }
                case 2: {
                    inc += 0.3;
                    continue block10;
                }
                case 3: {
                    inc += 0.2;
                }
            }
        }
        return inc;
    }

    private boolean isAlphaCNegative(int alphaAtom) {
        for (int i = 0; i < this.cHtab[alphaAtom].length; ++i) {
            int nb = this.cHtab[alphaAtom][i];
            if (this.getAtno(nb) != 6 || this.mol.getAtom(nb).getCharge() != -1) continue;
            return true;
        }
        return false;
    }

    private double getBondCharge24(int atom) {
        double sum = this.Qsum[atom];
        for (int j = 0; j < this.cHtab[atom].length; ++j) {
            int nb = this.cHtab[atom][j];
            if (this.getAtno(nb) != 6 || this.cHtab[nb].length != 4) continue;
            sum += this.mol.getAtom(nb).getCharge() == -1 ? 0.25 : this.Qsum[nb];
        }
        return sum;
    }

    protected double getAliphaticBasicpKa(int atom, boolean phosphine) {
        int subBaseType = this.pkaBasicType[atom][1];
        int aat = this.pkaBasicType[atom][3];
        int hx = this.pkaBasicType[atom][5];
        int semiAraType = this.pkaBasicType[atom][7];
        int fusedRing = this.pkaBasicType[atom][9];
        int agDel = this.pkaBasicType[atom][10];
        double pkaVal = Double.NaN;
        if (aat == 1) {
            double inc = 0.0;
            double c = this.getBondCharge2_0(atom);
            double a = 0.72 * this.setAtomSphere(atom, 3, 0.75) - 2.3;
            if (subBaseType == 20 || subBaseType == 25) {
                inc = -26.42 * this.Qpi[atom];
                inc += 3.0 * this.getAlphaAminInc(atom);
                if (phosphine) {
                    pkaVal += -4.5;
                }
            }
            pkaVal = phosphine ? -9.98 * c + a + inc - 5.6 : -29.98 * c + a + inc;
            pkaVal = subBaseType == 44 ? (pkaVal += -2.0) : (pkaVal += this.addsAmineAzaCorrection(atom));
            pkaVal += this.cyanoInc[atom];
            pkaVal += this.addsGammaAmineCorrection(atom);
            if (agDel == 31 || agDel == 32) {
                pkaVal += -3.0;
            }
            if (semiAraType == 100) {
                pkaVal += -2.0;
            }
        } else if (aat == 2) {
            int ind;
            double chg;
            double bc;
            double inc = 0.0;
            if (subBaseType == 20) {
                bc = 0.0;
                if (!phosphine) {
                    for (int i = 0; i < 6; ++i) {
                        int vi = this.getnVinilInfo(atom)[i];
                        if (vi == -1) continue;
                        bc += this.getBondCharge12(vi);
                        bc += -this.Qsum[vi];
                    }
                    inc = -9.25 * bc - 6.22;
                } else {
                    inc = -2.31;
                }
                if (semiAraType == 100) {
                    inc += -1.5;
                }
            } else if (subBaseType == 25) {
                bc = 0.0;
                if (!phosphine) {
                    for (int i = 0; i < 6; ++i) {
                        int vi = this.getnVinilInfo(atom)[i];
                        if (vi == -1) continue;
                        bc += this.getBondCharge12(vi);
                        bc += -this.Qsum[vi];
                    }
                    inc = -7.26 * bc - 8.31;
                } else {
                    inc = -4.31;
                }
                if (semiAraType == 100) {
                    inc += -2.5;
                }
            }
            pkaVal = phosphine ? -41.44 * this.getBondCharge2_0(atom) + 20.98 * this.Qsum[atom] + 0.64 * this.setAtomSphere(atom, 4, 0.75) + 0.03 + inc - 5.5 : -41.44 * this.getBondCharge2_0(atom) + 20.98 * this.Qsum[atom] + 0.64 * this.setAtomSphere(atom, 4, 0.75) + 0.03 + inc;
            if ((semiAraType == 200 || semiAraType == 300) && (chg = this.getAvarageAliphaticRingCharge(ind = this.pkaBasicType[atom][8])) > 0.0) {
                pkaVal += 2.5;
            }
            pkaVal += 0.95 * this.getAlphaAminInc(atom);
            if (semiAraType != 300 && semiAraType != 200) {
                pkaVal = subBaseType == 44 ? (pkaVal += -2.0) : (subBaseType == 444 ? (pkaVal += -5.5) : (pkaVal += this.addsAmineAzaCorrection(atom)));
            }
            pkaVal += this.cyanoInc[atom];
            pkaVal += this.addsGammaAmineCorrection(atom);
            if (agDel == 31 || agDel == 32) {
                pkaVal += -3.0;
            }
        } else if (aat == 3) {
            double inc = this.getBondCharge2_0(atom);
            pkaVal = fusedRing != -1 && this.aza[atom] == -1 && this.cyanoInc[atom] == 0.0 ? -24.36 * inc + 2.38 * this.setAtomSphere(atom, 6, 0.75) / (double)this.getAtomCount() + 0.87 : (phosphine ? -40.67 * inc + 0.96 * this.setAtomSphere(atom, 4, 0.75) - 6.95 : (this.pka.isInRing[atom] ? -40.67 * inc + 0.96 * this.setAtomSphere(atom, 6, 0.75) - 9.95 : -40.67 * inc + 0.96 * this.setAtomSphere(atom, 4, 0.75) - 9.95));
            if (subBaseType == 20 || subBaseType == 25) {
                pkaVal += -15.05 * this.Qpi[atom] + 1.6;
                if (phosphine) {
                    pkaVal += -1.9;
                }
            }
            if (hx != -1) {
                if (this.Qsum[atom] > -0.1) {
                    pkaVal += 2.5 * Math.pow(2.718282, -this.Qsum[atom]);
                } else if (this.Qsum[atom] > -0.16) {
                    pkaVal += 0.0013 * Math.pow(2.718282, -54.497 * this.Qsum[atom]);
                }
            }
            pkaVal += this.getAlphaAminInc(atom);
            if (semiAraType != 300 && semiAraType != 200) {
                pkaVal = subBaseType == 44 ? (pkaVal += -1.0) : (pkaVal += this.addsAmineAzaCorrection(atom));
            }
            pkaVal += this.cyanoInc[atom];
            if (agDel == 31 || agDel == 32) {
                pkaVal += -3.0;
            }
            if (semiAraType == 100 && pkaVal > 4.0) {
                pkaVal += -0.8528 * pkaVal + 3.698;
            }
            pkaVal += this.addsGammaAmineCorrection(atom);
            pkaVal += this.addsBetaAmineCorrection(atom);
        }
        if (pkaVal > 15.0) {
            return 15.0;
        }
        return pkaVal;
    }

    private double addsAmineAzaCorrection(int atom) {
        if (this.aza[atom] != -1) {
            int a = this.aza[atom];
            int t = this.isAzaAza(atom);
            if (t != -1) {
                if (this.pkaBasicType[atom][3] == 3) {
                    if (t == 500) {
                        return 0.0;
                    }
                    return -1.8;
                }
                if (this.pkaBasicType[atom][3] == 2 && t == 500) {
                    return -4.3;
                }
                return -3.3;
            }
            if (this.pkaBasicType[a][0] == 1000 && this.pkaBasicType[a][3] < 4) {
                if (this.pkaBasicType[a][10] == 31 || this.pkaBasicType[a][10] == 32) {
                    return -1.2;
                }
                if (this.isAzaAza(a) != -1) {
                    return -2.5;
                }
                return -2.5;
            }
            if (this.pkaBasicType[a][0] >= 0) {
                if (this.pkaBasicType[a][1] == 11) {
                    return -2.5;
                }
                if (this.mol.getAtom(a).getCharge() != 1) {
                    return -2.0;
                }
                return -1.0;
            }
        }
        return 0.0;
    }

    private int isAzaAza(int atom) {
        for (int i = 0; i < this.cHtab[atom].length; ++i) {
            int z;
            int nb = this.cHtab[atom][i];
            if (nb >= this.mol.getAtomCount() || (z = this.pg.getCarboxy()[nb]) != 200 && z != 600 && z != 700 && z != 800 && z != 500) continue;
            return z;
        }
        return -1;
    }

    private double addsGammaAmineCorrection(int atom) {
        double sum = 0.0;
        if (this.pka.HBT != null && this.pka.HBT[this.pka.hbtAIndex[atom]][23] != -1) {
            this.getAtomDistances(atom, 4);
            for (int i = 0; i < this.getSphereSize(); ++i) {
                int a = this.getSphere(i);
                if (this.pka.dist[a] == 4 && (this.pkaBasicType[a][0] == 1000 || this.pkaBasicType[a][0] == 5) && this.mol.getAtom(a).getCharge() == 1) {
                    sum -= 1.0;
                    continue;
                }
                if (this.pka.dist[a] != 3 || this.pkaBasicType[a][0] != 1000 && this.pkaBasicType[a][0] != 5) continue;
                if (this.mol.getAtom(a).getCharge() == 1) {
                    sum -= 0.6;
                    continue;
                }
                sum -= 0.4;
            }
        }
        return sum;
    }

    private double addsBetaAmineCorrection(int atom) {
        double sum = 0.0;
        if (this.pka.HBT != null && this.pka.HBT[this.pka.hbtAIndex[atom]][27] != -1) {
            this.getAtomDistances(atom, 3);
            for (int i = 0; i < this.getSphereSize(); ++i) {
                int a = this.getSphere(i);
                if (this.pka.dist[a] != 3 || this.pkaBasicType[a][0] != 1000 || this.mol.getAtom(a).getCharge() != 1) continue;
                sum += -1.0;
            }
        }
        return sum;
    }

    protected boolean isValid() {
        if (this.pka == null || this.pka.mol != this.mol) {
            return false;
        }
        if (this.Qsum == null) {
            return this.pka.Qsum == null;
        }
        if (this.pka.Qsum == null) {
            return false;
        }
        for (int i = 0; i < this.Qsum.length; ++i) {
            if (this.pka.Qsum[i] == this.Qsum[i]) continue;
            return false;
        }
        return true;
    }

    private double getBondCharge2(int atom) {
        return this.pka.getBondCharge2(atom);
    }

    private double getBondCharge6(int atom) {
        return this.pka.getBondCharge6(atom);
    }

    private double getBondCharge12(int atom) {
        return this.pka.getBondCharge12(atom);
    }

    private int getAtno(int index) {
        return this.pka.getAtno(index);
    }

    private double setAtomSphere(int atom, int shellCount, double dFact) {
        return this.pka.setAtomSphere(atom, shellCount, dFact);
    }

    private int getAtomCount() {
        return this.pka.atomCount;
    }

    private void getDelIsland(int atom, int deep) {
        this.pka.getDelIsland(atom, deep);
    }

    private int getSphereSize() {
        return this.pka.sphereSize;
    }

    private int getSphere(int index) {
        return this.pka.sphere[index];
    }

    private void getAtomDistances(int atom, int MAXSPHERE) {
        this.pka.getAtomDistances(atom, MAXSPHERE);
    }

    private double getBondCharge2_0(int atom) {
        return this.pka.getBondCharge2_0(atom);
    }

    private int[] getnVinilInfo(int atom) {
        return this.pka.getnVinilInfo(atom);
    }
}

