/*
 * Decompiled with CFR 0.152.
 */
package chemaxon.marvin.space.grid;

import chemaxon.common.util.GeomCalc;
import chemaxon.marvin.space.ProgressBarInterface;
import chemaxon.marvin.space.SurfaceComponent;
import chemaxon.marvin.space.grid.Grid;
import chemaxon.marvin.space.grid.TriangleTable;
import java.util.Arrays;

public class MarchingCubes {
    private static final float TOLERANCE = 5.0E-4f;
    private Grid grid;
    private float threshold = 0.02f;
    private IndexCache ic;
    private int[] nn;
    private int[] vertices;
    private int[] normals;
    private int[] triangles;
    private int vertexPointer = 0;
    private int normalPointer = 0;
    private int trianglePointer = 0;
    private int triangleCount = 0;
    private int[][] neighbours;
    private int[][] neighbourTriangles;
    private int smoothnessFactor = 3;
    private boolean simplification = true;
    private boolean correctionNecessary = false;
    private boolean[] dv;
    private ProgressBarInterface progressBar = null;
    private int startProgress;
    private int progressLength;
    private int pi;
    private int p;
    private int pstart;
    private int pstep;
    private int plength;
    private static float[][] edgeTable = new float[][]{{0.5f, 0.0f, 0.0f}, {1.0f, 0.5f, 0.0f}, {0.5f, 1.0f, 0.0f}, {0.0f, 0.5f, 0.0f}, {0.0f, 0.0f, 0.5f}, {1.0f, 0.0f, 0.5f}, {1.0f, 1.0f, 0.5f}, {0.0f, 1.0f, 0.5f}, {0.5f, 0.0f, 1.0f}, {1.0f, 0.5f, 1.0f}, {0.5f, 1.0f, 1.0f}, {0.0f, 0.5f, 1.0f}};

    public MarchingCubes(Grid grid) {
        this.grid = grid;
        this.ic = new IndexCache(grid.sizeY, grid.sizeX);
    }

    public MarchingCubes(Grid grid, float threshold) {
        this.grid = grid;
        this.threshold = threshold;
        this.ic = new IndexCache(grid.sizeY, grid.sizeX);
    }

    public MarchingCubes(SurfaceComponent surface) {
        this.vertices = new int[surface.getVertexCount()];
        this.normals = new int[surface.getVertexCount()];
        for (int i = 0; i < surface.getVertexCount(); ++i) {
            this.vertices[this.vertexPointer++] = GeomCalc.newVector(surface.getVertexX(i), surface.getVertexY(i), surface.getVertexZ(i));
            this.normals[this.normalPointer++] = surface.getNormal(i);
        }
        this.triangles = new int[surface.getPrimitives().length];
        System.arraycopy(surface.getPrimitives(), 0, this.triangles, 0, this.triangles.length);
        surface.getPrimitives();
        this.triangleCount = this.triangles.length / 3;
    }

    protected void finalize() throws Throwable {
        this.vertices = null;
        this.normals = null;
        this.triangles = null;
        this.neighbours = null;
        this.neighbourTriangles = null;
        this.nn = null;
        this.ic = null;
        this.grid = null;
    }

    public void setProgressBar(ProgressBarInterface progressBar, int p1, int p2) {
        this.progressBar = progressBar;
        this.startProgress = p1;
        this.progressLength = p2 - this.startProgress;
    }

    public void setThreshold(float th) {
        this.threshold = th;
    }

    public void setSmoothnessFactor(int factor) {
        this.smoothnessFactor = factor;
    }

    public void setSimplification(boolean b) {
        this.simplification = b;
    }

    public SurfaceComponent generate() {
        this.correctionNecessary = this.correctionNecessary();
        if (this.correctionNecessary) {
            this.correctGrid();
        }
        int vertexCount = this.vertexCount();
        this.triangles = new int[vertexCount * 3];
        if (vertexCount > 500) {
            vertexCount = (int)((double)vertexCount * 0.6);
        }
        this.nn = new int[vertexCount];
        this.vertices = new int[vertexCount];
        this.normals = new int[vertexCount];
        this.buildSurface();
        for (int i = 0; i < this.vertexPointer; ++i) {
            GeomCalc.scale(this.normals[i], this.nn[i]);
        }
        if (this.simplification) {
            if (this.progressBar != null) {
                this.progressStart(this.progressLength / 6, 1);
            }
            float tol = Math.min(Math.min(this.grid.stepX, this.grid.stepY), this.grid.stepZ);
            this.buildNeighbourTriangles();
            this.simplification(tol);
            if (this.vertexPointer > 25000) {
                this.buildNeighbourTriangles();
                this.simplification(tol);
            }
            if (this.progressBar != null) {
                this.progressEnd();
            }
        }
        return this.upload(this.simplification ? this.progressLength / 6 : this.progressLength / 4);
    }

    public SurfaceComponent reduceTriangleCount(float tol) {
        if (this.triangleCount < 10) {
            return this.upload(this.progressLength / 4);
        }
        if (this.progressBar != null) {
            this.pstart = this.startProgress;
            this.progressStart(this.progressLength / 4, 1);
        }
        this.buildNeighBours();
        if (this.progressBar != null) {
            this.progressEnd();
            this.progressStart(this.progressLength / 4, 1);
        }
        this.buildNeighbourTriangles();
        if (this.progressBar != null) {
            this.progressEnd();
            this.progressStart(this.progressLength / 4, 1);
        }
        this.simplification(tol);
        if (this.vertexPointer > 25000) {
            this.buildNeighbourTriangles();
            this.simplification(tol);
        }
        return this.upload(this.progressLength / 4);
    }

    private boolean correctionNecessary() {
        int j;
        int i;
        for (i = 0; i < this.grid.getSizeX(); ++i) {
            for (j = 0; j < this.grid.getSizeY(); ++j) {
                if (this.grid.get(i, j, 0) != 0.0f) {
                    return true;
                }
                if (this.grid.get(i, j, this.grid.getSizeZ() - 1) == 0.0f) continue;
                return true;
            }
        }
        for (i = 0; i < this.grid.getSizeY(); ++i) {
            for (j = 0; j < this.grid.getSizeZ(); ++j) {
                if (this.grid.get(0, i, j) != 0.0f) {
                    return true;
                }
                if (this.grid.get(this.grid.getSizeX() - 1, i, j) == 0.0f) continue;
                return true;
            }
        }
        for (i = 0; i < this.grid.getSizeX(); ++i) {
            for (j = 0; j < this.grid.getSizeZ(); ++j) {
                if (this.grid.get(i, 0, j) != 0.0f) {
                    return true;
                }
                if (this.grid.get(i, this.grid.getSizeY() - 1, j) == 0.0f) continue;
                return true;
            }
        }
        return false;
    }

    private void correctGrid() {
        int j;
        int i;
        for (i = 0; i < this.grid.getSizeX(); ++i) {
            for (j = 0; j < this.grid.getSizeY(); ++j) {
                this.grid.set(i, j, 0, 0.0f);
                this.grid.set(i, j, this.grid.getSizeZ() - 1, 0.0f);
            }
        }
        for (i = 0; i < this.grid.getSizeY(); ++i) {
            for (j = 0; j < this.grid.getSizeZ(); ++j) {
                this.grid.set(0, i, j, 0.0f);
                this.grid.set(this.grid.getSizeX() - 1, i, j, 0.0f);
            }
        }
        for (i = 0; i < this.grid.getSizeX(); ++i) {
            for (j = 0; j < this.grid.getSizeZ(); ++j) {
                this.grid.set(i, 0, j, 0.0f);
                this.grid.set(i, this.grid.getSizeY() - 1, j, 0.0f);
            }
        }
    }

    private void correctSurface() {
        this.buildNeighbourTriangles();
        this.deleteVerticesInBoundaryRange();
        boolean[] dt = this.deleteUnnecessaryTriangles();
        this.restoreContinuity(dt);
    }

    private void deleteVerticesInBoundaryRange() {
        float minx = this.grid.getStartX();
        float miny = this.grid.getStartY();
        float minz = this.grid.getStartZ();
        float maxx = this.grid.getStartX() + (float)this.grid.getSizeX() * this.grid.getStepX();
        float maxy = this.grid.getStartY() + (float)this.grid.getSizeY() * this.grid.getStepY();
        float maxz = this.grid.getStartZ() + (float)this.grid.getSizeZ() * this.grid.getStepZ();
        float ex = 1.5f * this.grid.getStepX();
        float ey = 1.5f * this.grid.getStepY();
        float ez = 1.5f * this.grid.getStepZ();
        this.dv = new boolean[this.vertexPointer];
        for (int i = 0; i < this.vertexPointer; ++i) {
            if (GeomCalc.getX(this.vertices[i]) - minx <= ex) {
                this.dv[i] = true;
                continue;
            }
            if (maxx - GeomCalc.getX(this.vertices[i]) <= ex) {
                this.dv[i] = true;
                continue;
            }
            if (GeomCalc.getY(this.vertices[i]) - miny <= ey) {
                this.dv[i] = true;
                continue;
            }
            if (maxy - GeomCalc.getY(this.vertices[i]) <= ey) {
                this.dv[i] = true;
                continue;
            }
            if (GeomCalc.getZ(this.vertices[i]) - minz <= ez) {
                this.dv[i] = true;
                continue;
            }
            if (!(maxz - GeomCalc.getZ(this.vertices[i]) <= ez)) continue;
            this.dv[i] = true;
        }
    }

    private boolean[] deleteUnnecessaryTriangles() {
        System.out.println("Nbt.length: " + this.neighbourTriangles.length);
        System.out.println("VertexPointer: " + this.vertexPointer);
        boolean[] isTriangleDeleted = new boolean[this.triangleCount];
        for (int i = 0; i < this.vertexPointer; ++i) {
            if (!this.dv[i]) continue;
            int j = 0;
            while (this.neighbourTriangles[i][j] >= 0) {
                isTriangleDeleted[this.neighbourTriangles[i][j]] = true;
                ++j;
            }
        }
        return isTriangleDeleted;
    }

    private void progressStart(int length, int step) {
        this.pi = 0;
        this.p = 0;
        this.plength = length;
        this.pstep = step;
    }

    private void progressStep() {
        if (this.p >= this.pstep && this.pi < this.plength) {
            this.progressBar.setProgress(this.pstart + this.pi);
            this.p = 0;
            ++this.pi;
        }
        ++this.p;
    }

    private void progressEnd() {
        this.progressBar.setProgress(this.pstart + this.plength);
        this.pstart += this.plength;
    }

    private void buildSurface() {
        if (this.progressBar != null) {
            this.pstart = this.startProgress;
            int length = this.progressLength / 2;
            int progressStep = Math.round((float)(this.grid.getSizeZ() - 1) / (float)length);
            this.progressStart(length, progressStep);
        }
        int point = GeomCalc.newVector();
        for (int z = 0; z < this.grid.getSizeZ() - 1; ++z) {
            for (int y = 0; y < this.grid.getSizeY() - 1; ++y) {
                for (int x = 0; x < this.grid.getSizeX() - 1; ++x) {
                    int code = this.readCube(z, y, x);
                    if (code != 0 && code != 255) {
                        GeomCalc.setVector(point, x, y, z);
                        this.generateTriangles(y, x, point, code);
                    }
                    this.ic.stepCell();
                }
                this.ic.stepRow();
            }
            this.ic.stepMatrix();
            if (this.progressBar == null) continue;
            this.progressStep();
        }
        if (this.progressBar != null) {
            this.progressEnd();
            int length = this.simplification ? this.progressLength / 6 : this.progressLength / 4;
            int progressStep = this.smoothnessFactor / length;
            this.progressStart(length, progressStep);
        }
        GeomCalc.deleteVector(point);
        this.buildNeighBours();
        for (int i = 0; i < this.smoothnessFactor; ++i) {
            this.smoothSurface();
            if (this.progressBar == null) continue;
            this.progressStep();
        }
        if (this.progressBar != null) {
            this.progressEnd();
        }
    }

    private SurfaceComponent upload(int length) {
        int i;
        SurfaceComponent surface = new SurfaceComponent(this.vertexPointer, this.triangleCount);
        if (this.progressBar != null) {
            int progressStep = (this.vertexPointer + this.triangleCount) / length;
            this.progressStart(length, progressStep);
        }
        for (i = 0; i < this.vertexPointer; ++i) {
            surface.putVertex(GeomCalc.getFloatVector(this.vertices[i]));
            surface.putNormal(GeomCalc.getFloatVector(this.normals[i]));
            GeomCalc.deleteVector(this.vertices[i]);
            GeomCalc.deleteVector(this.normals[i]);
            if (this.progressBar == null) continue;
            this.progressStep();
        }
        for (i = 0; i < this.triangleCount; ++i) {
            surface.putTriangle(this.triangles[i * 3], this.triangles[i * 3 + 1], this.triangles[i * 3 + 2]);
            if (this.progressBar == null) continue;
            this.progressStep();
        }
        if (this.progressBar != null) {
            this.progressEnd();
        }
        return surface;
    }

    private void buildNeighBours() {
        int i;
        this.neighbours = new int[this.vertexPointer][16];
        for (i = 0; i < this.vertexPointer; ++i) {
            Arrays.fill(this.neighbours[i], -1);
        }
        for (i = 0; i < this.triangleCount; ++i) {
            this.putNeighbour(this.triangles[i * 3], this.triangles[i * 3 + 1]);
            this.putNeighbour(this.triangles[i * 3], this.triangles[i * 3 + 2]);
            this.putNeighbour(this.triangles[i * 3 + 1], this.triangles[i * 3]);
            this.putNeighbour(this.triangles[i * 3 + 1], this.triangles[i * 3 + 2]);
            this.putNeighbour(this.triangles[i * 3 + 2], this.triangles[i * 3]);
            this.putNeighbour(this.triangles[i * 3 + 2], this.triangles[i * 3 + 1]);
        }
    }

    private void buildNeighbourTriangles() {
        int i;
        this.neighbourTriangles = new int[this.vertexPointer][16];
        for (i = 0; i < this.vertexPointer; ++i) {
            Arrays.fill(this.neighbourTriangles[i], -1);
        }
        for (i = 0; i < this.triangleCount; ++i) {
            int a = this.triangles[i * 3];
            int b = this.triangles[i * 3 + 1];
            int c = this.triangles[i * 3 + 2];
            this.putNeighbourTriangle(a, i);
            this.putNeighbourTriangle(b, i);
            this.putNeighbourTriangle(c, i);
        }
    }

    private int putNeighbourTriangle(int t, int v) {
        int i;
        if (this.neighbourTriangles.length <= t) {
            System.err.println("neighbourTriangles array is small, increasing size with 512 spaces.");
            int[][] newNBT = new int[this.neighbourTriangles.length + 512][];
            for (i = 0; i < this.neighbourTriangles.length; ++i) {
                newNBT[i] = this.neighbourTriangles[i];
            }
            for (i = this.neighbourTriangles.length; i < newNBT.length; ++i) {
                newNBT[i] = new int[16];
                Arrays.fill(newNBT[i], -1);
            }
            this.neighbourTriangles = newNBT;
        }
        for (i = 0; i < 16; ++i) {
            if (this.neighbourTriangles[t][i] == v) {
                return -1;
            }
            if (this.neighbourTriangles[t][i] != -1) continue;
            this.neighbourTriangles[t][i] = v;
            return i;
        }
        System.err.println("Too few spaces for neighbourTriangles.");
        return -1;
    }

    private int putNeighbour(int v, int n) {
        if (this.neighbours.length <= v) {
            return -1;
        }
        for (int i = 0; i < 16; ++i) {
            if (this.neighbours[v][i] == n) {
                return -1;
            }
            if (this.neighbours[v][i] != -1) continue;
            this.neighbours[v][i] = n;
            return i;
        }
        System.err.println("Too few spaces for neighbours.");
        return -1;
    }

    private void smoothSurface() {
        int pj = GeomCalc.newVector();
        int nj = GeomCalc.newVector();
        for (int i = 0; i < this.vertexPointer; ++i) {
            int j;
            GeomCalc.setNullVector(pj);
            GeomCalc.setNullVector(nj);
            int p = this.vertices[i];
            int n = this.normals[i];
            for (j = 0; j < 16 && this.neighbours[i][j] != -1; ++j) {
                int dp = GeomCalc.sub(this.vertices[this.neighbours[i][j]], p);
                int dn = GeomCalc.sub(this.normals[this.neighbours[i][j]], n);
                GeomCalc.increase(pj, dp);
                GeomCalc.increase(nj, dn);
                GeomCalc.deleteVector(dp);
                GeomCalc.deleteVector(dn);
            }
            if (j > 0) {
                GeomCalc.scale(pj, 1.0f / (float)j);
                GeomCalc.scale(nj, 1.0f / (float)j);
            }
            GeomCalc.increase(nj, n);
            GeomCalc.normalize(nj);
            GeomCalc.increase(pj, p);
            GeomCalc.setVector(p, pj);
            GeomCalc.setVector(n, nj);
        }
        GeomCalc.deleteVector(pj);
        GeomCalc.deleteVector(nj);
    }

    private int readCube(int z, int y, int x) {
        int ret = 0;
        if (this.threshold < 0.0f) {
            ret += this.grid.grid[z][y][x] < this.threshold ? 1 : 0;
            ret += this.grid.grid[z][y][x + 1] < this.threshold ? 2 : 0;
            ret += this.grid.grid[z][y + 1][x + 1] < this.threshold ? 4 : 0;
            ret += this.grid.grid[z][y + 1][x] < this.threshold ? 8 : 0;
            ret += this.grid.grid[z + 1][y][x] < this.threshold ? 16 : 0;
            ret += this.grid.grid[z + 1][y][x + 1] < this.threshold ? 32 : 0;
            ret += this.grid.grid[z + 1][y + 1][x + 1] < this.threshold ? 64 : 0;
            ret += this.grid.grid[z + 1][y + 1][x] < this.threshold ? 128 : 0;
        } else {
            ret += this.grid.grid[z][y][x] > this.threshold ? 1 : 0;
            ret += this.grid.grid[z][y][x + 1] > this.threshold ? 2 : 0;
            ret += this.grid.grid[z][y + 1][x + 1] > this.threshold ? 4 : 0;
            ret += this.grid.grid[z][y + 1][x] > this.threshold ? 8 : 0;
            ret += this.grid.grid[z + 1][y][x] > this.threshold ? 16 : 0;
            ret += this.grid.grid[z + 1][y][x + 1] > this.threshold ? 32 : 0;
            ret += this.grid.grid[z + 1][y + 1][x + 1] > this.threshold ? 64 : 0;
            ret += this.grid.grid[z + 1][y + 1][x] > this.threshold ? 128 : 0;
        }
        return ret;
    }

    private int get(int z, int y, int x) {
        return this.grid.get(x, y, z) < this.threshold ? 1 : 0;
    }

    private int computeVertex(int point, int code) {
        int wp = GeomCalc.newVector();
        GeomCalc.setVector(wp, edgeTable[code]);
        this.refineWeightpoint(point, wp);
        GeomCalc.increase(wp, point);
        GeomCalc.setVectorX(wp, GeomCalc.getX(wp) * this.grid.stepX + this.grid.startX);
        GeomCalc.setVectorY(wp, GeomCalc.getY(wp) * this.grid.stepY + this.grid.startY);
        GeomCalc.setVectorZ(wp, GeomCalc.getZ(wp) * this.grid.stepZ + this.grid.startZ);
        this.vertices[this.vertexPointer++] = wp;
        return this.vertexPointer - 1;
    }

    private void refineWeightpoint(int p, int wp) {
        float l = 0.0f;
        float h = 0.0f;
        int z = (int)(GeomCalc.getZ(p) + (float)((int)GeomCalc.getZ(wp)));
        int y = (int)(GeomCalc.getY(p) + (float)((int)GeomCalc.getY(wp)));
        int x = (int)(GeomCalc.getX(p) + (float)((int)GeomCalc.getX(wp)));
        if ((double)GeomCalc.getZ(wp) == 0.5) {
            int pz = (int)GeomCalc.getZ(p);
            h = this.grid.grid[pz + 1][y][x];
            if (h != (l = this.grid.grid[pz][y][x])) {
                GeomCalc.setVectorZ(wp, (this.threshold - l) / (h - l));
            }
            this.correctRefinement(wp, 2);
        } else if ((double)GeomCalc.getY(wp) == 0.5) {
            int py = (int)GeomCalc.getY(p);
            h = this.grid.grid[z][py + 1][x];
            if (h != (l = this.grid.grid[z][py][x])) {
                GeomCalc.setVectorY(wp, (this.threshold - l) / (h - l));
            }
            this.correctRefinement(wp, 1);
        } else {
            int px = (int)GeomCalc.getX(p);
            h = this.grid.grid[z][y][px + 1];
            if (h != (l = this.grid.grid[z][y][px])) {
                GeomCalc.setVectorX(wp, (this.threshold - l) / (h - l));
            }
            this.correctRefinement(wp, 0);
        }
    }

    private void correctRefinement(int wp, int i) {
        if (GeomCalc.get(wp, i) == 0.0f) {
            GeomCalc.setVectorCoordinate(wp, i, 5.0E-4f);
        } else if (GeomCalc.get(wp, i) == 1.0f) {
            GeomCalc.setVectorCoordinate(wp, i, 0.9995f);
        }
    }

    private void generateTriangles(int y, int x, int point, int code) {
        int i = 0;
        boolean b1 = false;
        boolean b2 = false;
        boolean b3 = false;
        int d = GeomCalc.newVector();
        int e = GeomCalc.newVector();
        while (TriangleTable.triTable[code][i] >= 0) {
            int i3;
            int i2;
            int c = TriangleTable.triTable[code][i];
            int i1 = this.ic.needComputing(y, x, c);
            if (i1 < 0) {
                b1 = true;
                i1 = this.computeVertex(point, c);
                this.ic.setNewValues(y, x, c, i1);
            }
            if ((i2 = this.ic.needComputing(y, x, c = TriangleTable.triTable[code][i + 1])) < 0) {
                b2 = true;
                i2 = this.computeVertex(point, c);
                this.ic.setNewValues(y, x, c, i2);
            }
            if ((i3 = this.ic.needComputing(y, x, c = TriangleTable.triTable[code][i + 2])) < 0) {
                b3 = true;
                i3 = this.computeVertex(point, c);
                this.ic.setNewValues(y, x, c, i3);
            }
            GeomCalc.setVector(d, this.vertices[i3]);
            GeomCalc.decrease(d, this.vertices[i1]);
            GeomCalc.setVector(e, this.vertices[i2]);
            GeomCalc.decrease(e, this.vertices[i1]);
            int n = GeomCalc.cross(e, d);
            GeomCalc.normalize(n);
            this.setNormal(b1, i1, n);
            b1 = false;
            this.setNormal(b2, i2, n);
            b2 = false;
            this.setNormal(b3, i3, n);
            b3 = false;
            this.triangles[this.trianglePointer++] = i1;
            this.triangles[this.trianglePointer++] = i2;
            this.triangles[this.trianglePointer++] = i3;
            ++this.triangleCount;
            i += 3;
            GeomCalc.deleteVector(n);
        }
        GeomCalc.deleteVector(d);
        GeomCalc.deleteVector(e);
    }

    private void setNormal(boolean b, int i, int n) {
        if (b) {
            this.normals[this.normalPointer] = GeomCalc.newVector();
            GeomCalc.setVector(this.normals[this.normalPointer++], n);
            this.nn[i] = 1;
        } else {
            int normal = this.normals[i];
            GeomCalc.increase(normal, n);
            int n2 = i;
            this.nn[n2] = this.nn[n2] + 1;
        }
    }

    private int vertexCount() {
        int vertexCount = 0;
        for (int z = 0; z < this.grid.getSizeZ() - 1; ++z) {
            for (int y = 0; y < this.grid.getSizeY() - 1; ++y) {
                for (int x = 0; x < this.grid.getSizeX() - 1; ++x) {
                    int code = this.readCube(z, y, x);
                    if (code == 0 || code == 255) continue;
                    int i = 0;
                    while (TriangleTable.triTable[code][i] >= 0) {
                        vertexCount += 3;
                        i += 3;
                    }
                }
            }
        }
        return vertexCount / 3;
    }

    private boolean isTriangleEdge(int t, int e1, int e2) {
        return e1 != e2 && this.isTriangleVertex(t, e1) && this.isTriangleVertex(t, e2);
    }

    private boolean isTriangleVertex(int t, int v) {
        return this.triangles[t * 3] == v || this.triangles[t * 3 + 1] == v || this.triangles[t * 3 + 2] == v;
    }

    private void exchangeTriangleVertex(int t, int v, int nv) {
        if (this.triangles[t * 3] == v) {
            this.triangles[t * 3] = nv;
        } else if (this.triangles[t * 3 + 1] == v) {
            this.triangles[t * 3 + 1] = nv;
        } else if (this.triangles[t * 3 + 2] == v) {
            this.triangles[t * 3 + 2] = nv;
        } else {
            System.out.println("exchange problem: triindex: " + t + "= [" + this.triangles[t * 3] + ", " + this.triangles[t * 3 + 1] + ", " + this.triangles[t * 3 + 2] + "] Oldindex: " + v + " Newindex: " + nv);
        }
    }

    private void eraseFromArray(int[] v, int i, int length) {
        for (int j = i; j < length - 1; ++j) {
            v[j] = v[j + 1];
        }
        v[j] = -1;
    }

    private int find(int[] n, int length, int e) {
        for (int i = 0; i < length; ++i) {
            if (n[i] != e) continue;
            return i;
        }
        return -1;
    }

    private void putVertex(int v) {
        if (this.vertices.length == this.vertexPointer) {
            int[] newV = new int[this.vertices.length + 512];
            System.arraycopy(this.vertices, 0, newV, 0, this.vertices.length);
            this.vertices = newV;
        }
        this.vertices[this.vertexPointer++] = v;
    }

    private void putVertex(int v, int n) {
        if (this.vertices.length == this.vertexPointer) {
            int[] newV = new int[this.vertices.length + 512];
            int[] newN = new int[this.normals.length + 512];
            System.arraycopy(this.vertices, 0, newV, 0, this.vertices.length);
            System.arraycopy(this.normals, 0, newN, 0, this.normals.length);
            this.vertices = newV;
            this.normals = newN;
        }
        this.vertices[this.vertexPointer++] = v;
        this.normals[this.normalPointer++] = n;
    }

    private void increaseVertexDeletedArray() {
        boolean[] newDV = new boolean[this.dv.length + 512];
        System.arraycopy(this.dv, 0, newDV, 0, this.dv.length);
        this.dv = newDV;
    }

    private int[] getNeighbours(int v) {
        if (this.neighbours.length <= v) {
            int[][] newNB = new int[this.neighbours.length + 512][];
            System.arraycopy(this.neighbours, 0, newNB, 0, this.neighbours.length);
            for (int i = this.neighbours.length; i < newNB.length; ++i) {
                newNB[i] = new int[16];
                Arrays.fill(newNB[i], -1);
            }
            this.neighbours = newNB;
        }
        return this.neighbours[v];
    }

    private int[] getNeighbourTriangles(int v) {
        if (this.neighbourTriangles.length <= v) {
            int[][] newNBT = new int[this.neighbourTriangles.length + 512][];
            System.arraycopy(this.neighbourTriangles, 0, newNBT, 0, this.neighbourTriangles.length);
            for (int i = this.neighbourTriangles.length; i < newNBT.length; ++i) {
                newNBT[i] = new int[16];
                Arrays.fill(newNBT[i], -1);
            }
            this.neighbourTriangles = newNBT;
        }
        return this.neighbourTriangles[v];
    }

    private int getLength(int[] n) {
        int i;
        for (i = 0; i < n.length; ++i) {
            if (n[i] != -1) continue;
            return i;
        }
        return i;
    }

    private void edgeDecimation(boolean[] dt, int e1, int e2) {
        int nnlength;
        int[] nnbs;
        int vi;
        int i;
        this.dv[e1] = true;
        this.dv[e2] = true;
        int newvertex = GeomCalc.half(this.vertices[e1], this.vertices[e2]);
        int newnormal = GeomCalc.half(this.normals[e1], this.normals[e2]);
        this.putVertex(newvertex, newnormal);
        int newindex = this.vertexPointer - 1;
        if (this.dv.length <= newindex) {
            this.increaseVertexDeletedArray();
        }
        int[] e1n = this.neighbourTriangles[e1];
        int[] e2n = this.neighbourTriangles[e2];
        int[] nbt = this.getNeighbourTriangles(newindex);
        int nbtPointer = 0;
        for (i = 0; i < e1n.length && e1n[i] != -1; ++i) {
            if (dt[e1n[i]]) continue;
            if (this.isTriangleEdge(e1n[i], e1, e2)) {
                --this.triangleCount;
                dt[e1n[i]] = true;
                continue;
            }
            if (this.isTriangleVertex(e1n[i], e1)) {
                this.exchangeTriangleVertex(e1n[i], e1, newindex);
                nbt[nbtPointer++] = e1n[i];
                continue;
            }
            if (!this.isTriangleVertex(e1n[i], e2)) continue;
            this.exchangeTriangleVertex(e1n[i], e2, newindex);
            nbt[nbtPointer++] = e1n[i];
        }
        for (i = 0; i < e2n.length && e2n[i] != -1; ++i) {
            if (dt[e2n[i]] || this.isTriangleEdge(e2n[i], e1, e2)) continue;
            if (this.isTriangleVertex(e2n[i], e1)) {
                this.exchangeTriangleVertex(e2n[i], e1, newindex);
                nbt[nbtPointer++] = e2n[i];
                continue;
            }
            if (!this.isTriangleVertex(e2n[i], e2)) continue;
            this.exchangeTriangleVertex(e2n[i], e2, newindex);
            nbt[nbtPointer++] = e2n[i];
        }
        int[] nbs = this.getNeighbours(newindex);
        int nbsPointer = 0;
        int[] n1 = this.neighbours[e1];
        int[] n2 = this.neighbours[e2];
        for (vi = 0; vi < n1.length && n1[vi] != -1; ++vi) {
            if (n1[vi] == e2) continue;
            nbs[nbsPointer++] = n1[vi];
            nnbs = this.neighbours[n1[vi]];
            nnlength = this.getLength(nnbs);
            for (int vj = 0; vj < nnlength; ++vj) {
                if (nnbs[vj] != e1) continue;
                this.eraseFromArray(nnbs, vj, nnlength);
                --nnlength;
                --vj;
            }
            nnbs[nnlength++] = newindex;
            int[] nbn = this.neighbourTriangles[n1[vi]];
            int nbnlength = this.getLength(nbn);
            for (int vj = 0; vj < nbnlength && nbn[vj] != -1; ++vj) {
                if (!this.isTriangleEdge(nbn[vj], e1, e2)) continue;
                this.eraseFromArray(nbn, vj, nbnlength);
                --nbnlength;
                --vj;
            }
        }
        for (vi = 0; vi < n2.length && n2[vi] != -1; ++vi) {
            if (n2[vi] != e1 && this.find(nbs, nbsPointer, n2[vi]) == -1) {
                nbs[nbsPointer++] = n2[vi];
            }
            nnbs = this.getNeighbours(n2[vi]);
            nnlength = this.getLength(nnbs);
            boolean isIn = false;
            for (int vj = 0; vj < nnlength; ++vj) {
                if (nnbs[vj] == e2) {
                    this.eraseFromArray(nnbs, vj, nnlength);
                    --nnlength;
                    --vj;
                    continue;
                }
                if (nnbs[vj] != newindex) continue;
                isIn = true;
            }
            if (!isIn) {
                nnbs[nnlength++] = newindex;
            }
            int[] nbn = this.neighbourTriangles[n2[vi]];
            int nbnlength = this.getLength(nbn);
            for (int vj = 0; vj < nbnlength && nbn[vj] != -1; ++vj) {
                if (!this.isTriangleEdge(nbn[vj], e1, e2)) continue;
                this.eraseFromArray(nbn, vj, nbnlength);
                --nbnlength;
                --vj;
            }
        }
    }

    private void moveValues(int from, int where) {
        GeomCalc.deleteVector(this.vertices[where]);
        this.vertices[where] = this.vertices[from];
        this.vertices[from] = -1;
        GeomCalc.deleteVector(this.normals[where]);
        this.normals[where] = this.normals[from];
        this.normals[from] = -1;
        System.arraycopy(this.neighbours[from], 0, this.neighbours[where], 0, 16);
        int[] trv = this.neighbourTriangles[from];
        for (int i = 0; i < trv.length; ++i) {
            if (trv[i] < 0) continue;
            this.exchangeTriangleVertex(trv[i], from, where);
        }
        System.arraycopy(this.neighbourTriangles[from], 0, this.neighbourTriangles[where], 0, 16);
    }

    private void restoreContinuity(boolean[] dt) {
        int what;
        int where = 0;
        int dvsize = this.vertexPointer;
        int dtsize = dt.length;
        while (true) {
            if (where < dvsize && !this.dv[where]) {
                ++where;
                continue;
            }
            for (what = dvsize - 1; what > where && (this.vertices[what] <= 0 || this.dv[what]); --what) {
            }
            if (what == where) break;
            this.moveValues(what, where);
            this.dv[what] = true;
            ++where;
        }
        if (where < dvsize - 1 && this.dv[where]) {
            this.vertexPointer = where;
            this.normalPointer = where;
        }
        where = 0;
        while (true) {
            if (where < dtsize && !dt[where]) {
                ++where;
                continue;
            }
            for (what = dtsize - 1; what > where && dt[what]; --what) {
            }
            if (what == where) break;
            this.triangles[where * 3] = this.triangles[what * 3];
            this.triangles[where * 3 + 1] = this.triangles[what * 3 + 1];
            this.triangles[where * 3 + 2] = this.triangles[what * 3 + 2];
            dt[what] = true;
            dt[where] = false;
            ++where;
        }
        if (where < dtsize - 1 && dt[where]) {
            this.triangleCount = where;
        }
        this.buildNeighBours();
    }

    private void simplification(double tolerance) {
        this.dv = new boolean[this.vertexPointer];
        boolean[] isTriangleDeleted = new boolean[this.triangleCount];
        int cv = this.vertexPointer;
        block0: for (int i = 0; i < cv; ++i) {
            if (this.dv[i]) continue;
            int p1 = this.vertices[i];
            int[] v = this.neighbours[i];
            for (int j = 0; j < v.length && v[j] != -1; ++j) {
                int p2;
                double d;
                if (this.dv[v[j]] || !((d = GeomCalc.distance(p1, p2 = this.vertices[v[j]])) < tolerance)) continue;
                this.edgeDecimation(isTriangleDeleted, i, v[j]);
                continue block0;
            }
        }
        if (this.vertexPointer > cv) {
            this.restoreContinuity(isTriangleDeleted);
        }
    }

    private void checkSurface() {
        int i;
        for (i = 0; i < this.vertexPointer; ++i) {
            if (this.vertices[i] <= 0) {
                System.out.println("Illegal index in vertices " + i);
            }
            if (GeomCalc.getVector(this.vertices[i]) == null) {
                System.out.println("Illegal vector in vertices " + i);
            }
            if (this.normals[i] <= 0) {
                System.out.println("Illegal index in normals " + i);
            }
            if (GeomCalc.getVector(this.normals[i]) != null) continue;
            System.out.println("Illegal normal in vertices " + i);
        }
        for (i = 0; i < this.triangleCount; ++i) {
            if (this.triangles[i * 3] < 0) {
                System.out.println("Illegal index in triangles " + i);
            }
            if (this.triangles[i * 3] > this.vertexPointer) {
                System.out.println("Illegal vector index in triangles " + i + ": " + this.triangles[i * 3]);
                continue;
            }
            if (this.vertices[this.triangles[i * 3]] > 0) continue;
            System.out.println("Illegal vector reference in triangles " + i);
        }
    }

    private class IndexCache {
        private int nx;
        private int ny;
        private int oldXFirst;
        private int oldXSecond;
        private int newXFirst;
        private int newXSecond;
        private int[] oldYFirst;
        private int[] oldYSecond;
        private int[] newYFirst;
        private int[] newYSecond;
        private int[] oldZFirst;
        private int[] oldZSecond;
        private int[] newZFirst;
        private int[] newZSecond;

        IndexCache(int ny, int nx) {
            this.nx = nx;
            this.ny = ny;
            this.oldYFirst = new int[nx];
            this.oldYSecond = new int[nx];
            this.newYFirst = new int[nx];
            this.newYSecond = new int[nx];
            this.initX();
            this.initOldY();
            this.initNewY();
            this.oldZFirst = new int[ny * nx];
            this.oldZSecond = new int[ny * nx];
            this.newZFirst = new int[ny * nx];
            this.newZSecond = new int[ny * nx];
            this.initOldZ();
            this.initNewZ();
        }

        private void initX() {
            this.newXFirst = -1;
            this.newXSecond = -1;
        }

        private void initOldY() {
            for (int i = 0; i < this.nx; ++i) {
                this.oldYFirst[i] = -1;
                this.oldYSecond[i] = -1;
            }
        }

        private void initNewY() {
            for (int i = 0; i < this.nx; ++i) {
                this.newYFirst[i] = -1;
                this.newYSecond[i] = -1;
            }
        }

        private void initOldZ() {
            for (int i = 0; i < this.nx * this.ny; ++i) {
                this.oldZFirst[i] = -1;
                this.oldZSecond[i] = -1;
            }
        }

        private void initNewZ() {
            for (int i = 0; i < this.nx * this.ny; ++i) {
                this.newZFirst[i] = -1;
                this.newZSecond[i] = -1;
            }
        }

        private int getNewZFirst(int i, int j) {
            return this.newZFirst[i * this.nx + j];
        }

        private int getNewZSecond(int i, int j) {
            return this.newZSecond[i * this.nx + j];
        }

        private int getOldZFirst(int i, int j) {
            return this.oldZFirst[i * this.nx + j];
        }

        private int getOldZSecond(int i, int j) {
            return this.oldZSecond[i * this.nx + j];
        }

        private void setNewZFirst(int i, int j, int v) {
            this.newZFirst[i * this.nx + j] = v;
        }

        private void setNewZSecond(int i, int j, int v) {
            this.newZSecond[i * this.nx + j] = v;
        }

        private void changeZ() {
            for (int i = 0; i < this.nx * this.ny; ++i) {
                this.oldZFirst[i] = this.newZFirst[i];
                this.oldZSecond[i] = this.newZSecond[i];
            }
        }

        private void stepCell() {
            this.oldXFirst = this.newXFirst;
            this.oldXSecond = this.newXSecond;
            this.newXFirst = -1;
            this.newXSecond = -1;
        }

        private void stepRow() {
            for (int i = 0; i < this.nx; ++i) {
                this.oldYFirst[i] = this.newYFirst[i];
                this.oldYSecond[i] = this.newYSecond[i];
            }
            this.initNewY();
        }

        private void stepMatrix() {
            this.changeZ();
            this.initNewZ();
        }

        private int needComputing(int y, int x, int c) {
            if (y == 0 && c == 0) {
                System.err.println("y=0 c=0 !!");
            }
            if (x == 0 && (c == 3 || c == 4)) {
                System.err.println("x=0! c=");
            }
            switch (c) {
                case 0: {
                    return this.getOldZSecond(y - 1, x);
                }
                case 1: {
                    return this.getOldZFirst(y, x);
                }
                case 2: {
                    return this.getOldZSecond(y, x);
                }
                case 3: {
                    return this.getOldZFirst(y, x - 1);
                }
                case 4: {
                    return this.oldYFirst[x - 1];
                }
                case 5: {
                    return this.oldYFirst[x];
                }
                case 6: {
                    return this.newXFirst;
                }
                case 7: {
                    return this.oldXFirst;
                }
                case 8: {
                    return this.oldYSecond[x];
                }
                case 9: {
                    return this.newXSecond;
                }
                case 10: {
                    return this.newYSecond[x];
                }
                case 11: {
                    return this.oldXSecond;
                }
            }
            return -1;
        }

        void setNewValues(int y, int x, int c, int v) {
            switch (c) {
                case 6: {
                    this.newYFirst[x] = v;
                    this.newXFirst = v;
                    break;
                }
                case 9: {
                    this.newXSecond = v;
                    this.setNewZFirst(y, x, v);
                    break;
                }
                case 10: {
                    this.newYSecond[x] = v;
                    this.setNewZSecond(y, x, v);
                    break;
                }
                default: {
                    System.err.println("Further setting is necessary: " + c + " is a newly computed edge." + y + " , " + x);
                }
            }
        }
    }
}

