#!/usr/bin/env python
# The Broad Institute
# SOFTWARE COPYRIGHT NOTICE AGREEMENT
# This software and its documentation are copyright 2010 by the
# Broad Institute/Massachusetts Institute of Technology. All rights are
# reserved.

# This software is supplied without any warranty or guaranteed support
# whatsoever. Neither the Broad Institute nor MIT can be responsible for its
# use, misuse, or functionality.
"""Convert qstat xml output to tabular format."""

from __future__ import division
import xml.etree.cElementTree
import optparse
import subprocess
import sys

lstQstatArgs=["-u", "*", "-s", "az"]

lstQstat2Args=["-u", "*", "-s", "h"]

lstColumns=["state", "JB_job_number", "JAT_prio", "JB_name", "JB_owner", "JAT_start_time", "JB_submission_time",
            "queue_name", "slots"]

def printJob(job, fOut):
    lstValues=[]
    for strColumn in lstColumns:
        strValue = job.get(strColumn)
        if strValue is None:
            subelement = job.find(strColumn)
            if subelement is not None:
                strValue = subelement.text
        if strValue is None:
            strValue = "null"
        lstValues.append(strValue)
    print >>fOut, "\t".join(lstValues)
    
def tabularize_qstat(fIn, fOut):
    print >>fOut, "\t".join(lstColumns)
    tree = xml.etree.cElementTree.parse(fIn)
    for queue_infoElement in tree.findall("queue_info"):
        for job_listElement in queue_infoElement.findall("job_list"):
            printJob(job_listElement, fOut)

    # Zombie jobs appear here
    for job_infoElement in tree.findall("job_info"):
        for job_listElement in job_infoElement.findall("job_list"):
            printJob(job_listElement, fOut)
        
    
def main(argv=None):
    if argv is None:
        argv = sys.argv
    global lstQstatArgs
    parser = optparse.OptionParser(usage=argv[0] + " [options] [ -- additional-qstat-args]" +
                                   "Get qstat xml output and convert to tabular format.\n" +
                                   "By default, passes '" + " ".join(lstQstatArgs) + "' to qstat, in addition to " +
                                   "-xml.\nTo add additional qstat args, put them on the command line after '--' argument."  )
    parser.add_option("-i", "--stdin", action="store_true", default=False,
                      help="Instead of invoking qstat -xml, read qstat output from stdin.")
    parser.add_option("-c", "--clear_defaults", action="store_true", default=False,
                      help="Clear the default qstat arguments (except for -xml, which is required).")
    dctOptions, lstArgs = parser.parse_args(sys.argv)
    if dctOptions.clear_defaults:
        lstQstatArgs = []
    if len(lstArgs) > 1 and dctOptions.stdin:
        print >>sys.stderr, "ERROR: Should not specify qstat args if using --stdin."
        return 1
    lstQstatArgs += lstArgs[1:]

    if dctOptions.stdin:
        fQstatInput = sys.stdin
    else:
        proc = subprocess.Popen(["qstat", "-xml"] + lstQstatArgs, stdout=subprocess.PIPE)
        fQstatInput = proc.stdout

    tabularize_qstat(fQstatInput, sys.stdout)
    if not dctOptions.stdin:
        if proc.wait() != 0:
            print >> sys.stderr, "ERROR: qstat returned " + proc.returncode
            return proc.returncode

    return 0
    
if __name__ == "__main__":
    sys.exit(main())
    
