package org.broadinstitute.cga.tools.seq;

import org.broadinstitute.cga.tools.seq.BamGrasp;
import net.sf.samtools.*;
import java.io.*;
import java.lang.*;
import java.util.*;

public class CoverageByBase {
  public static void main(String args[]) {
      if (args.length!=4) {
	  System.out.println("Usage: CoverageByBase <BAMFilename> <BlacklistFilename> <OutFilename> <Chromosome>");
	  System.out.println("   (or specify 'none' for BlacklistFilename)");
          System.exit(1);
      }

      String BAMFilename = args[0];
      String BlacklistFilename = args[1];
      String OutFilename = args[2];
      Integer Chromosome = Integer.parseInt(args[3]);
     
      Boolean write_binary_output = false;

      BamGrasp bam = null;
      try {
          bam = new BamGrasp(BAMFilename,BlacklistFilename);
      } catch (Exception e) {
          System.out.println(e.getMessage());
          System.exit(1);
      }

      // initialize BAM iterator

      String seqname = bam.getChrName(Chromosome);
      SAMRecordIterator c = bam.reader.queryOverlapping(seqname,1,260000000);

      // scan BAM

      long start_time = System.currentTimeMillis();
      long idx = 0;
      int start = -1, end = -1;
      int maxend = -1;

      int count[] = new int[300000000];   // java initializes to zero

      while(c.hasNext()) {
	SAMRecord x = c.next();
        start = x.getAlignmentStart();
	end = x.getAlignmentEnd();
        if (end>maxend) { maxend=end; }

        idx++;
        if ((idx%1000000)==0) {
            float elapsed = (System.currentTimeMillis() - start_time);
            elapsed /= 1000;
            float rate = idx / elapsed;
            System.out.println(idx + " records  " + elapsed + " seconds  " + rate + " records/second  "
			       + Chromosome + ":" + start);
        }

        // filtering
	if (x.getReadUnmappedFlag()) { continue; }          // filter out unmapped reads
        if (x.getDuplicateReadFlag()) { continue; }         // filter out duplicate reads
        if (x.getNotPrimaryAlignmentFlag()) { continue; }   // filter out non-primary alignments
        if (x.getMappingQuality()==0) { continue; }

	if (bam.isReadGroupBlacklisted(x.getAttribute("RG").toString())) { continue; }

	for (int i=start;i<=end;i++) { count[i]++; } 
      }

      bam.close();

      if (maxend==0) { 
	  System.out.println("Error: no coverage");
          System.exit(1);
      }

      // write output

      String outname_partial = OutFilename+".partial";
      int bsz = 10000000;

      try {
	  if (write_binary_output) {
	      DataOutputStream out = new DataOutputStream(new BufferedOutputStream(new FileOutputStream(outname_partial),bsz));
	      for (int i=0;i<=maxend;i++) {
                  out.writeInt(count[i]);
              }
	      out.close();
	  } else {
	      BufferedWriter out = new BufferedWriter(new FileWriter(outname_partial),bsz);
	      for (int i=0;i<=maxend;i++) {
		  out.write(count[i] + "\n");
	      }
	      out.close();
	  }
	  File f = new File(outname_partial);
	  f.renameTo(new File(OutFilename));
  
      } catch (Exception e) { System.out.println("File write failed"); System.exit(1); }

      System.out.println("Finished");
      System.exit(0);

  }
}
