/* Copyright (C) 2003-2008 Dan Arlow
 * 
 * This file is part of motifADE.
 * 
 * motifADE is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * motifADE is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with motifADE.  If not, see <http://www.gnu.org/licenses/>.
 */

/* 
 *  buffered_reader.hpp
 *
 *  Class that implements fast buffered reading from files
 *  by slurping them in their entirety into a allocated buffer.
 */


#ifndef BUFFERED_READER_H
#define BUFFERED_READER_H


#include <string>
#include <iterator>
#include <utility>
#include <fstream>
#include <iosfwd>


#include "common.hpp"


class BufferedReader {
public:
	virtual bool		good()				const = 0;
	virtual bool		bad()				const = 0;
	virtual bool		operator!()			const = 0;
	
	virtual bool		eof()				const = 0;
	
	virtual char		peek()				const = 0;
	virtual char		get()				      = 0;
	
	// gets a delimited line from the buffer -- mimics std::getline
	virtual BufferedReader& getline( string& s, char delim = '\n' ) = 0;
};


class IstreamBufferedReader : public BufferedReader {
public:
	IstreamBufferedReader() : is( 0 )					{}
	IstreamBufferedReader( istream* ist ) : is( ist )   {}
	virtual ~IstreamBufferedReader()					{}
	
	virtual bool		good()				const { return is->good(); }
	virtual bool		bad()				const { return is->bad(); }
	virtual bool		operator!()			const { return is->operator!(); }
	
	virtual bool		eof()				const { return is->eof(); }
	
	virtual char		peek()				const { return is->peek(); }
	virtual char		get()				      { return is->get(); }
	
	// gets a delimited line from the buffer -- mimics std::getline
	virtual BufferedReader& getline( string& s, char delim = '\n' ) { std::getline( *is, s, delim ); return *this; }
	
protected:
	istream*			is;
};


class IfstreamBufferedReader : public IstreamBufferedReader {
public:
	IfstreamBufferedReader() : ifs() { is = &ifs; }
	IfstreamBufferedReader( const char* file ) : ifs( file ) { is = &ifs; }
	virtual ~IfstreamBufferedReader() {}
	
	void				open( const char* file )	{ ifs.open( file ); }
	void				close()						{ ifs.close(); }
	
private:
	ifstream ifs;
};


// factory method for file BufferedReader classes
BufferedReader* openBufferedReader( const char* file );

#ifdef MOTIFADE_USE_FAST_IO


class FastBufferedReader : public BufferedReader {
public:
	typedef pair< const char*, const char* >	range_type;
	static const range_type						bad_range;
	static const char							bad_char;

public:
	FastBufferedReader() : isGood( false ), dataSize( 0 ), data( 0 ), tail( 0 ), pos( 0 ), end( 0 ) { }
	FastBufferedReader( const char* file ) : isGood( false ), dataSize( 0 ), data( 0 ), tail( 0 ), pos( 0 ), end( 0 ) { open( file ); }
	virtual ~FastBufferedReader() { close(); }
	
	void				open( const char* file );
	void				close()				{ freeStorage(); }
	
	virtual bool		good()				const { return isGood; }
	virtual bool		bad()				const { return !good(); }
	virtual bool		operator!()			const { return !good(); }
	
	off_t				size()				const { return dataSize; }
	
	virtual bool		eof()				const { return end == tail; }
	
	virtual char		peek()				const { if( good() && end < tail ) return *end; else return bad_char; }
	virtual char		get()				      { if( good() && end < tail ) return *end++; else return bad_char; }
	
	range_type			getrange()			const { if( good() ) return range_type( pos, end ); else return bad_range; }
	
	range_type			getline( char delim = '\n' );
	
	// gets a delimited line from the buffer -- mimics std::getline
	virtual BufferedReader& getline( string& s, char delim = '\n' );
	
private:
	bool				isGood;
	off_t				dataSize;
	char				*data;
	const char			*tail, *pos, *end;
	
	void				freeStorage() { isGood = false; if( data == 0 ) return; delete data; dataSize = 0; data = 0; }
};


#else


#define FastBufferedReader IfstreamBufferedReader


#endif // MOTIFADE_USE_FAST_IO


#endif // BUFFERED_READER_H
