/* Copyright (C) 2003-2008 Dan Arlow
 * 
 * This file is part of motifADE.
 * 
 * motifADE is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * motifADE is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with motifADE.  If not, see <http://www.gnu.org/licenses/>.
 */

/* 
 *  column_pair_reader.hpp
 *
 *  Simple class for reading a delimited text file and extracting two columns.
 *
 *  Copyright (c) 2005 Dan Arlow. All rights reserved. DO NOT DISTRIBUTE.
 */


#ifndef COLUMN_MAP_READER_H
#define COLUMN_MAP_READER_H


#include <map>
#include <vector>
#include <algorithm>
#include <string>
#include <iostream>


#include "common.hpp"
#include "map_pair_receiver.hpp"
#include "column_pair_reader.hpp"


template< typename MapType >
class ColumnMapReader {
public:
	typedef MapPairReceiver< MapType, typename MapType::key_type, typename MapType::mapped_type >	ReceiverType;
	typedef ColumnPairReader< ReceiverType >														PairReaderType;

public:
	struct Stats : public OperationStats {
		Stats() { clear(); }
		
		virtual void clear() {
			num_mappings_loaded = 0;
			num_missing_keys = 0;
			num_missing_values = 0;
			num_good_mappings = 0;
		}
		
		virtual ostream& print( ostream& os ) const {
			os << num_mappings_loaded << " mappings loaded." << endl;
			os << num_missing_keys << " mappings were missing keys." << num_missing_values << " missing values." << endl;
			os << num_good_mappings << " mappings had both keys and values." << endl;
			os << num_good_mappings << " mappings had both keys and values." << endl;
			os << num_mappings_retained << " mappings retained." << endl;
			return os;
		}
		
		unsigned int	num_mappings_loaded,
						num_missing_keys,
						num_missing_values,
						num_good_mappings,
						num_mappings_retained;
	};
	
	ColumnMapReader( BufferedReader& ist, unsigned int key_col, unsigned int value_col, char delimC = '\t', char quoteC = '"' )
		: is( ist ), keyCol( key_col ), valueCol( value_col ), delimiter( delimC ), quote( quoteC ) { }
	
	void	load( MapType& cMap, typename ColumnMapReader< MapType >::Stats& stats )
	{
		stats.clear();
		
		ReceiverType receiver( cMap );
		PairReaderType pairReader( is, keyCol, valueCol, delimiter, quote );
		typename PairReaderType::Stats readerStats;
		pairReader.load( receiver, readerStats );
		
		stats.num_mappings_loaded = readerStats.num_mappings_loaded;
		stats.num_missing_keys = readerStats.num_missing_keys;
		stats.num_missing_values = readerStats.num_missing_values;
		stats.num_good_mappings = readerStats.num_good_mappings;
		stats.num_mappings_retained = cMap.size(); 
	}

private:
	BufferedReader& is;
	const unsigned int keyCol, valueCol;
	const char delimiter, quote;
	
};


#endif // COLUMN_MAP_READER_H
