/* Copyright (C) 2003-2008 Dan Arlow
 * 
 * This file is part of motifADE.
 * 
 * motifADE is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * motifADE is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with motifADE.  If not, see <http://www.gnu.org/licenses/>.
 */

/* 
 *  iupac_alphabet.cpp
 *
 *  Implementation of an IUPAC character matching facility
 *
 *  Copyright (c) 2003-2005 Dan Arlow. All rights reserved. DO NOT DISTRIBUTE.
 *
 *  CREATED  07/18/05
 *  MODIFIED 07/18/05
 *
 *  Written by Patrick Varilly.
 *
 */


#include <ctype.h>

#include "common.hpp"
#include "iupac_alphabet.hpp"


// gPatternMatch[x][y] is true if IUPAC pattern character x matches y
// e.g. gPatternMatch['A']['A'] is true
//      gPatternMatch['N']['A' or 'C' or 'G' or 'T'] is true
// etc.
char gPatternMatch[256][256];

// Build the pattern matching table
// Note that even though this is huge (64K!), most of the table will remain
// off the cache, so it's not a big performance hit to keep nucleotides
// encoded as 8-bit characters.
static void setMatch( const char inIUPAC, const char inChar )
{
	// Match lower-case & upper-case
	gPatternMatch[ uint8_t( toupper(inIUPAC) ) ][ uint8_t( toupper(inChar) ) ] = true;
	gPatternMatch[ uint8_t( toupper(inIUPAC) ) ][ uint8_t( tolower(inChar) ) ] = true;
	gPatternMatch[ uint8_t( tolower(inIUPAC) ) ][ uint8_t( toupper(inChar) ) ] = true;
	gPatternMatch[ uint8_t( tolower(inIUPAC) ) ][ uint8_t( tolower(inChar) ) ] = true;
}

inline void setMatch( const char inIUPAC, const char inChar1, const char inChar2 )
{
	setMatch( inIUPAC, inChar1 );
	setMatch( inIUPAC, inChar2 );
}

inline void setMatch( const char inIUPAC, const char inChar1, const char inChar2, const char inChar3 )
{
	setMatch( inIUPAC, inChar1 );
	setMatch( inIUPAC, inChar2 );
	setMatch( inIUPAC, inChar3 );
}

inline void setMatch( const char inIUPAC, const char inChar1, const char inChar2, const char inChar3, const char inChar4 )
{
	setMatch( inIUPAC, inChar1 );
	setMatch( inIUPAC, inChar2 );
	setMatch( inIUPAC, inChar3 );
	setMatch( inIUPAC, inChar4 );
}

// Return a dummy int so we can create this table at program start by assigning
// a global variable
static int buildPatternMatchTable( void )
{
	// Initially, nothing matches
	for( int i = 0; i < 256; i++ )
		for( int j = 0; j < 256; j++ )
			gPatternMatch[i][j] = 0;

	// Now go through the IUPAC characters
	setMatch( 'A', 'A' );
	setMatch( 'C', 'C' );
	setMatch( 'G', 'G' );
	setMatch( 'T', 'T' );

	setMatch( 'M', 'A', 'C' );
	setMatch( 'R', 'A', 'G' );
	setMatch( 'W', 'A', 'T' );
	setMatch( 'S', 'C', 'G' );
	setMatch( 'Y', 'C', 'T' );
	setMatch( 'K', 'G', 'T' );

	setMatch( 'V', 'A', 'C', 'G' );
	setMatch( 'H', 'A', 'C', 'T' );
	setMatch( 'D', 'A', 'G', 'T' );
	setMatch( 'B', 'C', 'G', 'T' );

	setMatch( 'N', 'A', 'C', 'G', 'T' );

	return 0;
}

namespace StaticInitializers { int dummyPatternMatchTable = buildPatternMatchTable(); }

// reverse complementing of an IUPAC string
char gIUPACReverseComplChar[256];

inline void setIUPAC_RCs( const char inIUPAC, const char inIUPAC_RC )
{
	gIUPACReverseComplChar[uint8_t( toupper( inIUPAC ) )] = toupper(inIUPAC_RC);
	gIUPACReverseComplChar[uint8_t( tolower( inIUPAC ) )] = toupper(inIUPAC_RC);
	
	gIUPACReverseComplChar[uint8_t( toupper( inIUPAC_RC ) )] = toupper(inIUPAC);
	gIUPACReverseComplChar[uint8_t( tolower( inIUPAC_RC ) )] = toupper(inIUPAC);
}

static int buildIUPACReverseComplementTable( void )
{
	// Bad characters
	for( int i = 0; i < 256; i++ )
		gIUPACReverseComplChar[i] = 0;

	setIUPAC_RCs( 'A', 'T' );
	setIUPAC_RCs( 'C', 'G' );

	setIUPAC_RCs( 'M', 'K' );
	setIUPAC_RCs( 'R', 'Y' );
	setIUPAC_RCs( 'W', 'S' );
	
	setIUPAC_RCs( 'V', 'B' );
	setIUPAC_RCs( 'H', 'D' );

	setIUPAC_RCs( 'N', 'N' );

	return 0;
}

namespace StaticInitializers { int dummyIUPAC_RCTable = buildIUPACReverseComplementTable(); }
