/* Copyright (C) 2003-2008 Dan Arlow
 * 
 * This file is part of motifADE.
 * 
 * motifADE is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * motifADE is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with motifADE.  If not, see <http://www.gnu.org/licenses/>.
 */

/* 
 *  mann_whitney_u.hpp
 *
 *  Interface to functions that perform a fast caching Mann-Whitney U-test
 */


#ifndef MANN_WHITNEY_U_H
#define MANN_WHITNEY_U_H


#include <iostream>
#include <iomanip>
#include <algorithm>
#include <functional>
#include <numeric>
#include <vector>
#include <cmath>

#include "common.hpp"
#include "promoter.hpp"
#include "expression_statistics.hpp"
#include "univariate_expression_statistic_calculator.hpp"
#include "mark_set.hpp"


// Mann-Whitney U statistic class
class MannWhitneyU : public Statistic {
public:
    MannWhitneyU( double u_val, double z_val, double freq_val, double med1, double med2 );
    
    MannWhitneyU( double freq_val = 0 ) : Statistic( true ), z( 0 ), freq( freq_val ), m1( 0 ), m2( 0 ) { }
    MannWhitneyU( const MannWhitneyU& s ) : Statistic( s ), z( s.z ), freq( s.freq ), m1( s.m1 ), m2( s.m2 ) { }
    virtual ~MannWhitneyU() { }
    
    virtual void				print( ostream& os ) const;
    virtual void				printHeader( ostream& os ) const;
    
public: // (should be) constant public data members
    double						u, z;
    double						freq;
    double						m1, m2;
};


// class to implement a fast caching calculator for a the Mann-Whitney U statistic
class MannWhitneyUCalculator : public UnivariateExpressionStatisticCalculator {
public:
    MannWhitneyUCalculator( const PromoterVector& promoters, unsigned int dimension = 0 );
    MannWhitneyUCalculator( const MannWhitneyUCalculator& calc, bool copyRanks = false );
    virtual ~MannWhitneyUCalculator() { }
    
    void				buildRanks();
    
    virtual void		shuffleExpression();
    
    virtual void		compute( const MarkSet& marks, Statistic& statistic ); // fix this to take a MannWhitneyU&
    
    // ack! terrible!
    virtual double		computeValue( const MarkSet& marks );
    
    
    double				computeMedian( const MarkSet& marks, bool marked ) const;
    
    
    // accessors
    const uvector&		getSortedIndex()			const { return sortedIndex; }
    const dvector&		getRanks()					const { return ranks; }
	const dvector&		getNullDistributionMean()   const { return nullDistributionMean; }
	const dvector&		getNullDistributionStddev() const { return nullDistributionStddev; }

protected:
	double				computeRankSum( const MarkSet& marks, unsigned int& n1, unsigned int& n2 );
	
	virtual void		computeNullDistributions();
	
protected:
    uvector				sortedIndex;
    dvector				ranks;
	
	dvector				nullDistributionMean;
	dvector				nullDistributionStddev;

};


// simple increment functor
struct Increment {
	Increment() : value( 0 ) { }
	unsigned int							operator()() { return value++; }
	unsigned int							value;
};


// compare functor two indices based on the expression at those indices
struct ExpressionIndexCompare : public binary_function< unsigned int, unsigned int, bool > {
    ExpressionIndexCompare( const dvector& expr ) : expression( expr ) { }
    bool				operator()( unsigned int lhs, unsigned int rhs ) { return expression[ lhs ] < expression[ rhs ]; }
    const dvector&		expression;
};



#endif // MANN_WHITNEY_U_H
