/* Copyright (C) 2003-2008 Dan Arlow
 * 
 * This file is part of motifADE.
 * 
 * motifADE is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * motifADE is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with motifADE.  If not, see <http://www.gnu.org/licenses/>.
 */

/* 
 *  neighborhood_kmer_incidence_computer.cpp
 *
 *  Class that computes the incidence of (k,r)-motifs
 */


#include <string>
#include <vector>
#include <algorithm>

#include "common.hpp"
#include "motif_instance.hpp"
#include "motif_incidence_set.hpp"
#include "motif_incidence_map.hpp"
#include "kmer_encoder.hpp"
#include "kmer_incidence_map.hpp"
#include "motif_incidence_computer.hpp"
#include "neighborhood_kmer_incidence_computer.hpp"


void
NeighborhoodKmerIncidenceComputer::computeIncidence( MotifIncidenceMap::MotifNumber m, const MotifIncidenceMap* incidenceMap, MotifIncidenceSet& incidence )
{
	const KmerIncidenceMap* kmerIncidenceMap = dynamic_cast< const KmerIncidenceMap* >( incidenceMap ); // REDESIGN TO AVOID DOWNCAST...
	if( kmerIncidenceMap == 0 )
		throw MotifADEException( "NeighborhoodKmerIncidenceComputer::computeIncidence: not passed a KmerIncidenceMap pointer!" );
	
	string tmp;
	kmerIncidenceMap->encoder.kmerValueToSequence( m, tmp );
	
//	if( tmp != "TGACCTTGA" ) return;
//	
//	cout << "AAAAAA incidence before computeIncidence:\t" << incidence << endl;
	
	
	MotifIncidenceComputer::computeIncidence( m, incidenceMap, incidence ); // compute (k,0) incidence
	
//	cout << "AAAAAA after base class computeIncidence:\t" << incidence << endl;
	
	for( unsigned int i = 1; i <= radius; ++i )
		computeNeighborhoodIncidence( m, *kmerIncidenceMap, i, incidence ); // compute (k,i) incidence for i = 1..radius
	
	
//	cout << "AAAAAA incidence after computeIncidence:\t" << incidence << endl;
	
	incidence.makeUniqueIncidence();
	
//	cout << "AAAAAA incidence after makeUniqueIncidence:\t" << incidence << endl;
}


//TODO: FIX!

void
NeighborhoodKmerIncidenceComputer::computeNeighborhoodIncidence( KmerEncoder::KmerValue kmer, const KmerIncidenceMap& incidenceMap, unsigned int r, MotifIncidenceSet& incidence )
{
	unsigned int i;
	
	uvector positions( r );
	for( i = 0; i < r; ++i )
		positions[ i ] = i;
	
	while( positions[ 0 ] < incidenceMap.encoder.kmerSize - r ) {
		computeNeighborIncidence( kmer, incidenceMap, positions, incidence );
		++positions[ 0 ];
		unsigned int i = 0;
		while( i < r - 1 && positions[ i ] == positions[ i + 1 ] ) {
			positions[ i ] = i;
			++positions[ i + 1 ];
			++i;
		}
	}
	computeNeighborIncidence( kmer, incidenceMap, positions, incidence );
}


void
NeighborhoodKmerIncidenceComputer::computeNeighborIncidence( KmerEncoder::KmerValue kmer, const KmerIncidenceMap& incidenceMap, const uvector& positions, MotifIncidenceSet& incidence )
{
	unsigned int l = positions.size(), i;
	uvector c( l, 0 );
	
	vector< const vector< KmerEncoder::KmerValue >* > successors( l );
	for( i = 0; i < l; ++i )
		successors[ i ] = incidenceMap.encoder.getSuccessors( incidenceMap.encoder.getNtValue( kmer, positions[ i ] ) );
	
	while( c[ l - 1 ] < 3 ) {

		KmerEncoder::KmerValue v = kmer;
		for( unsigned int j = 0; j < l; ++j ) {
			v = incidenceMap.encoder.setNtValue( v, ( *successors[ j ] )[ c[ j ] ], positions[ j ] );
		}
		
		
//		if( kmer == incidenceMap.encoder.computeKmerValue( string("AGGTCA").begin() ) ) {
//			string motif;
//			incidenceMap.encoder.kmerValueToSequence( v, motif );
//			cout << "    unioning incidence for motif " << motif << ":\t" << *incidenceMap.getIncidence( v ) << endl;
//		}
		
		const MotifIncidenceSet* vIncidence = incidenceMap.getIncidence( v );
		incidence.addInstances( vIncidence ); // append vIncidence to incidence
		
		++c[ 0 ];
		i = 0;
		while( i < l - 1 && c[ i ] == 3 ) {
			c[ i ] = 0;
			++c[ i + 1 ];
			++i;
		}
	}
	
//		cout << endl;
}
