/* Copyright (C) 2003-2008 Dan Arlow
 * 
 * This file is part of motifADE.
 * 
 * motifADE is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * motifADE is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with motifADE.  If not, see <http://www.gnu.org/licenses/>.
 */

/* 
 *  sequence.cpp
 *
 *  Base class for representing DNA sequences.
 */

#include <cctype>

#include "common.hpp"

#include "sequence.hpp"


using namespace std;


// Sequence copy constructor -- creates this sequence to be a copy of s
Sequence::Sequence( const Sequence& s )
	: header( s.getHeader() ),
	  data( s.getData() )
{
	setID( s.getID() );
}


// Sequence constructor -- stores copies of arguments in internal fields
Sequence::Sequence( const HeaderType& init_header, const DataType& init_data )
	: header( init_header ),
	  data( init_data )
{	
	string_toupper( data );
	
	parseID();
}


// Sequence constructor -- stores copies of arguments in internal fields
Sequence::Sequence( const HeaderType& init_header, const IDType& init_id, const DataType& init_data )
	: header( init_header ),
	  data( init_data )
{
	string_toupper( data );
}


// Sequence method -- parses the FASTA record header for an NM_ID
void
Sequence::parseID()
{
        IDLoc loc( findID( header ) );
        
        if( loc.first == header.end() )
			throw( MotifADEException( "Sequence::parseID: No ID found!" ) );
		
        id.assign( loc.first, loc.second );
}


// Sequence method -- computes and returns the total number of bases
unsigned int
Sequence::totalSequenceLength() const
{
	unsigned int c = 0;
	for( DataType::const_iterator i = data.begin(); i < data.end(); ++i )
		switch( *i ) { case 'A': case 'C': case 'G': case 'T': ++c; }
	return c;
}


// Sequence method -- finds an ID in a header
Sequence::IDLoc
Sequence::findID( const HeaderType& h )
{
	return range_find_if( h.begin(), h.end(), IsAlphaNumeric() );
}

// method to read a sequence from a BufferedReader (need to make BufferedReader an istream!)
BufferedReader& operator>>( BufferedReader& is, Sequence& seq )
{
	if( !is ) throw( MotifADEException( "operator>>( BufferedReader&, Sequence& ): couldn't read from stream!" ) );
	
	is.getline( seq.header, '\n' );
	
	seq.data.resize( 0 );
	string buf;
	while( !is.eof() && is.peek() != '>' ) {
		is.getline( buf, '\n' );
		seq.data.append( buf );
	}
	
	string_toupper( seq.data );
	
	return is;
}


// method to print a sequence to an ostream
ostream& operator<<( ostream& os, const Sequence& seq )
{
	os << seq.getHeader() << endl;
	os << seq.getData() << endl;
	return os;
}
