/* Copyright (C) 2003-2008 Dan Arlow
 * 
 * This file is part of motifADE.
 * 
 * motifADE is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * motifADE is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with motifADE.  If not, see <http://www.gnu.org/licenses/>.
 */

/* 
 *  student_t.cpp
 *
 *  Class that performs Student's T-test
 */


#include <iostream>
#include <iomanip>

#include "common.hpp"
#include "promoter.hpp"
#include "expression_statistics.hpp"
#include "univariate_expression_statistic_calculator.hpp"
#include "mark_set.hpp"
#include "student_t.hpp"


// StudentT constructor -- computes its two-tailed p-values
StudentT::StudentT( double mean1, double var1, double num1, double mean2, double var2, double num2 )
    : Statistic( false ),
	  freq( num1 / ( num1 + num2 ) ),
      m1( mean1 ), v1( var1 ), n1( num1 ),
      m2( mean2 ), v2( var2 ), n2( num2 )
{
	computeParams();
	
    p = 2.0 * tCdf( t > 0 ? -t : t, df );
    
    p_adj = p * static_cast< double >( Statistic::NUMBER_OF_TESTS );
    if( p_adj > 1.0 ) p_adj = 1.0;
}


// StudentT method -- compute the parameters of the null distribution of this t statistic
void
StudentT::computeParams()
{
	double av1 = v1 / n1, av2 = v2 / n2, sv = av1 + av2;
	sd = sqrt( sv );
	df = sv * sv / ( av1 * av1 / ( n1 - 1 ) + av2 * av2 / ( n2 - 1 ) );
    t = ( m1 - m2 ) / sd;
}


// StudentT method -- print the summary of the statistic
void
StudentT::print( ostream& os ) const
{
    #ifndef MOTIFADE_CRAPPY_IOMANIP
        os << showpoint << fixed;
    #endif
    
    os << setprecision( 6 );
    
    os << freq << '\t';
    
    if( error ) {
        
        os << "Err";
        
    } else {
        
        #ifndef MOTIFADE_CRAPPY_IOMANIP
            os << showpos;
        #endif
        
        os << ( m1 - m2 );
        
        #ifndef MOTIFADE_CRAPPY_IOMANIP
            os << noshowpos;
        #endif
        
        os << '\t' << p << '\t' << p_adj;
        
    }
}


// StudentT method -- print the header of the fields
void
StudentT::printHeader( ostream& os ) const
{
    os << "Frequency" << '\t' << "Difference of Means" << '\t' << "P-value";
	if( USE_ADJUSTED )
		os << '\t' << "Adjusted P-value";
}




// StudentTCalculator method -- compute and store the T statistic for a given MarkSet
void
StudentTCalculator::compute( const MarkSet& marks, Statistic& statistic ) // fix this to take a StudentT&
{
    unsigned int i, n = expression.size();
    double x, dm, m1 = 0, v1 = 0, n1 = 0, m2 = 0, v2 = 0, n2 = 0, nd = n;
    
    StudentT* stat = dynamic_cast< StudentT* >( &statistic );
    if( stat == 0 ) throw( MotifADEException( "StudentTCalculator::compute: not passed a StudentT pointer!" ) );
    
    // accumulate the first means of the marked and unmarked data
	for( i = 0; i < n; ++i ) {
		x = expression[ i ];
		if( marks[ i ] ) {
			m1 += x;
			++n1;
		} else {
			m2 += x;
			++n2;
		}
	}	
	
	if( n1 < 5 || n2 < 5 ) {
		*stat = StudentT( n1 / nd );
		return;
	}

	m1 /= n1;
	m2 /= n2;
	dm = m1 - m2;
		
    // accumulate the squared deviations of the marked and unmarked data
	for( i = 0; i < n; ++i ) {
		x = expression[ i ];
		if( marks[ i ] ) {
			v1 += fast_square( x - m1 );
		} else {
			v2 += fast_square( x - m2 );
		}
	}
	v1 /= n1 - 1.0;
	v2 /= n2 - 1.0;
	
    *stat = StudentT( m1, v1, n1, m2, v2, n2 );
	
	if( useResiduals && stat->p_adj < alpha )
		for( i = 0; i < n; ++i )
			if( marks[ i ] )
				expression[ i ] -= dm;
	
    return;
}


// terrible!
// StudentTCalculator method -- compute and return the value of the t statistic for a given MarkSet
double
StudentTCalculator::computeValue( const MarkSet& marks )
{
	StudentT stat;
	compute( marks, stat );
	return stat.p;
}

