/* Copyright (C) 2003-2008 Dan Arlow
 * 
 * This file is part of motifADE.
 * 
 * motifADE is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * motifADE is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with motifADE.  If not, see <http://www.gnu.org/licenses/>.
 */

/* 
 *  bijective_map_filter.hpp
 *
 *  A simple class for reading a mapping between IDs and filtering it to a bijection.
 */

#ifndef BIJECTIVE_MAP_FILTER_H
#define BIJECTIVE_MAP_FILTER_H

#include <map>
#include <iostream>


#include "common.hpp"

#include "buffered_reader.hpp"
#include "sequence.hpp"
#include "operation_stats.hpp"


// nearly stateless class to read an MapType from a stream
template< typename MultiMapType, typename BijectiveMapType >
class BijectiveMapFilter {
public:
	typedef map< typename MultiMapType::key_type, unsigned int >	KeyCountMap;
	typedef map< typename MultiMapType::mapped_type, unsigned int > ValueCountMap;

public:
	struct Stats : OperationStats {
		Stats() { clear(); }
		
		virtual void clear() {
			keyCounts.clear();
			valueCounts.clear();
			num_mappings_loaded = 0;
			num_unique_keys = 0;
			num_unique_values = 0;
			num_repeated_keys = 0;
			num_repeated_values = 0;
			num_key_repeats = 0;
			num_value_repeats = 0;
			num_mappings_retained = 0;
			removeRepeatedKeys = true;
			removeRepeatedValues = true;
		}
		
		virtual ostream& print( ostream& os ) const {
			os << num_mappings_loaded << " mappings loaded." << endl;
			os << num_unique_keys << " unique keys, and " << num_unique_values << " unique values" << endl;
			os << num_repeated_keys << " repeated keys, repeated a total of " << num_key_repeats << " times;" << endl;
			os << num_repeated_values << " repeated values, repeated a total of " << num_value_repeats << " times." << endl;
			os << "mappings with repeated keys were " << ( removeRepeatedKeys ? "" : "not " ) << "removed, and" << endl;
			os << "mappings with repeated values were " << ( removeRepeatedValues ? "" : "not " ) << "removed." << endl;
			os << num_mappings_retained << " mappings retained." << endl;
			return os;
		}
		
		KeyCountMap		keyCounts;
		ValueCountMap	valueCounts;
		
		unsigned int	num_mappings_loaded,
						num_unique_keys,
						num_unique_values,
						num_repeated_keys,
						num_repeated_values,
						num_key_repeats,
						num_value_repeats,
						num_mappings_retained;
		
		bool			removeRepeatedKeys, removeRepeatedValues;
	};
		
	BijectiveMapFilter( bool removeRepeatKeys = true, bool removeRepeatValues = true )
		: removeRepeatedKeys( removeRepeatKeys ), removeRepeatedValues( removeRepeatValues ) { }
	
	void	filter( const MultiMapType& mMap, BijectiveMapType& bMap, Stats& stats )
	{
		stats.clear();
		stats.removeRepeatedKeys = removeRepeatedKeys;
		stats.removeRepeatedValues = removeRepeatedValues;
		
		bMap.clear();
		
		int count;
		
		for( typename MultiMapType::const_iterator i = mMap.begin(); i != mMap.end(); ++i ) {
			++stats.num_mappings_loaded;
			
			const typename MultiMapType::key_type& key = i->first;
			const typename MultiMapType::mapped_type& value = i->second;
			
			count = ++stats.keyCounts[ key ];
			if( count > 1 )
				++stats.num_key_repeats;
			if( count == 2 )
				++stats.num_repeated_keys;
			
			count = ++stats.valueCounts[ value ];
			if( count > 1 )
				++stats.num_value_repeats;
			if( count == 2 )
				++stats.num_repeated_values;
			
			bMap.insert( typename BijectiveMapType::value_type( key, value ) );
		}
		
		stats.num_unique_keys = stats.keyCounts.size();
		stats.num_unique_values = stats.valueCounts.size();
		
		for( typename BijectiveMapType::iterator i = bMap.begin(); i != bMap.end(); ) {
			typename BijectiveMapType::iterator tmp = i++;
			
			if( removeRepeatedKeys ) {
				typename KeyCountMap::iterator j = stats.keyCounts.find( tmp->first );
				if( j != stats.keyCounts.end() && j->second > 1 ) {
					bMap.erase( tmp );
					continue;
				}
			}
			
			if( removeRepeatedValues ) {
				typename ValueCountMap::iterator j = stats.valueCounts.find( tmp->second );
				if( j != stats.valueCounts.end() && j->second > 1 ) {
					bMap.erase( tmp );
					continue;
				}
			}
		}
			
		stats.num_mappings_retained = bMap.size();
	}

private:
	const bool		removeRepeatedKeys, removeRepeatedValues;
};

#endif // BIJECTIVE_MAP_FILTER_H
