/* Copyright (C) 2003-2008 Dan Arlow
 * 
 * This file is part of motifADE.
 * 
 * motifADE is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * motifADE is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with motifADE.  If not, see <http://www.gnu.org/licenses/>.
 */

/* 
 *  buffered_reader.cpp
 *
 *  Class that implements fast buffered reading from files
 *  by slurping them in their entirety into a allocated buffer.
 */


#include "common.hpp"


#include "buffered_reader.hpp"


// factory method for file BufferedReader classes
BufferedReader*
openBufferedReader( const char* file ) {
	if( string( "-" ) == file )
		return new IstreamBufferedReader( &cin );
	else
		#ifdef MOTIFADE_USE_FAST_IO
		return new FastBufferedReader( file );
		#else
		return new IfstreamBufferedReader( file );
		#endif
}


#ifdef MOTIFADE_USE_FAST_IO


#include <unistd.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <sys/types.h>


const FastBufferedReader::range_type FastBufferedReader::bad_range = range_type( 0, 0 );
const char FastBufferedReader::bad_char = '\0';


// BufferedReader method -- "opens" the file by slurping it into the buffer
void
FastBufferedReader::open( const char* file )
{
	int	fp;
	struct stat fs;
	
	close();
	
	fp = ::open( file, O_RDONLY );
	if( fp <= 0 ) return;
	
	::fstat( fp, &fs );
	dataSize = fs.st_size;
	
	data = new char[ dataSize ];
	tail = data + dataSize;
	
	pos = data;
	end = data;	
	
	::read( fp, data, dataSize );
	::close( fp );
	
	isGood = true;
}


// returns a range giving pointers to the start and end of the read line
FastBufferedReader::range_type
FastBufferedReader::getline( char delim )
{
	if( bad() || eof() ) return bad_range;
	
	if( end != data ) pos = end;
	for( end = pos; end < tail && *end != delim; ++end );
	if( end < tail ) ++end;
	
	return getrange();	
}


// gets a delimited line from the buffer -- mimics std::getline
BufferedReader&
FastBufferedReader::getline( string& s, char delim )
{
	FastBufferedReader::range_type r = getline();
	if( r != FastBufferedReader::bad_range ) {
		--r.second;
		s.resize( r.second - r.first );
		s.assign( r.first, r.second );
	}
	
	return *this;
}


#endif // MOTIFADE_USE_FAST_IO
