/* Copyright (C) 2003-2008 Dan Arlow
 * 
 * This file is part of motifADE.
 * 
 * motifADE is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * motifADE is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with motifADE.  If not, see <http://www.gnu.org/licenses/>.
 */

/* 
 *  expression_annotator.hpp
 *
 *  Class to associate expression vectors with promoters using an SequenceExpressionMap
 */


#ifndef EXPRESSION_ANNOTATOR_H
#define EXPRESSION_ANNOTATOR_H


#include <set>

#include "common.hpp"

#include "promoter.hpp"
#include "promoter_set.hpp"

#include "expression_vector.hpp"
#include "expression_set.hpp"
#include "sequence_expression_map.hpp"
#include "operation_stats.hpp"


class ExpressionAnnotator {
public:
	typedef		set< Sequence::IDType >		IDSet;

public:
	struct Stats : OperationStats {
		Stats() { clear(); }
		
		virtual void clear()
		{
			nonAnnotatedIDs.clear();
			removePromotersWithoutExpression = true;
			promoter_set_size = 0;
			expression_set_size = 0;
			num_mappings_in_map = 0;
			num_promoters_with_expression_in_map = 0;
			num_promoters_with_expression = 0;
			num_promoters_retained = 0;
		}
		
		virtual ostream& print( ostream& os ) const {
			os << promoter_set_size << " sequences in the promoter set, and " << expression_set_size << " expression vectors in the expression set." << endl;
			os << num_mappings_in_map << " mappings in the expression annotation map." << endl;
			os << num_promoters_with_expression_in_map << " promoters had mappings in the annotation map." << endl;
			os << num_promoters_with_expression << " had expression vectors in the expression set." << endl;
			os << "promoters without annotated expression were " << ( removePromotersWithoutExpression ? "" : "not " ) << "removed." << endl;
			os << num_promoters_retained << " promoters (with expression) retained." << endl;
			return os;
		}
		
		IDSet			nonAnnotatedIDs;
		
		bool			removePromotersWithoutExpression;
		
		unsigned int	promoter_set_size,
						expression_set_size,
						num_mappings_in_map,
						num_promoters_with_expression_in_map,
						num_promoters_with_expression,
						num_promoters_retained;
	};

public:
	ExpressionAnnotator( PromoterSet& p, bool removePromotersWithoutExpression = true )
		: removeNonAnnotated( removePromotersWithoutExpression ), promoters( p ) {}
	
	void	annotate( SequenceExpressionMap& seMap, ExpressionSet& expression, ExpressionAnnotator::Stats& stats );
	
private:
	const bool		removeNonAnnotated;
	PromoterSet&	promoters;
};



#endif // EXPRESSION_ANNOTATOR_H
