/* Copyright (C) 2003-2008 Dan Arlow
 * 
 * This file is part of motifADE.
 * 
 * motifADE is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * motifADE is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with motifADE.  If not, see <http://www.gnu.org/licenses/>.
 */

/* 
 *  expression_statistics.cpp
 *
 *  Implementation of expresssion statistics data abstractiosn.
 */


#include "common.hpp"
#include "promoter.hpp"
#include "expression_statistics.hpp"


using namespace std;


// procedure to compute values of the standard normal cumulative distribution
double normalcdf( double z ) {
    return 0.5 * ( erf( z / SQRT2 ) + 1.0 );
}


// function to compute the CDF of Student's distribution
// adapted from Numerical Recipies in C chapter 14.2 http://www.library.cornell.edu/nr/bookcpdf/c14-2.pdf
double
tCdf( double t, double df ) {
	return betai( 0.5 * df, 0.5, df / ( df + t * t ) );
}


// function to compute values of the incomplete beta function
// adapted from Numerical Recipies in C chapter 6.4 http://www.library.cornell.edu/nr/bookcpdf/c6-4.pdf
double
betai( double a, double b, double x ) {
	float bt;
	if( x <  0.0 || x >  1.0 ) throw MotifADEException( "Bad x in routine betai" );
	if( x == 0.0 || x == 1.0 ) bt = 0.0;
	else
		bt = exp( lgamma( a + b ) - lgamma( a ) - lgamma( b ) + a * log( x ) + b * log( 1.0 - x ) );
	if( x < ( a + 1.0 ) / ( a + b + 2.0 ) )
		return bt * betacf( a, b, x ) / a;
	else
		return 1.0 - bt * betacf( b, a, 1.0 - x ) / b;
}


// function to compute values of the beta function by continued fraction expansion
// adapted from Numerical Recipies in C chapter 6.4 http://www.library.cornell.edu/nr/bookcpdf/c6-4.pdf
double
betacf( double a, double b, double x ) {
	int m, m2;
	double aa, c, d, del, h, qab, qam, qap;
	
	qab = a + b;
	qap = a + 1.0;
	qam = a - 1.0;
	c   = 1.0;
	d   = 1.0 - qab * x / qap;
	if( fast_abs( d ) < FPMIN ) d = FPMIN;
	d = 1.0/d;
	h = d;
	
	for( m = 1; m <= MAXIT; m++ ) {
		m2 = 2 * m;
		aa = m * ( b - m ) * x / ( ( qam + m2 ) * ( a + m2 ) );
		d  = 1.0 + aa * d;
		if( fast_abs( d ) < FPMIN ) d = FPMIN;
		c = 1.0 + aa / c;
		if( fast_abs( c ) < FPMIN ) c = FPMIN;
		d = 1.0 / d;
		h *= d * c;
		aa = -( a + m ) * ( qab + m ) * x / ( ( a + m2 ) * ( qap + m2 ) );
		d = 1.0 + aa * d;
		if( fast_abs( d ) < FPMIN ) d = FPMIN;
		c = 1.0 + aa / c;
		if( fast_abs( c ) < FPMIN ) c = FPMIN;
		d = 1.0 / d; del = d * c;
		h *= del;
		if( fast_abs( del - 1.0 ) < EPS) break;
	}
	if( m > MAXIT ) throw MotifADEException( "a or b too big, or MAXIT too small in betacf" );
	return h;
}


// function to compute the mean and standard deviation of a dvector of data
void
descriptiveStats( const dvector& data, double& mean, double& stddev )
{
	double n = data.size(), u = n - 1;
	
	mean = 0.0;
	for( unsigned int s = 0; s < data.size(); ++s )
		mean += data[ s ];
	mean /= n;
    
    stddev = 0.0;
	for( unsigned int s = 0; s < data.size(); ++s )
		stddev += fast_square( data[ s ] - mean );
	stddev = sqrt( stddev / u );
}


// function to compute the min of two doubles
double
fast_min( double a, double b )
{
	return a <= b ? a : b;
}


// function to compute the max of a vector of doubles
double
max( const dvector& x )
{
	if( x.size() == 0 )
		throw( MotifADEException( "max: x.size() == 0" ) );
	
	double m = x[ 0 ];
	for( unsigned int i = 1; i < x.size(); ++i )
		if( x[ i ] > m )
			m = x[ i ];
	
	return m;
}


// function to compute the log of the sum of a vector of numbers represented as logs
double
addLogs( const dvector& x )
{
	double m = max( x ), s = 0.0;
	
	for( unsigned int i = 0; i < x.size(); ++i )
		s += exp( x[ i ] - m );
	
	return log( s ) + m;
}


// static members
unsigned int	Statistic::NUMBER_OF_TESTS = 1;
bool			Statistic::USE_RANKS = false;
bool			Statistic::USE_ADJUSTED = true;
