/* Copyright (C) 2003-2008 Dan Arlow
 * 
 * This file is part of motifADE.
 * 
 * motifADE is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * motifADE is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with motifADE.  If not, see <http://www.gnu.org/licenses/>.
 */

/* 
 *  expression_statistics.hpp
 *
 *  Expresssion statistics data abstractions.
 */


#ifndef STATISTICS_H
#define STATISTICS_H


#include <iostream>
#include <algorithm>
#include <functional>
#include <numeric>
#include <vector>


#include "common.hpp"
#include "promoter.hpp"
#include "mark_set.hpp"


using namespace std;


const double SQRT2 = sqrt( 2.0 );
const double LOG2 = log( 2.0 );

// procedure to compute values of the standard normal cumulative distribution
double
normalcdf( double z );


// function to compute the CDF of Student's distribution
// adapted from Numerical Recipies in C chapter 14.2 http://www.library.cornell.edu/nr/bookcpdf/c14-2.pdf
double
tCdf( double t, double df );


// function to compute values of the incomplete beta function
// adapted from Numerical Recipies in C chapter 6.4 http://www.library.cornell.edu/nr/bookcpdf/c6-4.pdf
double
betai( double a, double b, double x );


// function to compute values of the beta function by continued fraction expansion
// adapted from Numerical Recipies in C chapter 6.4 http://www.library.cornell.edu/nr/bookcpdf/c6-4.pdf
const int		MAXIT   = 100;
const double	EPS		= 3.0e-7;
const double	FPMIN   = 1.0e-30;
double
betacf( double a, double b, double x );


// function to compute the mean and standard deviation of a dvector of data
void
descriptiveStats( const dvector& data, double& mean, double& stddev );


// function to compute the min of two doubles
double
fast_min( double a, double b );


// function to compute the max of a vector of doubles
double
max( const dvector& x );


// function to compute the log of the sum of a vector of numbers represented as logs
double
addLogs( const dvector& x );


// abstract statistic class
class Statistic {
public:
    Statistic( bool init_error = true ) : error( init_error ) { }
    Statistic( const Statistic& statistic ) : error( statistic.error ) { }
    virtual ~Statistic() { }
    
    virtual void	print( ostream& os ) const = 0;
    virtual void	printHeader( ostream& os ) const = 0;
    
    static unsigned int NUMBER_OF_TESTS;
	static bool			USE_RANKS;
	static bool			USE_ADJUSTED;
	
public:
    bool		error;
    double		p, p_adj;
};


// abstract expression statistic calculator class
class ExpressionStatisticCalculator {
public:
    virtual void		compute( const MarkSet& marks, Statistic& statistic ) = 0;
    virtual double		computeValue( const MarkSet& marks ) = 0;
	
	// TODO
	// virtual void selectActiveSequences( const MarkSet& marks ) = 0;
};


#endif // STATISTICS_H
