/* Copyright (C) 2003-2008 Dan Arlow
 * 
 * This file is part of motifADE.
 * 
 * motifADE is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * motifADE is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with motifADE.  If not, see <http://www.gnu.org/licenses/>.
 */

/* 
 *  gene_set_motif_enrichment_base.cpp
 *
 *  Base class for expression statistic calculators that compare the incidence
 *  of motifs in a given gene set to background.
 */

#include "common.hpp"
#include "promoter.hpp"
#include "expression_statistics.hpp"
#include "mark_set.hpp"
#include "gene_set_motif_enrichment_base.hpp"


GeneSetEnrichmentBase::GeneSetEnrichmentBase( unsigned int num_inside, unsigned int inside_size, unsigned int num_outside, unsigned int outside_size )
	: Statistic( false ),
	  numInside( num_inside ), insideSize( inside_size ),
	  numOutside( num_outside ), outsideSize( outside_size ),
	  numTotal( num_inside + num_outside ), totalSize( inside_size + outside_size ),
	  freqInside( ( double ) num_inside / ( double ) inside_size ),
	  freqOutside( ( double ) num_outside / ( double ) outside_size ),
	  freq( ( double ) numTotal / ( double ) totalSize )
{
}


GeneSetEnrichmentBase::GeneSetEnrichmentBase()
	: Statistic( true )
{
}


GeneSetEnrichmentBaseCalculator::GeneSetEnrichmentBaseCalculator( const PromoterVector& promoters, const MarkSet& gene_set )
	: geneSet( gene_set ),
	  totalSize( promoters.size() ),
	  geneSetSize( geneSet.countMarked( true ) ), 
	  restSize( totalSize - geneSetSize ) 
{
	if( geneSet.size() != promoters.size() )
		throw( MotifADEException( "GeneSetEnrichmentBaseCalculator constructor: gene_set.size() != promoters.size()" ) );
}


GeneSetEnrichmentBaseCalculator::GeneSetEnrichmentBaseCalculator( const GeneSetEnrichmentBaseCalculator& calc )
	: geneSet( calc.getGeneSet() ),
	  totalSize( calc.getTotalSize() ),
	  geneSetSize( calc.getGeneSetSize() ),
	  restSize( calc.getRestSize() )
{
}


void
GeneSetEnrichmentBaseCalculator::countIntersection( const MarkSet& marks, unsigned int& numInside, unsigned int& numOutside ) const
{
	if( marks.size() != geneSet.size() )
		throw( MotifADEException( "GeneSetEnrichmentBaseCalculator::countIntersection: marks.size() != geneSet.size()" ) );
	
	numInside = 0;
	numOutside = 0;
	
	// could be made more efficient using SparseMarkSet iterators...
	for( unsigned int i = 0; i < marks.size(); ++i )
		if( marks[ i ] )
			if( geneSet[ i ] )
				++numInside;
			else
				++numOutside;
}
