/* Copyright (C) 2003-2008 Dan Arlow
 * 
 * This file is part of motifADE.
 * 
 * motifADE is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * motifADE is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with motifADE.  If not, see <http://www.gnu.org/licenses/>.
 */

/* 
 *  orthology_annotator.hpp
 *
 *  Class to associate orthologs with promoters using an OrthologyMap
 */


#ifndef ORTHOLOGY_ANNOTATOR_H
#define ORTHOLOGY_ANNOTATOR_H


#include <set>

#include "common.hpp"

#include "sequence.hpp"
#include "sequence_set.hpp"
#include "promoter.hpp"
#include "promoter_set.hpp"

#include "orthology_map.hpp"
#include "operation_stats.hpp"


class OrthologyAnnotator {
public:
	typedef		set< Sequence::IDType >		IDSet;

public:
	struct Stats : OperationStats {
		Stats() { clear(); }
		
		virtual void clear()
		{
			nonAnnotatedIDs.clear();
			removePromotersWithoutOrthologs = true;
			promoter_set_size = 0;
			ortholog_set_size = 0;
			num_mappings_in_map = 0;
			num_promoters_with_orthologs_in_map = 0;
			num_promoters_with_orthologs = 0;
			num_promoters_retained = 0;
		}
		
		virtual ostream& print( ostream& os ) const {
			os << promoter_set_size << " sequences in the promoter set, and " << ortholog_set_size << " sequences in the ortholog set." << endl;
			os << num_mappings_in_map << " mappings in the orthology map." << endl;
			os << num_promoters_with_orthologs_in_map << " promoters had mappings in the orthology map." << endl;
			os << num_promoters_with_orthologs << " had orthologs in the orthologous sequence set." << endl;
			os << "promoters without annotated orthologs were " << ( removePromotersWithoutOrthologs ? "" : "not " ) << "removed." << endl;
			os << num_promoters_retained << " promoters retained." << endl;
			return os;
		}
		
		IDSet			nonAnnotatedIDs;
		
		bool			removePromotersWithoutOrthologs;
		
		unsigned int	promoter_set_size,
						ortholog_set_size,
						num_mappings_in_map,
						num_promoters_with_orthologs_in_map,
						num_promoters_with_orthologs,
						num_promoters_retained;
	};

public:
	OrthologyAnnotator( PromoterSet& p, bool removePromotersWithoutOrthologs = true )
		: removeNonAnnotated( removePromotersWithoutOrthologs ), promoters( p ) {}
	
	void	annotate( OrthologyMap& oMap, SequenceSet& orthologs, OrthologyAnnotator::Stats& stats );
	
private:
	const bool		removeNonAnnotated;
	PromoterSet&	promoters;
};



#endif // ORTHOLOGY_ANNOTATOR_H
