/* Copyright (C) 2003-2008 Dan Arlow
 * 
 * This file is part of motifADE.
 * 
 * motifADE is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * motifADE is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with motifADE.  If not, see <http://www.gnu.org/licenses/>.
 */

/* 
 *  sequence.hpp
 *
 *  Base class for representing DNA sequences.
 */


#ifndef SEQUENCE_H
#define SEQUENCE_H


#include <iostream>
#include <functional>
#include <string>
#include <vector>
#include <iterator>
#include <utility>


#include "common.hpp"
#include "buffered_reader.hpp"
#include "tokenizer.hpp"
#include "id_object.hpp"


using namespace std;


// data structure to store a sequence with a header
class Sequence : public IDObject< string > {
	friend BufferedReader& operator>>( BufferedReader& is, Sequence& seq );

public:
	typedef string  HeaderType;
	typedef string  DataType;
	
	typedef pair< HeaderType::const_iterator, HeaderType::const_iterator >  IDLoc;

public:
    Sequence() {}
	Sequence( const Sequence& s );
    Sequence( const HeaderType& init_header, const string& init_data );
	Sequence( const HeaderType& init_header, const IDType& init_id, const DataType& init_data );
	    
    ~Sequence() {}
        
    void				parseID();
	unsigned int		totalSequenceLength() const;
	
    // accessor functions
    const HeaderType&	getHeader()									const { return header; }
    const DataType&		getData()									const { return data; }
	unsigned int		getLength()									const { return data.length(); }
    
    // mutator functions
    void				setHeader( const HeaderType& newHeader )	{ header = newHeader; }
    void				setData( const DataType& newData )			{ data = newData; }
	
	// helper functions
	static IDLoc		findID( const HeaderType& h );
	
protected:
    HeaderType			header;
    DataType			data;
	
	bool				owner;
};


BufferedReader& operator>>( BufferedReader& is, Sequence& seq );

ostream& operator<<( ostream& os, const Sequence& seq );


typedef vector< Sequence* >			SequenceVector;
typedef vector< const Sequence* >   ConstSequenceVector;


#endif // SEQUENCE_H
